/*
 *  support.c
 *  Specific support functions
 *
 *  Copyright (C) 1997 Martin von Lwis
 *  Copyright (C) 1997 Rgis Duchesne
 *
 */

#include "types.h"
#include "struct.h"
#include "support.h"

#include <stdarg.h>
#include <linux/malloc.h>
#include <linux/locks.h>
#include "util.h"
#include "inode.h"
#include "macros.h"

static char print_buf[1024];

#ifdef DEBUG
#include "sysctl.h"

/* Debugging output */
void ntfs_debug(int mask, const char *fmt, ...)
{
	va_list ap;

	/* Filter it with the debugging level required */
	if(ntdebug & mask){
		va_start(ap,fmt);
		strcpy(print_buf, KERN_DEBUG);
		vsprintf(print_buf + 3, fmt, ap);
		printk(print_buf);
		va_end(ap);
	}
}

/* Verbose kmalloc */
void *ntfs_malloc(int size)
{
	void *ret;

	ret = kmalloc(size, GFP_KERNEL);
	ntfs_debug(DEBUG_MALLOC, "Allocating %x at %p\n", size, ret);

	return ret;
}

/* Verbose kfree() */
void ntfs_free(void *block)
{
        ntfs_debug(DEBUG_MALLOC, "Freeing memory at %p\n", block);
	kfree(block);
}
#else
void ntfs_debug(int mask, const char *fmt, ...)
{
}

void *ntfs_malloc(int size)
{
	return kmalloc(size, GFP_KERNEL);
}

void ntfs_free(void *block)
{
	kfree(block);
}
#endif /* DEBUG */

void ntfs_bzero(void *s, int n)
{
	memset(s, 0, n);
}

void *ntfs_memcpy(void *dest, const void *src, ntfs_size_t n)
{
	return memcpy(dest, src, n);
}

void *ntfs_memmove(void *dest, const void *src, ntfs_size_t n)
{
	return memmove(dest, src, n);
}

/* Warn that an error occured */
void ntfs_error(const char *fmt,...)
{
        va_list ap;

        va_start(ap, fmt);
        strcpy(print_buf, KERN_ERR);
        vsprintf(print_buf + 3, fmt, ap);
        printk(print_buf);
        va_end(ap);
}

int ntfs_read_mft_record(ntfs_volume *vol, int mftno, char *buf)
{
	int error;
	ntfs_io io;

	ntfs_debug(DEBUG_OTHER, "read_mft_record %x\n",mftno);
	if(mftno==FILE_MFT)
	{
		ntfs_memcpy(buf,vol->mft,vol->mft_recordsize);
		return 0;
	}
	if(!vol->mft_ino)
	{
		printk("ntfs:something is terribly wrong here\n");
		return ENODATA;
	}
 	io.fn_put=ntfs_put;
	io.fn_get=0;
	io.param=buf;
	io.size=vol->mft_recordsize;
	error=ntfs_read_attr(vol->mft_ino,vol->at_data,NULL,
			     mftno*vol->mft_recordsize,&io);
	if(error || (io.size!=vol->mft_recordsize))
	{
		ntfs_debug(DEBUG_OTHER, "read_mft_record: read %x failed (%d,%d,%d)\n",
			   mftno,error,io.size,vol->mft_recordsize);
		return error?error:ENODATA;
	}
	ntfs_debug(DEBUG_OTHER, "read_mft_record: finished read %x\n",mftno);
	if(!ntfs_check_mft_record(vol,buf))
	{
		printk("Invalid MFT record for %x\n",mftno);
		return EINVAL;
	}
	ntfs_debug(DEBUG_OTHER, "read_mft_record: Done %x\n",mftno);
	return 0;
}

int ntfs_getput_clusters(ntfs_volume *vol, int cluster,	ntfs_size_t start_offs,
	 ntfs_io *buf)
{
	struct super_block *sb=NTFS_SB(vol);
	struct buffer_head *bh;
	ntfs_size_t to_copy;
	int length=buf->size;
	if(buf->do_read)
		ntfs_debug(DEBUG_OTHER, "get_clusters %d %d %d\n",cluster,start_offs,length);
	else
		ntfs_debug(DEBUG_OTHER, "put_clusters %d %d %d\n",cluster,start_offs,length);
	while(length)
	{
		if(!(bh=bread(sb->s_dev,cluster,vol->clustersize)))
		{
			ntfs_debug(DEBUG_OTHER, "%s failed\n", buf->do_read?"Reading":"Writing");
			return EIO;
		}
		to_copy=min(vol->clustersize-start_offs,length);
		lock_buffer(bh);
		if(buf->do_read)
			buf->fn_put(buf,bh->b_data+start_offs,to_copy);
		else
		{
			buf->fn_get(bh->b_data+start_offs,buf,to_copy);
			mark_buffer_dirty(bh,1);
		}
		unlock_buffer(bh);
		length-=to_copy;
		start_offs=0;
		cluster++;
		brelse(bh);
	}
	return 0;
}

ntfs_time64_t ntfs_now(void)
{
	return ntfs_unixutc2ntutc(CURRENT_TIME);
}

int ntfs_dupuni2map(ntfs_volume *vol, ntfs_u16 *in, int in_len, char **out,
        int *out_len)
{
	/* Not supported here */
	return EINVAL;
}

int ntfs_dupmap2uni(ntfs_volume *vol, char* in, int in_len, ntfs_u16 **out,
        int *out_len)
{
	/* Not supported here */
	return EINVAL;
}

/*
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
