/* parseabc.c */
/* code to parse an abc file */
/* this is part of the abc2midi abc to MIDI converter */
/* James Allwright (jra@ecs.soton.ac.uk) */
/* Department of Electronics and computer Science, */
/* University of Southampton, UK */
/* 19th August 1996 */

/* Macintosh port 30th July 1996 */
/* DropShell integration   27th Jan  1997 */
/* Wil Macaulay (wil@syndesis.com) */ 


#define MAXLINE 200
#define TAB 9
#include "abc.h"
#include <stdio.h>

#ifdef __MWERKS__
#define __MACINTOSH__ 1
#endif /* __MWERKS__ */

#ifdef __MACINTOSH__
#define main macabc2midi_main
#include <string.h>
#define index(a,b) (int)strchr((a),(b))
#endif /* __MACINTOSH__ */

int lineno;
int parsing, slur;
int inhead, inbody;
char* index();

parseron()
{
  parsing = 1;
  slur = 0;
};

parseroff()
{
  parsing = 0;
  slur = 0;
};

skipspace(p)
char **p;
{
  /* skip space and tab */
  while(((int)**p == ' ') || ((int)**p == TAB)) *p = *p + 1;
};

int readnumf(num)
char *num;
{
  int t;
  char* p;

  p =num;
  if (index("0123456789", *p) == NULL) {
    event_error("Missing Number");
  };
  t = 0;
  while (((int)*p >= '0') && ((int)*p <= '9')) {
    t = t * 10 + (int) *p - '0';
    p = p + 1;
  };
  return (t);
};

int readnump(p)
char **p;
{
  int t;

  t = 0;
  while (((int)**p >= '0') && ((int)**p <= '9')) {
    t = t * 10 + (int) **p - '0';
    *p = *p + 1;
  };
  return (t);
};

void readsig(a, b, sig)
int *a, *b;
char **sig;
{
  int t;
  char errmsg[40];

  if ((int)**sig == 'C') {
    *a = 4;
    *b = 4;
    return;
  };
  *a = readnump(sig);
  if ((int)**sig != '/') {
    event_error("Missing / ");
  };
  *sig = *sig + 1;
  *b = readnump(sig);
  if ((*a == 0) || (*b == 0)) {
    sprintf(errmsg, "Not read a valid meter %d/%d", *a, *b);
    event_error(errmsg);
  };
  t = *b;
  while (t > 1) {
    if (t%2 != 0) {
      event_error("divisor must be a power of 2");
      t = 1;
    } else {
      t = t/2;
    };
  };
};

void readlen(a, b, p)
int *a, *b;
char **p;
{
  int t;

  *a = readnump(p);
  if (*a == 0) {
    *a = 1;
  };
  *b = 1;
  if (**p == '/') {
    *p = *p + 1;
    *b = readnump(p);
    if (*b == 0) {
      *b = 2;
    };
  };
  t = *b;
  while (t > 1) {
    if (t%2 != 0) {
      event_warning("divisor not a power of 2");
      t = 1;
    } else {
      t = t/2;
    };
  };
};


void parsekey(str)
char* str;
{
  int sf, minor;
  char *s;
  static char *key = "FCGDAEB";
  static char *mode[10] = {"maj", "min", "m", 
                       "aeo", "loc", "ion", "dor", "phr", "lyd", "mix"};
  static int modeshift[10] = {0, -3, -3,
                         -3, -5, 0, -2, -4, 1, -1 };
  static int modeminor[10] = {0, 1, 1,
                          1, 0, 0, 0, 0, 0, 0};
  char modestr[4];
  int i;
  char modmap[7];
  int modmul[7];

  minor = 0;
  s=str;
  if ((strncmp(s, "Hp", 2) == 0) || (strncmp(s, "HP", 2) == 0)) {
    sf = 2;
    s = s + 2;
  } else {
    if (index(key, *s) != NULL) {
      int foundmode;

      /* parse key itself */
      sf = (int) index(key, *s) - (int) &key[0] - 1;
      s = s + 1;
      /* deal with sharp/flat */
      if (*s == '#') {
        sf += 7;
        s += 1;
      } else if (*s == 'b') {
        sf -= 7;
        s += 1;
      }
      skipspace(&s);
      /* now get mode */
      i = 0;
      while (isalpha(*s) && (i<3)) {
        if (isupper(*s)) {
          modestr[i] = tolower(*s);
        } else {
          modestr[i] = *s;
        };
        s = s + 1;
        i = i + 1;
      };
      /* skip any alphabetic characters after the first 3 */
      while ((i == 3) && (isalpha(*s))) s = s + 1;
      modestr[i] = '\0';
      foundmode = 0;
      for (i = 0; i<10; i++) {
        if (strcmp(modestr, mode[i]) == 0) {
          foundmode = 1;
          sf = sf + modeshift[i];
          minor = modeminor[i];
        };
      };
      if ((strlen(modestr) > 0) && (!foundmode)) {
        event_error("Key mode not recognized");
      };
    } else {
      event_error("Key not recognized");
    };
  };
  if (isalpha(*s)) {
    event_error("Unrecognized characters in key");
  };
  if (sf > 7) {
    event_warning("Unusual key representation");
   sf = sf - 12;
  } ;
  if (sf < -7) {
    event_warning("Unusual key representation");
    sf = sf + 12;
  };
  /* now look for modifiers */
  for (i=0; i<7; i++) {
    modmap[i] = ' ';
    modmul[i] = 1;
  };
  skipspace(&s);
  while (*s != '\0') {
    int count;
    char acc;
    int j;

    while (((int)*s != '\0') && (index("^_=", *s) == NULL)) {
      event_error("Spurious character in key");
      s = s + 1;
    };
    if (index("^_=", *s) != NULL) {
      acc = *s;
      count = 1;
      s = s + 1;
      while (*s == acc) {
        count = count + 1;
        s = s + 1;
      };
      if (((int)*s >= 'a') && ((int)*s <= 'g')) {
        j = (int)*s - 'a';
        s = s + 1;
        if ((count > 2) || ((count > 1) && (acc == '='))) {
          event_error("accent must be __ _ = ^ or ^^");
          count = 1;
        };
        skipspace(&s);
        modmap[j] = acc;
        modmul[j] = count;
      } else {
        event_error("accent must be applied to a-g");
        s = s + 1;
      };
    };
  };
  event_key(sf, str, minor, modmap, modmul);
}

parsenote(s)
char **s;
{
  int decorators[DECSIZE];
  int i, t;
  int mult;
  char accidental, note;
  int octave, n, m;

  mult = 1;
  accidental = ' ';
  note = ' ';
  for (i = 0; i<DECSIZE; i++) decorators[i] = 0;
  while (index(decorations, **s) != NULL) {
    t = (int) index(decorations, **s) -  (int) decorations;
    decorators[t] = 1;
    *s = *s + 1;
  };
  /* read accidental */
  switch (**s) {
  case '_':
    accidental = **s;
    *s = *s + 1;
    if (**s == '_') {
      *s = *s + 1;
      mult = 2;
    };
    break;
  case '^':
    accidental = **s;
    *s = *s + 1;
    if (**s == '^') {
      *s = *s + 1;
      mult = 2;
    };
    break;
  case '=':
    accidental = **s;
    *s = *s + 1;
    if ((**s == '^') || (**s == '_')) {
      accidental = **s;
    };
    break;
  default:
    /* do nothing */
    break;
  };
  if ((**s >= 'a') && (**s <= 'g')) {
    note = **s;
    octave = 1;
    *s = *s + 1;
    while (**s == '\'') {
      octave = octave + 1;
      *s = *s + 1;
    };
  } else {
    if ((**s >= 'A') && (**s <= 'G')) {
      note = **s + 'a' - 'A';
      octave = 0;
      *s = *s + 1;
      while (**s == ',') {
        octave = octave - 1;
        *s = *s + 1;
      };
    };
  };
  if (note == ' ') {
    event_error("Malformed note : expecting a-g or A-G");
  } else {
    readlen(&n, &m, s);
    event_note(decorators, accidental, mult, note, octave, n, m);
  };
}

parsemusic(field)
char* field;
{
  char* p;
  char* comment;
  int iscomment;

  comment = field;
  iscomment = 0;
  while ((*comment != '\0') && (*comment != '%')) {
    comment = comment + 1;
  };
  if (*comment == '%') {
    iscomment = 1;
    *comment = '\0';
    comment = comment + 1;
  };

  /* printf("parsing music line\n"); */
  p = field;
  skipspace(&p);
  while(*p != '\0') {
    if (((*p >= 'a') && (*p <= 'g')) || ((*p >= 'A') && (*p <= 'G')) ||
        (index("_^=", *p) != NULL) || (index(decorations, *p) != NULL)) {
      parsenote(&p);
    } else {
      switch(*p) {
      case '+':
        event_chord();
        p = p + 1;
        break;
      case '"':
        {
          int i;
          char gchord[MAXLINE];
   
          p = p + 1;
          i = 0;
          while ((*p != '"') && (*p != '\0')) {
            gchord[i] = *p;
            i = i + 1;
            p = p + 1;
          };
          gchord[i] = '\0';
          if (*p == '\0') {
            event_error("Guitar chord name not properly closed");
          } else {
            p = p + 1;
          };
          event_gchord(gchord);
          break;
        };
      case '|':
        p = p + 1;
        switch(*p) {
          case ':':
            event_bar(BAR_REP);
            p = p + 1;
            break;
          case '|' :
            event_bar(DOUBLE_BAR);
            p = p + 1;
            break;
          case ']' :
            event_bar(THIN_THICK);
            p = p + 1;
            break;
          case '1':
            event_bar(BAR1);
            p = p + 1;
            break;
          default :
            event_bar(SINGLE_BAR);
        };
        break;
      case ':':
        p = p + 1;
        switch(*p) {
          case ':':
            event_bar(DOUBLE_REP);
            p = p + 1;
            break;
          case '|':
            p = p + 1;
            if (*p == '2') {
              event_bar(REP_BAR2);
              p = p + 1;
            } else {
              event_bar(REP_BAR);
            };
            break;
          default:
            event_error("Single colon in bar");
        };
        break;
      case ' ':
        event_space();
        skipspace(&p);
        break;
      case TAB:
        event_space();
        skipspace(&p);
        break;
      case '(':
        p = p + 1;
        {
          int t, q, r;

          t = 0;
          q = 0;
          r = 0;
          t = readnump(&p);
          if ((t != 0) && (*p == ':')) {
            p = p + 1;
            q = readnump(&p);
            if ((q != 0) && (*p == ':')) {
              p = p + 1;
              r = readnump(&p);
            };
          };
          if (t == 0) {
            if (slur > 0) {
              event_warning("Slur within slur");
            };
            slur = slur + 1;
            event_sluron(slur);
          } else {
            event_tuple(t, q, r);
          };
        };
        break;
      case ')':
        p = p + 1;
        if (slur == 0) {
          event_error("No slur to close");
        } else {
          slur = slur - 1;
        };
        event_sluroff(slur);
        break;
      case '{':
        p = p + 1;
        event_graceon();
        break;
      case '}':
        p = p + 1;
        event_graceoff();
        break;
      case '[':
        p = p + 1;
        switch(*p) {
        case '1':
          p = p + 1;
          event_rep1();
          break;
        case '2':
          p = p + 1;
          event_rep2();
          break;
        case '|':
          p = p + 1;
          event_bar(THICK_THIN);
          break;
        default:
          event_chordon();
          break;
        };
        break;
      case ']':
        p = p + 1;
        event_chordoff();
        break;
      case 'z':
        {
          int n, m;

          p = p + 1;
          readlen(&n, &m, &p);
          event_rest(n, m);
          break;
        };
      case 'Z':
        {
          int n, m;

          p = p + 1;
          readlen(&n, &m, &p);
          event_rest(n, m);
          break;
        };
      case '>':
        {
          int n;

          n = 0;
          while (*p == '>') {
            n = n + 1;
            p = p + 1;
          };
          if (n>3) {
            event_error("Too many >'s");
          } else {
            event_broken(GT, n);
          };
          break;
        };
      case '<':
        {
          int n;

          n = 0;
          while (*p == '<') {
            n = n + 1;
            p = p + 1;
          };
          if (n>3) {
            event_error("Too many <'s");
          } else {
            event_broken(LT, n);
          };
          break;
        };
      case 's':
        if (slur == 0) {
          slur = 1;
        } else {
          slur = slur - 1;
        };
        event_slur(slur);
        p = p + 1;
        break;
      case '-':
        event_tie();
        p = p + 1;
        break;
      case '\\':
        event_lineend('\\', 1);
        p = p + 1;
        break;
      case '!':
        {
          int i;
          char instruction[MAXLINE];
          char *s;
   
          p = p + 1;
          s = p;
          i = 0;
          while ((*p != '!') && (*p != '\0')) {
            instruction[i] = *p;
            i = i + 1;
            p = p + 1;
          };
          instruction[i] = '\0';
          if (*p != '!') {
            p = s;
            event_lineend('!', 1);
          } else {
            event_instruction(instruction);
            p = p + 1;
          };
        };
        break;
      case '*':
        p = p + 1;
        if (*p == '*') {
          event_lineend('*', 2);
          p = p + 1;
        } else {
          event_lineend('*', 1);
        };
        break;
      default:
        {
          char msg[40];

          if ((*p >= 'H') && (*p <= 'Z')) {
            event_reserved(*p);
          } else {
            sprintf(msg, "Unrecognized character: %c", *p);
            event_error(msg);
          };
        };
        p = p + 1;
      };
    };
  };
  event_musiclinebreak();
  if (iscomment) {
    event_precomment(comment);
  };
};

parse_tempo(place)
char* place;
{
  char* p;
  int a, b;
  int relative;

  relative = 0;
  p = place;
  while ((*p != '\0') && (*p != '=')) p = p + 1;
  if (*p == '=') {
    p = place;
    skipspace(&p);
    if (((*p >= 'A') && (*p <= 'G')) || ((*p >= 'a') && (*p <= 'g'))) {
      relative = 1;
      p = p + 1;
    };
    readlen(&a, &b, &p);
    skipspace(&p);
    if (*p != '=') {
      event_error("Expecting = in tempo");
    };
    p = p + 1;
  } else {
    a = 0;
    b = 0;
    p = place;
  };
  skipspace(&p);
  event_tempo(readnumf(p), a, b, relative);
};

parsefield(key, field)
char key;
char* field;
{
  char* comment;
  char* place;
  int iscomment;

  if ((inbody) && (index("EIKLMPQTVwW", key) == NULL)) {
    event_error("Field not allowed in tune body");
  };
  comment = field;
  iscomment = 0;
  while ((*comment != '\0') && (*comment != '%')) {
    comment = comment + 1;
  };
  if (*comment == '%') {
    iscomment = 1;
    *comment = '\0';
    comment = comment + 1;
  };
  place =field;
  skipspace(&place);
  switch (key) {
  case 'X':
    {
      int x;
    
      x = readnumf(place);
      if (inhead) {
        event_error("second X: field in header");
      };
      event_refno(x);
      inhead = 1;
      inbody = 0;
      break;
    };
  case 'K':
    parsekey(place);
    if (inhead || inbody) {
      inbody = 1;
      inhead = 0;
    } else {
      event_error("No X: field preceding K:");
    };
    break;
  case 'M':
    {
      int num, denom;

      readsig(&num, &denom, &place);
      if ((*place == 's') || (*place == 'l')) {
        event_error("s and l in M: field not supported");
      };
      event_timesig(num, denom);
      break;
    };
  case 'L':
    {
      int num, denom;

      readsig(&num, &denom, &place);
      if (num != 1) {
        event_error("Default length must be 1/X");
      };
      event_length(denom);
      break;
    };
  case 'P':
    event_part(place);
    break;
  case 'V':
    {
      int num;

      skipspace(&place);
      num = readnump(&place);
      skipspace(&place);
      event_voice(num, place);
      break;
    };
  case 'Q':
    parse_tempo(place);
    break;
  case 'w':
    event_words(place);
    break;
  default:
    event_field(key, place);
  };
  if (iscomment) {
    event_precomment(comment);  
  };
};

readstr(out, in)
char *out;
char **in;
{
  char *p;

  p = out; 
  while (((**in >= 'a') && (**in <= 'z')) || ((**in >= 'A') && (**in <= 'Z'))) {
    *p = **in;
    p = p + 1;
    *in = *in + 1;
  };
  *p = '\0';
};

event_precomment(s)
char* s;
{
  char package[40];
  char *p;

  if (*s == '%') {
    p = s+1;
    readstr(package, &p);
    event_specific(package, p);
  } else {
    event_comment(s);
  };
};

parseline(line)
char line[MAXLINE];
{
  char *p, *q;

  /* printf("parsing : %s\n", line);  */
  p = line;
  skipspace(&p);
  if (strlen(p) == 0) {
    event_blankline();
    inhead = 0;
    inbody = 0;
    return;
  };
  if ((int)*p == '\\') {
    if (parsing) {
      event_tex(p);
    };
    return;
  };
  if ((int)*p == '%') {
    if (parsing) {
      event_precomment(p+1);
    };
    return;
  };
  if (index("ABCDEFGHIKLMNOPQRSTVwWXZ", *p) != NULL) {
    q = p + 1;
    skipspace(&q);
    if ((int)*q == ':') {
      if (*(line+1) != ':') {
        event_warning("whitespace in field declaration");
      };
      if ((*(q+1) == ':') || (*(q+1) == '|')) {
        event_warning("potentially ambiguous line");
      };
      parsefield(*p, q+1);
    } else {
      if (inbody) {
        if (parsing) parsemusic(p);
      } else {
        event_text(p);
      };
    };
  } else {
    if (inbody) {
      if (parsing) parsemusic(p);
    } else {
      event_text(p);
    };
  };
};

parsefile(name)
char* name;
{
  FILE *fp;
  int reading;
  char line[MAXLINE];
  int t, len;

  /* printf("parsefile called %s\n", name); */
  fp = fopen(name, "r");
  if (fp == NULL) {
    printf("Failed to open file %s\n", name);
    exit(1);
  };
  inhead = 0;
  inbody = 0;
  parseroff();
  reading = 1;
  len = 0;
  lineno = 1;
  while (reading) {
    t = getc(fp);
    if (t == EOF) {
      reading = 0;
      if (len>0) {
        line[len] = '\0';
        parseline(line);
        event_linebreak();
        lineno = lineno + 1;
      };
    } else {
      line[len] = (char) t;
      len = len + 1;
      if (len >= MAXLINE) {
        event_error("Line too Long");
        len = 0;
      };
      if (t == '\n') {
        len = len - 1;
        line[len] = '\0';
        /* throw away carriage return in DOS input */
        if ((len > 0) && ((int)line[len-1] == '\r')) {
          line[len-1] ='\0';
        };
        parseline(line);
        event_linebreak();
        lineno = lineno + 1;
        len = 0;
      };
    };
  };
  fclose(fp);
  event_eof();
};    

main(argc,argv)
int argc;
char *argv[];
{
  char *filename;

  event_init(argc, argv, &filename);
  if (argc < 2) {
    /* printf("argc = %d\n", argc); */
  } else {
    parsefile(filename);
  };
};

/*
int getline ()
{
  return (lineno);
}
*/
