/*
 * dnode.c - PTX node functions for lsof
 */


/*
 * Copyright 1995 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1995 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.6 98/03/06 08:27:34 abe Exp $";
#endif


#include "lsof.h"

#if	PTXV<400
#define	_INKERNEL
#endif	/* PTXV<400 */

#include <sys/ufsinode.h>

#if	PTXV<400
#undef	_INKERNEL
#endif	/* PTXV<400 */


#if	PTXV>=400
_PROTOTYPE(static void ent_fa,(KA_T *a1, KA_T *a2, char *d));
#endif	/* PTXV>=400 */

_PROTOTYPE(static struct l_dev * finddev,(dev_t *dev, int stream));
_PROTOTYPE(static int isip,(char *mn));
_PROTOTYPE(static char isvlocked,(struct vnode *va));


struct iptab_proto {
	char *mn;			/* module name */
	char *nm;			/* protocol name */
	int fam;			/* protocol family */
	int ipid;			/* IP protocol ID */
} IPtab[] = {
	{ "tcp",	"TCP",	AF_INET,	IPPROTO_TCP	},
	{ "udp",	"UDP",	AF_INET,	IPPROTO_UDP	},
	{ NULL,		NULL,	0,		0		},
};


#if	PTXV>=400
/*
 * ent_fa() - enter fattach addresses in NAME column addition
 */

static void
ent_fa(a1, a2, d)
	KA_T *a1;			/* first fattach address (NULL OK) */
	KA_T *a2;			/* second fattach address */
	char *d;			/* direction ("->" or "<-") */
{
	char buf[128], *cp;
	MALLOC_S len;

	if (Lf->nma)
	    return;
	if (!a1)
	    (void) sprintf(buf, "(FA:%s%s)", d, print_kptr(*a2, (char *)NULL));
	else
	    (void) sprintf(buf, " (FA:%s%s%s)", print_kptr(*a1, (char *)NULL),
		d, print_kptr(*a2, (char *)NULL));
	len = strlen(buf) + 1;
	if ((cp = (char *)malloc(len)) == NULL) {
	    (void) fprintf(stderr,
		"%s: no space for fattach addresses at PID %d, FD %s\n",
		Pn, Lp->pid, Lf->fd);
	    Exit(1);
	}
	(void) strcpy(cp, buf);
	Lf->nma = cp;
}
#endif	/* PTXV>=400 */


/*
 * finddev() - look up device by device number
 */

static struct l_dev *
finddev(dev, stream)
	dev_t *dev;			/* device */
	int stream;			/* stream if 1 */
{
	struct clone *c;
	struct l_dev *dp;
/*
 * Search device table for match.
 */

#if	defined(HASDCACHE)

finddev_again:

#endif	/* defined(HASDCACHE) */

	if ((dp = lkupdev(dev, 0, 0)))
	    return(dp);
/*
 * Search for clone.
 */
	if (stream && Clone) {
	    for (c = Clone; c; c = c->next) {
		if (major(*dev) == minor(Devtp[c->dx].rdev)) {

#if	defined(HASDCACHE)
		    if (DCunsafe && !Devtp[c->dx].v && !vfy_dev(&Devtp[c->dx]))
			goto finddev_again;
#endif	/* defined(HASDCACHE) */

		    return(&Devtp[c->dx]);
		}
	    }
	}
	return((struct l_dev *)NULL);
}


/*
 * isip() - is the module name an IP name?
 */

int
isip(mn)
	char *mn;			/* module_info name */
{
	int i;

	for (i = 0; IPtab[i].mn; i++) {
	    if (strcasecmp(mn, IPtab[i].mn) == 0)
		return(i);
	}
	return(-1);
}


/*
 * isvlocked() - is a vnode locked
 */

static char
isvlocked(va)
	struct vnode *va;		/* local vnode address */
{

	struct filock f;
	KA_T flf, flp;
	int l;

#if	PTXV>=410
	struct vreclck vl;
#endif	/* PTXV<410 */

#if	PTXV<410
	if (!(flf = (KA_T)va->v_filocks))
	    return(' ');
#else	/* PTXV>=410 */
	if (!va->v_recp
	||  kread((KA_T)va->v_recp, (char *)&vl, sizeof(vl))
	||  !(flf = (KA_T)vl.vr_filocks))
	    return(' ');
#endif	/* PTXV<410 */

	flp = flf;
	do {
	    if (kread(flp, (char *)&f, sizeof(f)))
		break;
	    if (f.set.l_sysid || f.set.l_pid != (pid_t)Lp->pid)
		continue;
	    if (f.set.l_whence == 0 && f.set.l_start == 0

#if	PTXV<420
	    &&  f.set.l_len == 0x7fffffff
#else	/* PTXV>=420 */
	    &&  f.set.l_len == 0x7fffffffffffffffLL
#endif	/* PTXV<420 */

	    )
		l = 1;
	    else
		l = 0;
	    switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {
	    case F_RDLCK:
		return((l) ? 'R' : 'r');
	    case F_WRLCK:
		return((l) ? 'W' : 'w');
	    case (F_RDLCK + F_WRLCK):
		return('u');
	    }
	    return(' ');
	} while ((flp = (KA_T)f.next) && flp != flf);
	return(' ');
}


/*
 * process_node() - process vnode
 */

void
process_node(va)
	KA_T va;			/* kernel vnode address */
{
	char *cp, tbuf[32];
	static KA_T cvops = (KA_T)0;
	dev_t dev = 0;
	int dl, fp, j, k, l, lp, pt, px;
	struct l_dev *dp;
	struct fifonode f;
	struct in_addr *fa = (struct in_addr *)NULL;
	static int ft = 1;
	static KA_T fvops = (KA_T)0;
	struct inode i;
	KA_T ka, p, qp, sop;
	struct in_addr *la = (struct in_addr *)NULL;
	struct module_info mi;
	static KA_T nfsvops = (KA_T)0;
	static KA_T nmvops = (KA_T)0;
	struct inpcb pcb;
	struct queue q;
	struct qinit qi;
	static KA_T sckvops = (KA_T)0;
	static KA_T spvops = (KA_T)0;
	struct so_so so;
	struct sockaddr_un ua;
	static KA_T uvops = (KA_T)0;
	static KA_T vvops = (KA_T)0;

#if	defined(HAS_NFS)
	struct rnode r;
#endif	/* defined(HAS_NFS) */

	struct snode s;
	KA_T sa;
	struct stdata sd;
	int sin = 0;
	int sns = 0;
	KA_T sp;
	int srn = 0;
	int srv = 0;
	char *ty;
	enum vtype type;
	struct vnode v, rv, uv;
	struct l_vfs *lvfs;
	int usst = 0;

#if	PTXV>=400
	struct vfs favfs;
	struct namenode n;
	int nmst = 1;
#endif	/* PTXV>=400 */

#if	PTXV>=413
	struct domain dm;
	struct socket_peer peer;
	struct protosw pr;
	struct socket sck;
#endif	/* PTXV>=413 */

#if	PTXV>=421
	struct datab db;
	struct msgb mb;
	struct unpcb ub;
	struct sockaddr_un un;
#endif	/* PTXV>=421 */

#if	defined(HASVXFS)
	int svx = 0;
#endif	/* defined(HASVXFS) */

/*
 * Do first-time only operations.
 */
	if (ft) {
	    if (get_Nl_value("cvops", Drive_Nl, &cvops) < 0 || !cvops)
		cvops = (KA_T)0;
	    if (get_Nl_value("fvops", Drive_Nl, &fvops) < 0 || !fvops)
		fvops = (KA_T)0;
	    if (get_Nl_value("nfsvops", Drive_Nl, &nfsvops) < 0 || !nfsvops)
		nfsvops = (KA_T)0;
	    if (get_Nl_value("nmvops", Drive_Nl, &nmvops) < 0 || !nmvops)
		nmvops = (KA_T)0;
	    if (get_Nl_value("sckvops", Drive_Nl, &sckvops) < 0 || !sckvops)
		sckvops = (KA_T)0;
	    if (get_Nl_value("spvops", Drive_Nl, &spvops) < 0 || !spvops)
		spvops = (KA_T)0;
	    if (get_Nl_value("uvops", Drive_Nl, &uvops) < 0 || !uvops)
		uvops = (KA_T)0;
	    if (get_Nl_value("vvops", Drive_Nl, &vvops) < 0 || !vvops)
		vvops = (KA_T)0;
	    ft = 0;
	}
/*
 * Read the vnode.
 */

read_vnode:

	if (!va) {
	    enter_nm("no vnode address");
	    return;
	}
	if (readvnode(va, &v)) {
	    enter_nm(Namech);
	   return;
	}

# if	defined(HASNCACHE)
	Lf->na = va;
# endif	/* defined(HASNCACHE) */

/*
 * Determine the vnode type.
 */
	if (uvops && uvops == (KA_T)v.v_op)
	    Ntype = N_REGLR;
	else if (fvops && fvops == (KA_T)v.v_op)
	    Ntype = N_FIFO;

#if	defined(HAS_NFS)
	else if (nfsvops && nfsvops == (KA_T)v.v_op)
	    Ntype = N_NFS;
#endif	/* defined(HAS_NFS) */

#if	PTXV>=413
	else if (sckvops && sckvops == (KA_T)v.v_op)
	    Ntype = N_SOCK;
#endif	/* PTXV>=413 */

	else if (spvops && spvops == (KA_T)v.v_op)
	    Ntype = N_SPEC;

#if	defined(HAS_CDFS)
	else if (cvops && cvops == (KA_T)v.v_op)
	    Ntype = N_CDFS;
#endif	/* defined(HAS_CDFS) */

#if	defined(HASVXFS)
	else if (vvops && vvops == (KA_T)v.v_op)
	    Ntype = N_VXFS;
#endif	/* defined(HASVXFS) */

#if	PTXV>=400
	else if (nmvops && nmvops == (KA_T)v.v_op) {
	    Ntype = N_NM;
	/*
	 * Read the namenode and enter the fattach addresses.
	 */
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&n, sizeof(n))) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read namenode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (n.nm_mountpt)
		ent_fa(&va, (KA_T *)&n.nm_mountpt, "->");
	/*
	 * If this is the first time a namenode has been examined, and
	 * if it has an nm_filevp pointer, reset the vnode address to
	 * it and start over.
	 */
	    if (nmst && n.nm_filevp) {
		va = (KA_T)n.nm_filevp;
		nmst = 0;
		goto read_vnode;
	    }
	}
#endif	/* PTXV>=400 */

	else {
	
	/*
	 * The file system type is unknown.
	 */
	    (void) sprintf(Namech,
		"unknown file system type, v_op: %s",
		print_kptr((KA_T)v.v_op, (char *)NULL));
	    enter_nm(Namech);
	    return;
	}

#if	PTXV>=400
/*
 * If this node has a v_vfsmountedhere pointer, read it.  If it represents
 * a "namefs" file system, and has a private data pointer, record the
 * fattach addresses.
 */
	if (v.v_vfsmountedhere
	&&  !kread((KA_T)v.v_vfsmountedhere, (char *)&favfs, sizeof(favfs))
	&&  favfs.vfs_data)
	    ent_fa(&va, (KA_T *)&favfs.vfs_data, "<-");
#endif	/* PTXV>=400 */

/*
 * Determine the lock type.
 */
	Lf->lock = isvlocked(&v);
/*
 * Establish the local virtual file system structure.
 */
	if (!v.v_vfsp)
	    lvfs = (struct l_vfs *)NULL;
	else {
	    if (!(lvfs = readvfs(&v))) {
		(void) sprintf(Namech, "bad vfs for %s at %s",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_vfsp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Read the cd_inode, fifonode, inode, rnode, snode, or vx_inode.
 */
	switch (Ntype) {

#if	defined(HAS_CDFS)
	case N_CDFS:
	    if (Selinet)
		return;
	    if (!v.v_data || read_cdnode(&v, lvfs, &dev)) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read cd_inode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    break;
#endif	/* defined(HAS_CDFS) */

#if	defined(HASVXFS)
	case N_VXFS:
	    if (Selinet)
		return;
	    if (!v.v_data || read_vxinode(&v, lvfs, &dev)) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read vx_inode (%s)",
		    print_kptr(va, tptr),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }

# if	defined(HASNCACHE) && defined(HASVXFSDNLC)
	    Lf->is_vxfs = 1;
	    Lf->na = (KA_T)v.v_data;
# endif	/* defined(HASNCACHE) && defined(HASVXFSDNLC) */

	    break;
#endif	/* defined(HASVXFS) */

	case N_FIFO:
	    if (Selinet)
		return;
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&f, sizeof(f))) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read fifonode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (f.fn_snode.s_realvp
	    &&  !readvnode((KA_T)f.fn_snode.s_realvp, &rv)
	    &&  rv.v_data) {
		srv = 1;

#if	defined(HAS_NFS)
		if (nfsvops && nfsvops == (KA_T)rv.v_op
		&& readrnode((KA_T)rv.v_data, &r) == 0) {
		    srn = 1;
		    break;
		}
#endif	/* defined(HAS_NFS) */

#if	defined(HASVXFS)
		if (vvops && vvops == (KA_T)rv.v_op
		&&  read_vxinode(&rv, lvfs, &dev) == 0) {
		    svx = 1;
		    break;
		}
#endif	defined(HASVXFS) */

		if (uvops && uvops == (KA_T)rv.v_op
		&&  !kread((KA_T)rv.v_data, (char *)&i, sizeof(i))) {
		    sin = 1;
		    break;
		}
	    }
	    break;

#if	defined(HAS_NFS)
	case N_NFS:
	    if (Selinet)
		return;
	    if (!v.v_data || readrnode((KA_T)v.v_data, &r)) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read rnode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    srn = 1;
	    break;
#endif	/* defined(HAS_NFS) */

#if	PTXV>=400
	case N_NM:
	    if (Selinet)
		return;
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&n, sizeof(n))) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read namenode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    break;
#endif	/* PTXV>=400 */

#if	PTXV>=413
	case N_SOCK:
	    if (!v.v_snode && !v.v_stream) {

	    /*
	     * Process a vnode attached directly to a socket.
	     */
		if (v.v_rdev != NODEV
		||  !v.v_data
		||  kread((KA_T)v.v_data, (char *)&sck, sizeof(sck)))
		    break;
		if (!sck.so_pcb
		||  !sck.so_proto
		||  kread((KA_T)sck.so_proto, (char *)&pr, sizeof(pr)))
		    break;
		if (!pr.pr_domain
		||  kread((KA_T)pr.pr_domain, (char *)&dm, sizeof(dm)))
		    break;

#if	PTXV>=421
		if (dm.dom_family == AF_UNIX) {

		/*
		 * Process a UNIX domain VSOCK vnode.
		 */
		    if (usst)
			return;	/* avoid loops */
		    if (kread((KA_T)sck.so_pcb, (char *)&ub, sizeof(ub)))
			break;
		/*
		 * Read sockaddr_un.
		 */
		    if (!kread((KA_T)ub.unp_addr, (char *)&mb, sizeof(mb))
		    &&  mb.b_datap
		    &&  !kread((KA_T)mb.b_datap, (char *)&db, sizeof(db))
		    &&  db.db_base
		    &&  !kread((KA_T)db.db_base, (char *)&un, sizeof(un))
		    &&  un.sun_family == AF_UNIX
		    &&  un.sun_path)
		    {

		    /*
		     * Enter path from sockaddr_un.
		     */
			un.sun_path[sizeof(un.sun_path) - 1] - '\0';
			if ((l = strlen(un.sun_path)) >= (MAXPATHLEN - 1))
			    l = MAXPATHLEN - 1;
			(void) strncpy(Namech, un.sun_path, l);
			Namech[l] = '\0';
		    }
		    if (ub.unp_vnode) {
			
		     /*
		      * If the socket is associated with a vnode, try to
		      * interpret it instead, possibly using only the name
		      * from the sockaddr_un of this VSOCK vnode.
		      */
			if (Funix)
			    Lf->sf |= SELUNX;
			usst = 1;
			va = (KA_T)ub.unp_vnode;
			goto read_vnode;
		     }
		    (void) process_unix_socket(&sck, &ub);
		    return;
		}
# endif	/* PTXV>=421 */

		if (dm.dom_family != AF_INET)
		    break;
		switch (pr.pr_protocol) {
		case IPPROTO_ICMP:
		    (void) process_socket("ICMP", AF_INET, IPPROTO_ICMP,
			(KA_T)sck.so_pcb);
		    return;
		case IPPROTO_TCP:
		    (void) process_socket("TCP", AF_INET, IPPROTO_TCP,
			(KA_T)sck.so_pcb);
		    return;
		case IPPROTO_UDP:
		    (void) process_socket("UDP", AF_INET, IPPROTO_UDP,
			(KA_T)sck.so_pcb);
		    return;
		default:
		    (void) printiproto(pr.pr_protocol);
		    Lf->inp_ty = 2;
		}
		break;
	    }
	    /* Change node type to N_SPEC and fall through. */
	    Ntype = N_SPEC;
#endif	/* PTXV>=413 */

	case N_SPEC:
	    p = (KA_T)NULL;
	    pt = -1;
	    sop = (KA_T)NULL;
	    if (!(sa = (KA_T)v.v_snode)
	    &&  !(sa = (KA_T)v.v_data)) {
		sns = 0;
		break;
	    }
	/*
	 * Read the snode.
	 */
	    if (kread(sa, (char *)&s, sizeof(s))) {
		(void) sprintf(Namech,
		    "vnode at %s: can't read snode (%s)",
		    print_kptr(va, tbuf), print_kptr(sa, (char *)NULL));
		enter_nm(Namech);
		return;
	   }
	   sns = 1;
	   if (s.s_realvp) {

	   /*
	    * Read the "real" vnode, if there is one, then read its inode.
	    */
		if (readvnode((KA_T)s.s_realvp, &rv)) {
		    (void) sprintf(Namech,
			"snode at %s; can't read real vnode at %s",
			print_kptr(sa, tbuf),
			print_kptr((KA_T)s.s_realvp, (char *)NULL));
		    return;
		}
		srv = 1;
		if (rv.v_data) {
		    if (!kread((KA_T)v.v_data, (char *)&i, sizeof(i))) {
			sin = 1;

#if	defined(HASLFILEADD)
			Lf->ina = (KA_T)rv.v_data;
			Lf->ina_def = 1;
#endif	/* defined(HASLFILEADD) */

		
		    }
		}
	    }

#if	PTXV<400
	    if (!(sp = (KA_T)s.s_sptr))
#else	/* PTXV>=400 */
	    if (!(sp = (KA_T)v.v_stream))
#endif	/*PTXV<400 */
		break;
	/*
	 * Namech may be:
	 *    /dev/* name if it exists for s.s_dev;
	 *    "STR:" otherwise.
	 */
	    (void) strcpy(Namech, "STR:");
	    Lf->is_stream = 1;
	    k = strlen(Namech);
	    cp = NULL;
	    if ((dp = finddev(&s.s_dev, 1)) != NULL) {
		(void) strcpy(&Namech[k], dp->name);
		k += strlen(dp->name);
		if ((cp = strrchr(dp->name, '/')) != NULL)
		    cp++;
	    }
	/*
	 * Get the module names of all queue elements of the stream's
	 * sd_wrq queue.  Skip module names that end in "head" or
	 * match the last component of the /dev name.
	 *
	 * If the module name at the "end" ("head") of the stream
	 * is "tcp" or "udp" treat this as a socket file.
	 *
	 * If a stream entry contains the "sockmod" module name, save
	 * the private pointer of its queue structure as a possible
	 * struct so_so address.
	 */
	    if (kread(sp, (char *)&sd, sizeof(sd)) == 0) {
		dl = sizeof(tbuf) - 1;
		tbuf[dl] = '\0';
		qp = (KA_T)sd.sd_wrq;
		for (j = 0; qp && j < 20; j++, qp = (KA_T)q.q_next) {
		    if (kread(qp, (char *)&q, sizeof(q)))
			break;
		    if (!(ka = (KA_T)q.q_qinfo)
		    ||  kread(ka, (char *)&qi, sizeof(qi)))
			continue;
		    if (!(ka = (KA_T)qi.qi_minfo)
		    ||  kread(ka, (char *)&mi, sizeof(mi)))
			continue;
		    if (!(ka = (KA_T)mi.mi_idname) || kread(ka, tbuf, dl))
			continue;
		    if ((l = strlen(tbuf)) < 1)
			continue;
		    if (l >= 4 && strcasecmp(&tbuf[l - 4], "head") == 0)
			continue;
		    if (!q.q_next && q.q_ptr && (px = isip(tbuf)) >= 0

#if	PTXV>=413
		    &&  !kread((KA_T)q.q_ptr, (char *)&peer, sizeof(peer))
		    &&  peer.sop_inpcb
#endif	/* PTXV>=413 */

		    ) {
			process_socket(IPtab[px].nm, IPtab[px].fam,
				       IPtab[px].ipid,

#if	PTXV>=413
				       (KA_T)peer.sop_inpcb
#else	/* PTXV<413 */

				       (KA_T)q.q_ptr
#endif	/* PTXV>=413 */

			);
			    return;
		    }
		    if (cp && strcmp(cp, tbuf) == 0) {
			if (q.q_ptr && pt < 0 && (px = isip(cp)) >= 0

#if	PTXV>=413
			 && !kread((KA_T)q.q_ptr, (char *)&peer, sizeof(peer))
			 &&  peer.sop_inpcb
#endif	/* PTXV>=413 */

			) {

			/*
			 * If this is a TCP or UDP module and the
			 * queue structure has a private pointer in
			 * q_ptr, save it as a PCB address.
			 */
			    switch (IPtab[px].ipid) {
			    case IPPROTO_TCP:
				pt = 0;
				break;
			    case IPPROTO_UDP:
				pt = 1;
			    }
			    if (pt >= 0) {

#if	PTXV>=413
				p = (KA_T)peer.sop_inpcb;
#else	/* PTXV<413 */
				p = (KA_T)q.q_ptr;
#endif	/* PTXV>=413 */

				(void) strcpy(Lf->iproto, IPtab[px].nm);
			    }
			}
			continue;
		    }
		    if (q.q_ptr && strcasecmp(tbuf, "sockmod") == 0 && !sop)
			sop = (KA_T)q.q_ptr;
		    if (k) {
			if ((k + 2) > (MAXPATHLEN - 1))
			    break;
			(void) strcpy(&Namech[k], "->");
			k += 2;
		    }
		    if ((k + l) > (MAXPATHLEN - 1))
			break;
		    (void) strcpy(&Namech[k], tbuf);
		    k += l;
		}
	    }
	    if (p && pt >= 0) {
		sop = (KA_T)NULL;

	/*
	 * If the stream has a TCP or UDP module with a PCB
	 * pointer, print any associated local and foreign
	 * Internet addresses.
	 */
		if (kread(p, (char *)&pcb, sizeof(pcb)))
		    break;
		if (Fnet)
		    Lf->sf |= SELNET;
		if ((k + 1) > (MAXPATHLEN - 1))
		    break;
		Namech[k++] = ' ';
		Namech[k] = '\0';
	 	
	    /*
	     * Print the local and foreign addresses from the PCB.
	     */
		la = (struct in_addr *)&pcb.inp_laddr;
		lp = (int)ntohs(pcb.inp_lport);

#if	defined(HASINADDRSTR)
		if (pcb.inp_faddr.s_addr != INADDR_ANY || pcb.inp_fport != 0)
#else	/* !defined(HASINADDRSTR) */
		if (pcb.inp_faddr != INADDR_ANY || pcb.inp_fport != 0)
#endif	/* defined(HASINADDRSTR) */

		{
		    fa = (struct in_addr *)&pcb.inp_faddr;
		    fp = (int)ntohs(pcb.inp_fport);
		}
		if (fa || la)
		    (void) ent_inaddr(la, lp, fa, fp);
		Lf->inp_ty = 2;
	    } else if (sop) {
		    
	    /*
	     * The stream has a "sockmod" member and its queue structure
	     * has a private pointer; see if it points to Unix domain
	     * socket information; if it does, read its so_so structure.
	     */
		if (kread(sop, (char *)&so, sizeof(so)))
		    sop = (KA_T)NULL;
	    }
	    break;
	case N_REGLR:
	default:
	    if (Selinet)
		return;
	    if (v.v_type != VBLK) {
		if (!v.v_data
		||  kread((KA_T)v.v_data, (char *)&i, sizeof(i))) {
		    (void) sprintf(Namech,
			"vnode at %s: can't read inode (%s)",
			print_kptr(va, tbuf),
			print_kptr((KA_T)v.v_data, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		sin = 1;
	    }

#if	defined(HASLFILEADD)
	    Lf->ina = (KA_T)v.v_data;
	    Lf->ina_def = 1;
#endif	/* defined(HASLFILEADD) */

	}
	if (Selinet)
	    return;
/*
 * Get device and type for printing.
 */
	type = v.v_type;
	switch (Ntype) {

#if	defined(HAS_CDFS)
	case N_CDFS:
	    /* dev is set in read_cdnode() */
	    break;
#endif	/* defined(HAS_CDFS) */


#if	defined(HAS_NFS)
	case N_NFS:
	    if (lvfs) {
		dev = lvfs->dev;
		Lf->dev_def = 1;
	    }
	    break;
#endif	/* defined(HAS_NFS) */

#if	PTXV>=400
	case N_NM:
	    dev = n.nm_vattr.va_fsid;
	    Lf->dev_def = 1;
	    break;
#endif	/* PTXV>=400 */

	case N_FIFO:
	    if (srv && sin && lvfs) {
		dev = lvfs->dev;
		Lf->dev_def = 1;
	    }

#if	defined(HAS_NFS)
	    else if (srv && srn && lvfs) {
		dev = lvfs->dev;
		Lf->dev_def = 1;
	    }
#endif	/* defined(HAS_NFS) */

#if	defined(HASVXFS)
	    else if (srv && svx)
		Lf->dev_def = 1;	/* dev is set in read_vxnode() */
#endif	/* defined(HASVXFS) */

	    else
		enter_dev_ch(print_kptr((KA_T)va, (char *)NULL));
	    break;

#if	defined(HASVXFS)
	case N_VXFS:
	    /* dev is set in read_vxnode() */
	    break;
#endif	/* defined(HASVXFS) */

#if	PTXV>=413
	case N_SOCK:
	    if (v.v_rdev == NODEV)
		enter_dev_ch("NODEV");
	    else {
		dev = v.v_rdev;
		Lf->dev_def = 1;
	    }
	    break;
#endif	/* PTXV>=413 */

	case N_SPEC:
	    if (sop)
		dev = so.lux_dev.addr.tu_addr.dev;
	    else
		dev = sns ? s.s_dev : v.v_rdev;
	    Lf->dev_def = 1;
	    break;
	default:
	    if (type == VCHR || type == VBLK) {
		dev = v.v_rdev;
		Lf->dev_def = 1;
	    } else {
		if (srv) {
		    dev = rv.v_rdev;
		    Lf->dev_def = 1;
		} else if (i.i_devvp
		       &&  !kread((KA_T)i.i_devvp, (char *)&uv, sizeof(uv)))
		{
		    dev = uv.v_rdev;
		    Lf->dev_def = 1;
		}
	    }
	}
/*
 * Obtain the inode number.
 */
	switch (Ntype) {

#if	defined(HAS_CDFS)
	case N_CDFS:
	    /* Lf->inode and Lf->inp_ty are set in read_cdnode() */
	    break;
#endif	/* defined(HAS_CDFS) */

#if	defined(HASVXFS)
	case N_VXFS:
	    /* Lf->inode and Lf->inp_ty are set in read_vxnode() */
	    break;
#endif	/* defined(HASVXFS) */


	case N_FIFO:
	    if (sin) {
		Lf->inode = (unsigned long)i.i_number;
		Lf->inp_ty = 1;
	    }

#if	defined(HAS_NFS)
	    else if (srn) {
		Lf->inode = (unsigned long)r.r_nfsattr.na_nodeid;
		Lf->inp_ty = 1;
	    }
#endif	/* defined(HAS_NFS) */

#if	defined(HASVXFS)
	    else if (svx)
		Lf->inp_ty = 1;		/* inode set in read_vxnode() */
#endif	/* defined(HASVXFS) */

	     break;

#if	defined(HAS_NFS)
	case N_NFS:
	     Lf->inode = (unsigned long)r.r_nfsattr.na_nodeid;
	     Lf->inp_ty = 1;
	     break;
#endif	/* defined(HAS_NFS) */

#if	PTXV>=400
	case N_NM:
	     Lf->inode = (unsigned long)n.nm_vattr.va_nodeid;
	     Lf->inp_ty = 1;
	     break;
#endif	/* PTXV>=400 */

	case N_REGLR:
	     if (sin) {
		Lf->inode = (unsigned long)i.i_number;
		Lf->inp_ty = 1;
	     }
	     break;

#if	PTXV>=413
	case N_SOCK:
	    /* The sock_vnodeops type vnode has no inode number. */
	    if (!Lf->iproto[0]) {
		(void) strcpy(Lf->iproto, "SOCK");
		Lf->inp_ty = 2;
	    }
	    break;
#endif	/* PTXV>=413 */

	case N_SPEC:
	    if (sop) {
		Lf->inode = (unsigned long)so.lux_dev.addr.tu_addr.ino;
		Lf->inp_ty = 1;
	    }
	    break;
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
	     Lf->off_def = 1;
	else {
	     switch (Ntype) {

#if	defined(HAS_CDFS)
	     case N_CDFS:
		/* Lf->sz and Lf->sz_def are set in read_cdnode() */
		break;
#endif	/* defined(HAS_CDFS) */

#if	defined(HASVXFS)
	     case N_VXFS:
		/* Lf->sz and Lf->sz_def are set in read_vxnode() */
		break;
#endif	/* defined(HASVXFS) */

	     case N_FIFO:

#if	defined(HASVXFS)
		if (srv && svx)
		    Lf->sz_def = 0;	/* unset what read_vxnode() set */
#endif	/* defined(HASVXFS) */


#if	PTXV<400
		if (Fsize || (Lf->access != 'r' && Lf->access != 'w')) {
		    (void) sprintf(Namech, "rd=%#x; wr=%#x",
			f.fn_rptr, f.fn_wptr);
		    Lf->sz = (SZOFFTYPE)f.fn_size;
		    Lf->sz_def = 1;
		    break;
		}
		Lf->off = (unsigned long)
			  (Lf->access == 'r') ? f.fn_rptr : f.fn_wptr;
		(void) sprintf(Namech, "%s=%#x",
			       (Lf->access == 'r') ? "wr" : "rd",
			       (Lf->access == 'r') ? f.fn_wptr : f.fn_rptr);
#endif	/* PTXV<400 */

		Lf->off_def = 1;
		break;

#if	defined(HAS_NFS)
	     case N_NFS:
		Lf->sz = (SZOFFTYPE)r.r_size;
		Lf->sz_def = 1;
		break;
#endif	/* defined(HAS_NFS) */

#if	PTXV>=400
	     case N_NM:
		Lf->sz = (SZOFFTYPE)n.nm_vattr.va_size;
		Lf->sz_def = 1;
		break;
#endif	/* PTXV>=400 */

	     case N_REGLR:
		if (type == VREG || type == VDIR || (type == VSOCK && usst)) {
		    if (sin) {
			Lf->sz = (SZOFFTYPE)

#if	PTXV<420
				  i.i_ic.ic_size.val[0];
#else	/* PTXV>=420 */
				  i.i_ic.ic_size;
#endif	/* PTXV<420 */

			Lf->sz_def = 1;
		    }
		} else if ((type == VCHR || type == VBLK) && !Fsize)
		    Lf->off_def = 1;
		break;

#if	PTXV>=413
	     case N_SOCK:
		Lf->off_def = 1;
		break;
#endif	/* PTXV>=413 */

	     case N_SPEC:
		if (type == VCHR && !Fsize)
		    Lf->off_def = 1;
		break;
	    }
	}

#if	defined(HAS_NFS)
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
#endif	/* defined(HAS_NFS) */

/*
 * Save the file system names.
 */
	if (lvfs) {
	    Lf->fsdir = lvfs->dir;
	    Lf->fsdev = lvfs->fsname;

#if	defined(HASFSINO)
	    Lf->fs_ino = lvfs->fs_ino;
#endif	/* defined(HASFSINO) */

	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {

	case VNON:
	    ty ="VNON";
	    break;
	case VREG:
	case VDIR:
	    ty = (type == VREG) ? "VREG" : "VDIR";
	    Lf->dev = dev;
	    break;
	case VBLK:
	    ty = "VBLK";
	    Lf->dev = dev;
	    Ntype = N_BLK;
		break;
	case VCHR:
	    ty = "VCHR";
	    Lf->dev = dev;
	    if (Ntype != N_SPEC || !sop)
		Ntype = N_CHR;
	    break;
	case VLNK:
	    ty = "VLNK";
	    break;
	case VSOCK:
	    ty = usst ? "unix" : "SOCK";
	    Lf->dev = dev;
	    break;
	case VBAD:
	    ty = "VBAD";
	    break;
	case VFIFO:
	    Lf->dev = dev;
	    ty = "FIFO";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%4d", type);
	    (void) strcpy(Namech, "unknown type");
	    ty = NULL;
	}
	if (ty)
	    (void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;

#if	defined(HASBLKDEV)
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VBLK && Lf->dev_def)
	    find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
	    find_ch_ino();

#if	PTXV>=400
/*
 * If a PTX 4 or greater FIFO node has a mate, if it has no "real" inode
 * or rnode, and if nothing has been put in Namech already, put the mate's
 * address there.
 */
	if (Ntype == N_FIFO && f.fn_mate && Namech[0] == '\0' && !sin && !srn)
	    (void) sprintf(Namech, "->%s",
		print_kptr((KA_T)f.fn_mate, (char *)NULL));
#endif	/* PTXV>=400 */

/*
 * If this is a "sockmod" special file and it has a so_so structure,
 * try to retrieve the Unix domain socket path.
 */
	if (Ntype == N_SPEC && sop) {
	    if (Funix)
		Lf->sf |= SELUNX;
	    (void) strcpy(Lf->type, "unix");
	    if ((Namech[0] == '\0' || !strncmp(Namech, "STR:", strlen("STR:")))
	    &&  so.laddr.buf
	    &&  so.laddr.len == sizeof(ua))
	    {
		if (kread((KA_T)so.laddr.buf, (char *)&ua, sizeof(ua)) == 0) {
		    ua.sun_path[sizeof(ua.sun_path) - 1] = '\0';
		    (void) strcpy(Namech, ua.sun_path);
		}
	    }
	}
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, type))
	    Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}
