/*

 gzilla

 Copyright 1997 Raph Levien <raph@acm.org>

 This code is free for commercial and non-commercial use,
 modification, and redistribution, as long as the source code release,
 startup screen, or product packaging includes this copyright notice.

 */


#include <ctype.h>		/* for tolower */
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <errno.h>            /* for errno */

#include <gtk/gtk.h>

#include "gzillabytesink.h"

typedef struct _GzillaFile GzillaFile;

struct _GzillaFile {
  gint fd;
  GzillaByteSink *bytesink;
  gint abort_id;
};

static gint num_files;
static gint num_files_max;
static GzillaFile *files;

static gint idle_tag = 0;

void
gzilla_file_init (void)
{
  num_files = 0;
  num_files_max = 16;
  files = g_new (GzillaFile, num_files_max);
}


/* The idle function. For each open file, try reading a buffer and
   writing it out to the bytesink. If we reach eof, then close the
   file and bytesink. */

static gint
gzilla_file_idle_func (void)
{
  gint i;
  char buf[8192];
  gint num_bytes;

  for (i = 0; i < num_files; i++)
    {
      num_bytes = read (files[i].fd, buf, sizeof (buf));
      if (num_bytes < 0)
	{
	  if (errno == EINTR)
	    break;
	  num_bytes = 0;
	}
      if (num_bytes > 0)
	gzilla_bytesink_write (files[i].bytesink, buf, num_bytes);
      else
	{
	  gtk_signal_disconnect (GTK_OBJECT (files[i].bytesink),
				 files[i].abort_id);
	  close (files[i].fd);
	  gzilla_bytesink_close (files[i].bytesink);
	  files[i--] = files[--num_files];
	}
    }
  if (num_files > 0)
    return TRUE;
  else
    {
      idle_tag = 0;
      return FALSE;
    }
}

/* This function gets called when the bytesink we're writing into gets
   aborted. It shuts down the connection and deletes the file
   structure, freeing any resources that were taken. */

void
gzilla_file_status (GzillaByteSink *bytesink,
		    GzillaStatusDir dir,
		    gboolean abort,
		    GzillaStatusMeaning meaning,
		    const char *text,
		    void *data)
{
  gint i;

  if (!abort)
    return;

  for (i = 0; i < num_files; i++)
    {
      if (files[i].bytesink == bytesink)
	break;
    }
  if (i < num_files)
    {
      close (files[i].fd);
      files[i] = files[--num_files];
    }
  else
    g_warning ("gzilla_file_abort: trying to abort a nonexistent file\n");
}

/* Return TRUE if the extension matches that of the filename. */

static gboolean
gzilla_file_ext (const char *filename, const char *ext)
{
  gint i, j;

  i = strlen(filename);
  while (i > 0 && filename[i - 1] != '.')
    i--;
  if (i == 0)
    return FALSE;
  for (j = 0; ext[j] != '\0'; j++)
    if (tolower(filename[i++]) != tolower(ext[j]))
      return FALSE;
  return (filename[i] == '\0');
}

/* Create a new file connection for URL url, and asynchronously
   feed the bytes that come back to bytesink. */
void
gzilla_file_get (const char *url,
		 GzillaByteSink *bytesink,
		 void (*status_callback) (const char *status, void *data),
		 void *data)
{
  char *filename;
  gint fd;
  char *content_type;
  char header[1024];

  filename = (char *)url + 5;
  fd = open(filename, O_RDONLY);
  if (fd >= 0)
    {
      /* set close-on-exec */
      fcntl (fd, F_SETFD, FD_CLOEXEC | fcntl(fd, F_GETFD));

      /* could set nonblocking too, which might be helpful on some file
	 systems (like NFS), but we won't bother. */

      if (gzilla_file_ext(filename, "gif"))
	{
	  content_type = "image/gif";
	}
      else if (gzilla_file_ext(filename, "jpg") ||
	       gzilla_file_ext(filename, "jpeg"))
	{
	  content_type = "image/jpeg";
	}
      else if (gzilla_file_ext(filename, "html") ||
	       gzilla_file_ext(filename, "htm"))
	{
	  content_type = "text/html";
	}
      else
	{
	  content_type = "text/plain";
	}
      /* todo: add size fields, etc. */
      sprintf (header,
	       "HTTP/1.0 200 Document found\n"
	       "Server: gzilla internal\n"
	       "Content-type: %s\n"
	       "\n",
	       content_type);
      gzilla_bytesink_write (bytesink, header, strlen (header));

      if (num_files == num_files_max)
	{
	  num_files_max <<= 1;
	  files = g_realloc (files, num_files_max * sizeof(GzillaFile));
	}
      files[num_files].fd = fd;
      files[num_files].bytesink = bytesink;
      if (idle_tag == 0)
	idle_tag = gtk_idle_add ((GtkFunction) gzilla_file_idle_func, NULL);

      /* add abort handler */
      files[num_files].abort_id =
	gtk_signal_connect (GTK_OBJECT (bytesink),
			    "status",
			    (GtkSignalFunc) gzilla_file_status,
			    NULL);

      num_files++;
    } else {
      sprintf (header,
	       "HTTP/1.0 404 Not Found\n"
	       "Server: gzilla internal\n"
	       "Content-type: text/html\n"
	       "\n"
	       "<html><head><title>404 Not Found</title></head>\n"
	       "<body><h1>404 Not Found</h1>"
	       "<p>The requested file %.750s was not found in the filesystem.</p>\n"
	       "</body></html>\n",
	       filename);
      gzilla_bytesink_write (bytesink, header, strlen (header));
      gzilla_bytesink_close (bytesink);
    }
}
