%{

/*****************************************************************************/
/*                                                                           */
/*  wmmail - WindowMaker Mail                                                */
/*                                                                           */
/*  Copyright (c) 1996, 1997, 1998  Per Liden                                */
/*  Copyright (c) 1997, 1998  Bryan Chan                                     */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*                                                                           */
/*    wmmailrc_y.y: parser of the wmmail configuration file                  */
/*                                                                           */
/*****************************************************************************/

#include "wmmail.h"

extern char *yytext;
extern int yydebug;

/* function prototypes */
void rc_add_frame_to_anim(int, char *);
void rc_add_mailbox(char *);
void rc_handle_file(char *);
void rc_load_frame(XpmIcon *, char *);
void rc_init_mailbox(Mailbox *, char *);

%}

%union {
  char *string;
  int   number;
}

%token ALWAYS_NEW_MAIL_EXECUTE  ANIM_SPEED  END  EOL  EXECUTE 
%token EXECUTE_ON_UPDATE  MAIL_FILE  NEW_MAIL_T  NEW_MAIL_EXECUTE
%token NO_BEEP  NO_MAIL_T  NUM_OF_MSG_COLOR NUM_OF_MSG_FONT  NUM_OF_MSG_MODE
%token NUM_OF_MSG_POSITION  OLD_MAIL_T  TIME_STAMP_MODE  UPDATE

%token <string> STRING
%token <number> NUMBER

%%

rcfile	:
	| rclines
	;

rclines	: EOL
	| statement
	| rclines EOL
	| rclines statement
	;

statement :
	  ALWAYS_NEW_MAIL_EXECUTE	{ always_new_mail_exec = TRUE; }
	| ANIM_SPEED NUMBER NUMBER NUMBER   { anim_speed[NO_MAIL]  = $2 * 100;
					      anim_speed[OLD_MAIL] = $3 * 100;
					      anim_speed[NEW_MAIL] = $4 * 100; }
	| EXECUTE STRING		{ exec_on_click_path  = strdup($2); }
	| EXECUTE_ON_UPDATE STRING	{ exec_on_update_path = strdup($2); }
	| NEW_MAIL_EXECUTE STRING	{ exec_on_new_path    = strdup($2); }
	| NO_BEEP			{ use_beep = FALSE; }
	| NUM_OF_MSG_COLOR STRING	{ wmmail_color = strdup($2); }
	| NUM_OF_MSG_FONT STRING	{ wmmail_font = XLoadFont(disp, $2); }
	| NUM_OF_MSG_MODE NUMBER	{ 
					  if (use_time_stamp && $2)
					  {
					    croak("cannot use NumOfMsgMode "
					          "with TimeStampMode");
					    exit(ERR_SYNTAX);
					  }
					  else if (mbox_list != NULL && 
                                                   mbox_list->next != NULL &&
					           $2)
					  {
					    croak("cannot use NumOfMsgMode "
					          "with more than one mailbox");
					    exit(ERR_SYNTAX);
					  }
					  else
					    num_of_msg_mode = $2;
					}
	| NUM_OF_MSG_POSITION NUMBER NUMBER	{ 
						  num_of_msg_x = $2;
						  num_of_msg_y = $3;
						}
	| TIME_STAMP_MODE		{
					  if (num_of_msg_mode) 
					  {
					    croak("cannot use TimeStampMode "
					          "with NumOfMsgMode");
					    exit(ERR_SYNTAX);
					  }
					  else
					    use_time_stamp = TRUE;
					}
	| UPDATE NUMBER			{ update_interval = $2; }
	| MAIL_FILE EOL flist END	{ rc_mode = NORMAL; }
	| NO_MAIL_T EOL flist END	{ rc_mode = NORMAL; }
	| OLD_MAIL_T EOL flist END	{ rc_mode = NORMAL; }
	| NEW_MAIL_T EOL flist END	{ rc_mode = NORMAL; }
	;

flist	: file STRING EOL 		{ rc_handle_file($2); }
	;

file	:				{ /* empty */ }
	| file STRING EOL		{ rc_handle_file($2); }
	;

%%


yyerror (s)
char *s;
{
  croak("%s line %d: %s at %s", config_file_path, rc_lineno, s, yytext);
  rc_errno++;
}


void rc_add_frame_to_anim(int which, char *path)
{
  XpmIcon *current;

  if (animations[which] == NULL)
  {
    animations[which] = (XpmIcon *) safe_malloc(sizeof(XpmIcon));
    current = animations[which];
  }
  else
  {
    for (current = animations[which]; 
         current->next != animations[which];	/* until wrapped around */
         current = current->next);

    current->next = (XpmIcon *) safe_malloc(sizeof(XpmIcon));
    current = current->next;
  }

  rc_load_frame(current, path);
  current->next = animations[which];	/* make a looping animation */
}


void rc_add_mailbox(char *path)
{
  if (path == NULL)
    return;

  if (mbox_list == NULL)
  {
    mbox_list = (Mailbox *) safe_malloc(sizeof(Mailbox));

    rc_init_mailbox(mbox_list, path);
  }
  else
  {
    Mailbox *i;

    if (num_of_msg_mode)
    {
      croak("cannot use NumOfMsgMode with more than one mailbox");
      exit(ERR_SYNTAX);
    }

    for (i = mbox_list; i->next != NULL; i = i->next);

    i->next = (Mailbox *) safe_malloc(sizeof(Mailbox));

    rc_init_mailbox(i->next, path);
  }
}


void rc_handle_file(char *path)
{
  switch(rc_mode)
  {
    case NORMAL:
      croak("%s line %d: parse error at %s",
            config_file_path, rc_lineno, yytext);
      rc_errno++;
      break;
    case EXPECT_ANIM_NO_MAIL:
      rc_add_frame_to_anim(NO_MAIL, path);
      break;

    case EXPECT_ANIM_OLD_MAIL:
      rc_add_frame_to_anim(OLD_MAIL, path);
      break;

    case EXPECT_ANIM_NEW_MAIL:
      rc_add_frame_to_anim(NEW_MAIL, path);
      break;

    case EXPECT_MAIL_FILE:
      rc_add_mailbox(path);
      break;
  }
}


void rc_load_frame(XpmIcon *anim, char *path)
{
  int ret;

  anim->attributes.valuemask   = (XpmExactColors | XpmCloseness);
  anim->attributes.exactColors = FALSE;
  anim->attributes.closeness   = 40000;

  ret = XpmReadFileToPixmap(disp, root, path, &(anim->pixmap), &(anim->mask),
                            &(anim->attributes));

  switch(ret)
  {
    case XpmSuccess: 
      break;

    case XpmColorError: 
    case XpmColorFailed: 
      croak("not enough free colours loading '%s'", path);
      break;

    case XpmOpenFailed:
      croak("cannot open '%s'", path);
      break;

    case XpmFileInvalid:
      croak("invalid XPM file '%s'", path);
      break;

    case XpmNoMemory:
      croak("not enough free memory loading '%s'", path);
      break;

    default:
      croak("unknown error (%d) loading '%s'", ret, path);
      break;
  }

  if (ret < 0)
    exit(ERR_XPM);

  if (anim->mask == 0)
  {
    croak("XPM file '%s' does not have a transparent colour");
    exit(ERR_XPM);
  }
}


void rc_init_mailbox(Mailbox *new_mbox, char *path)
{
  struct stat t;

  new_mbox->path           = strdup(path);
  new_mbox->last_check     = 0;
  new_mbox->mail_count     = 0;
  new_mbox->new_mail_count = 0;
  new_mbox->status         = NO_MAIL;
  new_mbox->next           = NULL;

  if (!stat(path, &t))
  {
    /* if named mailbox is a directory, assume that it is a MH folder
     * otherwise, assume that it is a UNIX-style mailbox
     */
    if (S_ISDIR(t.st_mode))
      new_mbox->mbox_type  = TYPE_MH;
    else
      new_mbox->mbox_type  = TYPE_UNIX;

    new_mbox->size         = t.st_size;
  }
  else
  {
    croak("cannot stat mailbox %s", path);
    exit(ERR_IO);
  }
}

