/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995, 1996, 1997
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: snapshot.c,v 3.19 1997/10/31 20:54:15 erik Rel $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/param.h>
#include <assert.h>
#include "z80.h"
#include "debug.h"
#include "mem.h"
#include "resource.h"
#include "io.h"
#include "util.h"
#include "dialog.h"
#ifdef XZX_IF1
#include "if1.h"
#endif
#include "main.h"
#include "snapshot.h"


#if defined(sun) && !defined(SVR4)
#define NO_FPOS_T
#endif


struct z80_header {
	uns8 a, f;
	uns8 c, b;
	uns8 l, h;
	uns8 lpc, hpc;
	uns8 lsp, hsp;
	uns8 i;
	uns8 r;
	uns8 flags1;
	uns8 e, d;
	uns8 c_alt, b_alt;
	uns8 e_alt, d_alt;
	uns8 l_alt, h_alt;
	uns8 a_alt, f_alt;
	uns8 liy, hiy;
	uns8 lix, hix;
	uns8 iff1;
	uns8 iff2;
	uns8 flags2;

	uns8 add_header[2];
	uns8 lpc1, hpc1;
	uns8 hw_mode;
	uns8 last0x7FFD;
	uns8 if1_paged;
	uns8 flags3;
	uns8 last0xFFFD;
	uns8 psg[16];

	uns8 tstates[3];
	uns8 flags4;
	uns8 mgt_paged;
	uns8 mf128_paged;
	uns8 ram_at_0x0000;
	uns8 ram_at_0x2000;
	uns8 joystick_keys[10];
	uns8 key_names[10];
	uns8 mgt_type;
	uns8 disciple_inhibit;
	uns8 disciple_rom;

	uns8 last0x1FFD;
};
#define Z80_V1_HDRLGT		30
#define Z80_V2_HDRLGT		55
#define Z80_V3_HDRLGT		86
#define Z80_V3X_HDRLGT		87
struct z80_page_header {
	uns8 page_length[2];
	uns8 page_number;
};
struct slt_block {
	short type;
	short identifier;
	long length;
};

struct sna_header {
	uns8 i;
	uns8 l_alt, h_alt;
	uns8 e_alt, d_alt;
	uns8 c_alt, b_alt;
	uns8 f_alt, a_alt;
	uns8 l, h;
	uns8 e, d;
	uns8 c, b;
	uns8 liy, hiy;
	uns8 lix, hix;
	uns8 iff2;
	uns8 r;
	uns8 f, a;
	uns8 lsp, hsp;
	uns8 im;
	uns8 border;
};

static int writeSnaFormat(char *);
static int readSnaFormat(char *);
static int writeZ80Page(int, int, FILE *);
static int writeZ80Format(char *, FILE **);
static int readZ80Page(FILE *, int, int);
static int readZ80VX(FILE *);
static int readZ80V3(FILE *);
static int readZ80V2(FILE *);
static int readZ80Format(char *, FILE **);
static int readSltFormat(char *);
static int readSltBlock(void);

char LastSnapshotName[MAXPATHLEN];
int InSlt = 0;


static FILE *sltFile = NULL;
#ifdef NO_FPOS_T
static long sltPos;
static long blkPos;
#else
static fpos_t sltPos;
static fpos_t blkPos;
#endif
static struct slt_block *sltBlocks = NULL;


int
WriteScreenDump(char *fname)
{
	FILE *ofp;
	if (!(ofp = Fopen(fname, "wb")))
	{
		Msg(M_PERR, "couldn't open <%s> for writing", fname);
		return -1;
	}

	if (fwrite(RealMemory[ScreenSelect], 1, PIXEL_LENGTH + ATTR_LENGTH, ofp) !=
		PIXEL_LENGTH + ATTR_LENGTH)
	{
		Msg(M_PERR, "couldn't write screen dump");
		(void)fclose(ofp);
		return -1;
	}
	(void)fclose(ofp);
	return 0;
}

static int
writeSnaFormat(char *fname)
{
	FILE *ofp;
	struct sna_header header;
	if (!(ofp = Fopen(fname, "wb")))
	{
		Msg(M_PERR, "couldn't open <%s> for writing", fname);
		return -1;
	}
	PUSH(PC);

	header.i     = I;
	header.l_alt = L1;
	header.h_alt = H1;
	header.e_alt = E1;
	header.d_alt = D1;
	header.c_alt = C1;
	header.b_alt = B1;
	header.f_alt = F1;
	header.a_alt = A1;
	header.l     = L;
	header.h     = H;
	header.c     = C;
	header.b     = B;
	header.liy   = LIY;
	header.hiy   = HIY;
	header.lix   = LIX;
	header.hix   = HIX;
	header.iff2  = IFF2 << 2;
	header.r     = R;
	header.f     = F;
	header.a     = A;
	header.lsp   = LSP;
	header.hsp   = HSP;
	header.im    = IM;
	header.border = LastBorderColor;

	if (fwrite(&header, 1, sizeof(header), ofp) != sizeof(header))
	{
		Msg(M_PERR, "couldn't write snapshot header");
		POP(PC);
		(void)fclose(ofp);
		return -1;
	}

	if (fwrite(RealMemory[RAM5], 1, XZX_PAGESIZE, ofp) != XZX_PAGESIZE
		|| fwrite(RealMemory[RAM2], 1, XZX_PAGESIZE, ofp) != XZX_PAGESIZE
		|| fwrite(RealMemory[RAM0], 1, XZX_PAGESIZE, ofp) != XZX_PAGESIZE)
	{
		Msg(M_PERR, "couldn't write memory dump");
		POP(PC);
		(void)fclose(ofp);
		return -1;
	}
	POP(PC);
	(void)fclose(ofp);
	return 0;
}

static int
readSnaFormat(char *fname)
{
	FILE *ifp;
	struct stat buf;
	struct sna_header header;
	if (stat(fname, &buf) || buf.st_size > 49179)
	{
		Msg(M_ERR, "snapshot <%s> is not in 48K SNA format", fname);
		return -1;
	}
	if (!(ifp = Fopen(fname, "rb")))
	{
		Msg(M_PERR, "couldn't open <%s> for reading", fname);
		return -1;
	}
	if (fread(&header, 1, sizeof(header), ifp) != sizeof(header))
	{
		Msg(M_PERR, "couldn't read snapshot header");
		(void)fclose(ifp);
		return -1;
	}
	Msg(M_INFO, "snapshot format is SNA");
	Msg(M_INFO, "hardware type is 48K");
#ifdef XZX_IF1
	If1OnOff(False);
#endif
	(void)SelectModel(48, False);
	I    = header.i;
	L1   = header.l_alt;
	H1   = header.h_alt;
	E1   = header.e_alt;
	D1   = header.d_alt;
	C1   = header.c_alt;
	B1   = header.b_alt;
	F1   = header.f_alt;
	A1   = header.a_alt;
	L    = header.l;
	H    = header.h;
	E    = header.e;
	D    = header.d;
	C    = header.c;
	B    = header.b;
	LIY  = header.liy;
	HIY  = header.hiy;
	LIX  = header.lix;
	HIX  = header.hix;
	IFF2 = (header.iff2 >> 2) & 0x01;
	R    = header.r;
	F    = header.f;
	A    = header.a;
	LSP  = header.lsp;
	HSP  = header.hsp;
	IM   = header.im;
	OutByte(P_ULA, header.border);
	if (fread(RealMemory[RAM5], 1, XZX_PAGESIZE, ifp) != XZX_PAGESIZE
		|| fread(RealMemory[RAM2], 1, XZX_PAGESIZE, ifp) != XZX_PAGESIZE
		|| fread(RealMemory[RAM0], 1, XZX_PAGESIZE, ifp) != XZX_PAGESIZE)
	{
		Msg(M_ERR, "premature end on <%s>", fname);
		(void)fclose(ifp);
		return -1;
	}
	RETN();
	(void)fclose(ifp);
	return 0;
}

static int
writeZ80Page(int xzx_page, int z80_page, FILE *ofp)
{
	int i, j;
	int nrep;
	static uns8 buffer[sizeof(struct z80_page_header) + XZX_PAGESIZE + 4];
	int nbytes = 3;
	uns8 byte;

	i = 0;
	while (i < XZX_PAGESIZE && nbytes < XZX_PAGESIZE)
	{
		byte = RealMemory[xzx_page][i];
		nrep = 0;
		j = i;
		for (;;)
		{
			nrep++;
			j++;
			if (nrep == 255
				|| j == XZX_PAGESIZE
				|| RealMemory[xzx_page][j] != byte)
			{
				break;
			}
		}
		i += nrep;

		if (nrep < 5 && byte != 0xed)
		{

			for (j = 0; j < nrep; j++)
			{
				buffer[nbytes++] = byte;
			}
		}
		else if (nrep == 1 && byte == 0xed)
		{

			buffer[nbytes++] = 0xed;
			if (i < XZX_PAGESIZE)
			{
				buffer[nbytes++] = RealMemory[xzx_page][i++];
			}
		}
		else
		{

			buffer[nbytes++] = 0xed;
			buffer[nbytes++] = 0xed;
			buffer[nbytes++] = (uns8)nrep;
			buffer[nbytes++] = byte;
		}
	}
	if (i < XZX_PAGESIZE - 1
		|| nbytes >= sizeof(struct z80_page_header) + XZX_PAGESIZE)
	{

		(void)memcpy(&buffer[sizeof(struct z80_page_header)],
					 RealMemory[xzx_page], XZX_PAGESIZE);
		nbytes = sizeof(struct z80_page_header) + XZX_PAGESIZE;
		buffer[0] = LOWBYTE(-1);
		buffer[1] = HIGHBYTE(-1);
	}
	else
	{
		nbytes -= 3;
		buffer[0] = LOWBYTE(nbytes);
		buffer[1] = HIGHBYTE(nbytes);
		nbytes += 3;
	}
	buffer[2] = (uns8)z80_page;
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		Msg(M_DEBUG, "write page: XZX-page %2d, block length %5d", xzx_page,
			nbytes - 3);
	}
#endif

	if (fwrite((void *)buffer, 1, nbytes, ofp) != nbytes)
	{
		Msg(M_PERR, "couldn't write memory block %d", xzx_page);
		return -1;
	}
	return 0;
}

static int
writeZ80Format(char *fname, FILE **retofp)
{
	FILE *ofp;
	struct z80_header header;
	int i;
	if (!(ofp = Fopen(fname, "wb")))
	{
		Msg(M_PERR, "couldn't open <%s> for writing", fname);
		return -1;
	}

	header.a   = A;
	header.f   = F;
	header.c   = C;
	header.b   = B;
	header.l   = L;
	header.h   = H;
	header.lpc = 0;
	header.hpc = 0;
	header.lsp = LSP;
	header.hsp = HSP;
	header.i   = I;
	header.r   = R & 0x7f;
	header.flags1 = (R & 0x80) >> 7 ;
	header.flags1 |= LastBorderColor << 1;
	header.e     = E;
	header.d     = D;
	header.c_alt = C1;
	header.b_alt = B1;
	header.e_alt = E1;
	header.d_alt = D1;
	header.l_alt = L1;
	header.h_alt = H1;
	header.f_alt = F1;
	header.a_alt = A1;
	header.liy   = LIY;
	header.hiy   = HIY;
	header.lix   = LIX;
	header.hix   = HIX;
	header.iff1  = IFF1;
	header.iff2  = IFF2;
	header.flags2 = IM;
	header.flags2 |= GETCFG(issue) == 2 ? 0x04 : 0x00;
	header.flags2 |= 0x40;

	header.add_header[0] = LOWBYTE((Z80_V3X_HDRLGT - Z80_V1_HDRLGT - 2));
	header.add_header[1] = HIGHBYTE((Z80_V3X_HDRLGT - Z80_V1_HDRLGT - 2));
	header.lpc1 = LPC;
	header.hpc1 = HPC;
	switch (GETCFG(machine))
	{
		default:
		case 48:
			header.hw_mode = 0;
			break;
		case 128:
			header.hw_mode = 4;
			break;
#ifdef XZX_PLUS3
		case 3:
			header.hw_mode = 7;
			break;
#endif
	}
#ifdef XZX_IF1
	header.hw_mode += GETCFG(if1_active) ? 1 : 0;
#endif
	header.last0x7FFD = Last0x7FFD;
#ifdef XZX_IF1
	header.if1_paged = RPAGE(0) == IF1ROM ? 0xff : 0x00;
#else
	header.if1_paged = 0;
#endif
	header.flags3 = 3;
	header.flags3 |= GETCFG(machine) == 48 ? 4 : 0;
	header.last0xFFFD = Last0xFFFD;
	for (i = 0; i < sizeof(PSG); i++)
	{
		header.psg[i] = PSG[1];
	}
	header.tstates[0] = 0;
	header.tstates[1] = 0;
	header.tstates[2] = 0;
	header.flags4 = 0;
	header.mgt_paged = 0;
	header.mf128_paged = 0;
	header.ram_at_0x0000 = 0;
	header.ram_at_0x2000 = 0;
	header.joystick_keys[0] = 0x03;
	header.joystick_keys[1] = 0x01;
	header.joystick_keys[2] = 0x03;
	header.joystick_keys[3] = 0x02;
	header.joystick_keys[4] = 0x03;
	header.joystick_keys[5] = 0x04;
	header.joystick_keys[6] = 0x03;
	header.joystick_keys[7] = 0x08;
	header.joystick_keys[8] = 0x03;
	header.joystick_keys[9] = 0x10;
	header.key_names[0] = 0x31;
	header.key_names[1] = 0x00;
	header.key_names[2] = 0x32;
	header.key_names[3] = 0x00;
	header.key_names[4] = 0x33;
	header.key_names[5] = 0x00;
	header.key_names[6] = 0x34;
	header.key_names[7] = 0x00;
	header.key_names[8] = 0x35;
	header.key_names[9] = 0x00;
	header.mgt_type = 0;
	header.disciple_inhibit = 0;
	header.disciple_rom = 0;
#ifdef XZX_PLUS3
	if (GETCFG(machine) == 3)
	{
		header.last0x1FFD = Last0x1FFD;
	}
	else
#endif
	{
		header.last0x1FFD = 0;
	}

	if (fwrite(&header, 1, Z80_V3X_HDRLGT, ofp) != Z80_V3X_HDRLGT)
	{
		Msg(M_PERR, "couldn't write snapshot header");
		(void)fclose(ofp);
		return -1;
	}

	if (GETCFG(machine) == 48)
	{
		if (writeZ80Page(RAM2, 4, ofp) == -1 ||
			writeZ80Page(RAM0, 5, ofp) == -1 ||
			writeZ80Page(RAM5, 8, ofp) == -1)
		{
			(void)fclose(ofp);
			return -1;
		}
	}
	else
	{
		for (i = 0; i < 8; i++)
		{
			if (writeZ80Page(i + RAM0, i + 3, ofp) == -1)
			{
				(void)fclose(ofp);
				return -1;
			}
		}
	}
	if (retofp)
	{
		*retofp = ofp;
	}
	else
	{
		(void)fclose(ofp);
	}
	return 0;
}

static int
readZ80Page(FILE *ifp, int block, int skip)
{
	struct z80_page_header header;
	int blockLen, pageNo, xzxPageNo;
	int addr, count;
	int i, ch;
	if (fread(&header, 1, sizeof(header), ifp) != sizeof(header))
	{
		Msg(M_PERR, "couldn't read page header");
		return -1;
	}
	blockLen = (header.page_length[1] << 8) | header.page_length[0];
	pageNo = header.page_number;
	if (skip)
	{
		if (blockLen == 65535)
		{
			blockLen = XZX_PAGESIZE;
		}
#ifdef DEBUG
		if (GETCFG(debug) & D_SNAP)
		{
			Msg(M_DEBUG, "skip page: Z80-page %2d, block length %5d", pageNo,
				blockLen);
		}
#endif

		return fseek(ifp, (long)blockLen, SEEK_CUR);
	}

	if (GETCFG(machine) == 48)
	{
		switch(pageNo)
		{
			case 4:
				xzxPageNo = RAM2;
				break;
			case 5:
				xzxPageNo = RAM0;
				break;
			case 8:
				xzxPageNo = RAM5;
				break;
			default:
				Msg(M_ERR, "invalid page number %d in 48K snapshot", pageNo);
				return -1;
		}
	}
	else
	{
		xzxPageNo = RAM0 + pageNo - 3;
	}
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		Msg(M_DEBUG, "load page: Z80-page %2d, block length %5d", pageNo,
			blockLen == 65535 ? XZX_PAGESIZE : blockLen);
	}
#endif

	if (blockLen == 65535)
	{
		if (fread(RealMemory[xzxPageNo], 1, XZX_PAGESIZE, ifp) != XZX_PAGESIZE)
		{
			Msg(M_PERR, "couldn't read block %d", block);
			return -1;
		}
	}
	else
	{
		addr = 0;
		for (i = 0; i < blockLen;)
		{
			i++;
			if ((ch = fgetc(ifp)) != 0xed)
			{
				RealMemory[xzxPageNo][addr++] = (uns8)ch;
			}
			else
			{
				i++;
				if ((ch = fgetc(ifp)) != 0xed)
				{
					RealMemory[xzxPageNo][addr++] = 0xed;
					(void)ungetc(ch, ifp);
					i--;
				}
				else
				{
					count = fgetc(ifp);
					i++;
					ch = fgetc(ifp);
					i++;
					while (count--)
					{
						RealMemory[xzxPageNo][addr++] = (uns8)ch;
					}
				}
			}
		}
		if (addr != XZX_PAGESIZE)
		{
			Msg(M_WARN, "block %d contains only %d bytes", block, addr);
		}
	}
	return 0;
}

static int
readZ80VX(FILE *ifp)
{
	struct z80_header header;
	int i;
	static char *hw[] = {
		"48k", "48k+IF1", "SamRam", "48k+MGT",
		"128k", "128k+IF1", "128k+MGT", "+3"
	};
	if (fseek(ifp, 0L, SEEK_SET) == -1
		|| fread(&header, 1, Z80_V3X_HDRLGT, ifp) != Z80_V3X_HDRLGT)
	{
		Msg(M_PERR, "couldn't read snapshot header");
		return -1;
	}
	Msg(M_INFO, "snapshot format is Z80 v3.X");
	Msg(M_INFO, "hardware type is %s", hw[header.hw_mode]);
	LPC = header.lpc1;
	HPC = header.hpc1;
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		if (header.hw_mode == 7)
		{
			Msg(M_DEBUG, "last0x1FFD      : %02x", header.last0x1FFD);
		}
		else
		{
			Msg(M_DEBUG, "mgt_paged       : %d", header.mgt_paged);
			Msg(M_DEBUG, "mf128_paged     : %d", header.mf128_paged);
			Msg(M_DEBUG, "ram_at_0x0000   : %d", header.ram_at_0x0000);
			Msg(M_DEBUG, "ram_at_0x2000   : %d", header.ram_at_0x2000);
			Msg(M_DEBUG, "mgt_type        : %d", header.mgt_type);
			Msg(M_DEBUG, "disciple_inhibit: %d", header.disciple_inhibit);
			Msg(M_DEBUG, "disciple_rom    : %d", header.disciple_rom);
		}
	}
#endif
	if (header.hw_mode == 3 || header.hw_mode == 6)
	{
		Msg(M_WARN, "XZX does not support the M.G.T");
	}
	if (header.hw_mode == 2)
	{
		Msg(M_WARN, "XZX does not support the SamRam");
		(void)SelectModel(48, False);
		for (i = 0; i < 5; i++)
		{
			if (readZ80Page(ifp, i, (i == 2 || i == 3)) == -1)
			{
				return -1;
			}
		}
	}
	else if (header.hw_mode < 4)
	{
		(void)SelectModel(48, False);
		for (i = 0; i < 3; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}
	}
	else
	{
		(void)SelectModel((header.hw_mode != 7 ? 128 : 3), False);
		for (i = 0; i < 8; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}

		OutByte(P_BANK128, header.last0x7FFD);
		if (header.hw_mode == 7)
		{
			OutByte(P_BANK3, header.last0x1FFD);
		}
	}
	if (GETCFG(machine) != 48 || header.flags3 & 4)
	{
		for (i = 0; i < sizeof(PSG); i++)
		{
			OutByte(P_SNDCONTROL, i);
			OutByte(P_SNDDATA, header.psg[i]);
		}
		OutByte(P_SNDCONTROL, header.last0xFFFD);
	}
	if (header.hw_mode == 1 || header.hw_mode == 5)
	{
#ifdef XZX_IF1
		If1OnOff(True);
		if (header.if1_paged)
		{
			PageIn(0, IF1ROM);
		}
#else
		Msg(M_WARN,
			"snapshot uses IF1 but XZX wasn't compiled with IF1 support");
#endif
	}
	return 0;
}

static int
readZ80V3(FILE *ifp)
{
	struct z80_header header;
	int i;
	static char *hw[] = {
		"48k", "48k+IF1", "SamRam", "48k+MGT",
		"128k", "128k+IF1", "128k+MGT"
	};
	if (fseek(ifp, 0L, SEEK_SET) == -1
		|| fread(&header, 1, Z80_V3_HDRLGT, ifp) != Z80_V3_HDRLGT)
	{
		Msg(M_PERR, "couldn't read snapshot header");
		return -1;
	}
	Msg(M_INFO, "snapshot format is Z80 v3.0");
	Msg(M_INFO, "hardware type is %s", hw[header.hw_mode]);
	LPC = header.lpc1;
	HPC = header.hpc1;
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		Msg(M_DEBUG, "mgt_paged       : %d", header.mgt_paged);
		Msg(M_DEBUG, "mf128_paged     : %d", header.mf128_paged);
		Msg(M_DEBUG, "ram_at_0x0000   : %d", header.ram_at_0x0000);
		Msg(M_DEBUG, "ram_at_0x2000   : %d", header.ram_at_0x2000);
		Msg(M_DEBUG, "mgt_type        : %d", header.mgt_type);
		Msg(M_DEBUG, "disciple_inhibit: %d", header.disciple_inhibit);
		Msg(M_DEBUG, "disciple_rom    : %d", header.disciple_rom);
	}
#endif
	if (header.hw_mode == 3 || header.hw_mode == 6)
	{
		Msg(M_WARN, "XZX does not support the M.G.T");
	}
	if (header.hw_mode == 2)
	{
		Msg(M_WARN, "XZX does not support the SamRam");
		(void)SelectModel(48, False);
		for (i = 0; i < 5; i++)
		{
			if (readZ80Page(ifp, i, (i == 2 || i == 3)) == -1)
			{
				return -1;
			}
		}
	}
	else if (header.hw_mode < 4)
	{
		(void)SelectModel(48, False);
		for (i = 0; i < 3; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}
	}
	else
	{
		(void)SelectModel(128, False);
		for (i = 0; i < 8; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}

		OutByte(P_BANK128, header.last0x7FFD);
	}
	if (GETCFG(machine) != 48 || header.flags3 & 4)
	{
		for (i = 0; i < sizeof(PSG); i++)
		{
			OutByte(P_SNDCONTROL, i);
			OutByte(P_SNDDATA, header.psg[i]);
		}
		OutByte(P_SNDCONTROL, header.last0xFFFD);
	}
	if (header.hw_mode == 1 || header.hw_mode == 5)
	{
#ifdef XZX_IF1
		If1OnOff(True);
		if (header.if1_paged)
		{
			PageIn(0, IF1ROM);
		}
#else
		Msg(M_WARN,
			"snapshot uses IF1 but XZX wasn't compiled with IF1 support");
#endif
	}
	return 0;
}

static int
readZ80V2(FILE *ifp)
{
	struct z80_header header;
	int i;
	static char *hw[] = { "48k", "48k+IF1", "SamRam", "128k", "128k+IF1" };
	if (fseek(ifp, 0L, SEEK_SET) == -1
		|| fread(&header, 1, Z80_V2_HDRLGT, ifp) != Z80_V2_HDRLGT)
	{
		Msg(M_PERR, "couldn't read snapshot header");
		return -1;
	}
	Msg(M_INFO, "snapshot format is Z80 v2.0");
	Msg(M_INFO, "hardware type is %s", hw[header.hw_mode]);
	LPC = header.lpc1;
	HPC = header.hpc1;
	if (header.hw_mode == 2)
	{
		Msg(M_WARN, "XZX does not support the SamRam");
		(void)SelectModel(48, False);
		for (i = 0; i < 5; i++)
		{
			if (readZ80Page(ifp, i, (i == 2 || i == 3)) == -1)
			{
				return -1;
			}
		}
	}
	else if (header.hw_mode < 3)
	{
		(void)SelectModel(48, False);
		for (i = 0; i < 3; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}
	}
	else
	{
		(void)SelectModel(128, False);
		for (i = 0; i < 8; i++)
		{
			if (readZ80Page(ifp, i, False) == -1)
			{
				return -1;
			}
		}

		OutByte(P_BANK128, header.last0x7FFD);
	}

	if (GETCFG(machine) != 48 || header.flags3 & 4)
	{
		for (i = 0; i < sizeof(PSG); i++)
		{
			OutByte(P_SNDCONTROL, i);
			OutByte(P_SNDDATA, header.psg[i]);
		}
		OutByte(P_SNDCONTROL, header.last0xFFFD);
	}
	if (header.hw_mode == 1 || header.hw_mode == 4)
	{
#ifdef XZX_IF1
		If1OnOff(True);
		if (header.if1_paged)
		{
			PageIn(0, IF1ROM);
		}
#else
		Msg(M_WARN,
			"snapshot uses IF1 but XZX wasn't compiled with IF1 support");
#endif
	}
	return 0;
}

static int
readZ80Format(char *fname, FILE **retifp)
{
	FILE *ifp;
	struct z80_header header;
	int compressed;
	int ret = 0;
	int addr, count, ch;
	if (!(ifp = Fopen(fname, "rb")))
	{
		Msg(M_PERR, "couldn't open <%s> for reading", fname);
		return -1;
	}
	if (fread(&header, 1, Z80_V1_HDRLGT, ifp) != Z80_V1_HDRLGT)
	{
		Msg(M_PERR, "couldn't read snapshot header");
		(void)fclose(ifp);
		return -1;
	}
	A   = header.a;
	F   = header.f;
	C   = header.c;
	B   = header.b;
	L   = header.l;
	H   = header.h;
	LPC = header.lpc;
	HPC = header.hpc;
	LSP = header.lsp;
	HSP = header.hsp;
	I   = header.i;
	R   = header.r & 0x7f;
	if (header.flags1 == 255)
	{
		header.flags1 = 1;
	}

	R |= (header.flags1 & 0x01) << 7;
	OutByte(P_ULA, (header.flags1 & 0x0e) >> 1);
	compressed = header.flags1 & 0x20;
	E    = header.e;
	D    = header.d;
	C1   = header.c_alt;
	B1   = header.b_alt;
	E1   = header.e_alt;
	D1   = header.d_alt;
	L1   = header.l_alt;
	H1   = header.h_alt;
	A1   = header.a_alt;
	F1   = header.f_alt;
	LIY  = header.liy;
	HIY  = header.hiy;
	LIX  = header.lix;
	HIX  = header.hix;
	IFF1 = (header.iff1 != 0);
	IFF2 = (header.iff2 != 0);

	IM = header.flags2 & 0x03;
	SETCFG(issue, (header.flags2 & 0x04 ? 2 : 3));
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		static char *joy[] = {
			"Cursor", "Kempston", "Sinclair 1", "Sinclair 2"
		};
		Msg(M_DEBUG, "joystick type is %s", joy[(header.flags2 & 0xc0) >> 6]);
		Msg(M_DEBUG, "issue type is %d", GETCFG(issue));
	}
#endif
#ifdef XZX_IF1
	If1OnOff(False);
#endif
	if (PC)
	{
		Msg(M_INFO, "snapshot format is Z80 V1.0");
		Msg(M_INFO, "hardware type is 48k");

		(void)SelectModel(48, False);
		if (compressed)
		{
			addr = 16384;
			while ((ch = fgetc(ifp)) != EOF)
			{
				if (ch != 0xed)
				{
					WR_BYTE(addr, ch);
					addr++;
				}
				else
				{
					if ((ch = fgetc(ifp)) != 0xed)
					{
						WR_BYTE(addr, 0xed);
						addr++;
						(void)ungetc(ch, ifp);
					}
					else
					{
						count = fgetc(ifp);
						ch = fgetc(ifp);
						while (count)
						{
							WR_BYTE(addr, ch);
							addr++;
							count--;
						}
					}
				}
			}
			if (addr < 65535)
			{
				Msg(M_ERR, "premature end on <%s>", fname);
				(void)fclose(ifp);
				return -1;
			}
		}
		else
		{
			for (addr = 16384; addr < 65536; addr++)
			{
				if ((ch = fgetc(ifp)) == EOF)
				{
					Msg(M_ERR, "premature end on <%s>", fname);
					(void)fclose(ifp);
					return -1;
				}
				WR_BYTE(addr, ch);
			}
		}
	}
	else
	{
		switch(fgetc(ifp) | (fgetc(ifp) << 8))
		{
			case Z80_V2_HDRLGT - Z80_V1_HDRLGT - 2:
				ret = readZ80V2(ifp);
				break;
			case Z80_V3_HDRLGT - Z80_V1_HDRLGT - 2:
				ret = readZ80V3(ifp);
				break;
			case Z80_V3X_HDRLGT - Z80_V1_HDRLGT - 2:
				ret = readZ80VX(ifp);
				break;
			default:
				Msg(M_WARN, "unknown Z80 snapshot version");
				ret = -1;
		}
	}
	if (!ret && retifp)
	{
		*retifp = ifp;
	}
	else
	{
		(void)fclose(ifp);
	}
	return ret;
}

static int
readSltFormat(char *fname)
{

	uns8 tblock[sizeof(struct slt_block)];
	int blcount, blnum;
	if (sltFile)
	{

		(void)fclose(sltFile);
	}

	if (readZ80Format(fname, &sltFile) == -1 ||	!sltFile)
	{
		Msg(M_ERR, "SLT file does not contain a valid Z80 snapshot");
		(void)fclose(sltFile);
		return -1;
	}

	if (fseek(sltFile, 3L, SEEK_CUR) == -1
		|| fgetc(sltFile) != 'S'
		|| fgetc(sltFile) != 'L'
		|| fgetc(sltFile) != 'T')
	{
		Msg(M_ERR, "couldn't find SLT signature");
		(void)fclose(sltFile);
		return -1;
	}
#ifdef NO_FPOS_T
	if ((sltPos = ftell(sltFile)) == -1)
#else
	if (fgetpos(sltFile, &sltPos))
#endif
	{
		Msg(M_ERR, "premature end on SLT <%s>", fname);
		(void)fclose(sltFile);
		return -1;
	}

	blkPos = sltPos;

	blcount = 0;
	do
	{

		if (!fread(&tblock, sizeof(tblock), 1, sltFile))
		{
			Msg(M_ERR, "premature end on SLT <%s>", fname);
			(void)fclose(sltFile);
			return -1;
		}
		blcount++;
	} while (tblock[0] || tblock[1]);

#ifdef NO_FPOS_T
	if (fseek(sltFile, sltPos, SEEK_SET) == -1)
#else
	if (fsetpos(sltFile, &sltPos))
#endif
	{
		Msg(M_ERR, "file error on SLT <%s>", fname);
		(void)fclose(sltFile);
		return -1;
	}

	if (sltBlocks)
	{
		free(sltBlocks);
	}
	sltBlocks = (struct slt_block*)Malloc(blcount * sizeof(struct slt_block),
										  "readSLTFormat");

	for (blnum = 0; blnum < blcount; blnum++)
	{
		if (!fread(tblock, sizeof(tblock), 1, sltFile))
		{
			Msg(M_ERR, "premature end on SLT <%s>", fname);
			(void)fclose(sltFile);
			return -1;
		}

		sltBlocks[blnum].type = tblock[0] + (tblock[1] << 8);
		sltBlocks[blnum].identifier = tblock[2] + (tblock[3] << 8);
		sltBlocks[blnum].length = tblock[4] + (tblock[5] << 8) +
								  (tblock[6] << 16) + (tblock[7] << 24);
#ifdef DEBUG
		if (GETCFG(debug) & D_SNAP)
		{
			Msg(M_DEBUG, "block %2d, type %1d, id %2d, length %5d",
				blnum,
				sltBlocks[blnum].type,
				sltBlocks[blnum].identifier,
				sltBlocks[blnum].length);
		}
#endif
	}

#ifdef NO_FPOS_T
	if ((sltPos = ftell(sltFile)) == -1)
#else
	if (fgetpos(sltFile, &sltPos))
#endif
	{
		Msg(M_PERR, "file error in SLT <%s>", fname);
		(void)fclose(sltFile);
		return -1;
	}

	InSlt = 1;
	return 0;
}

static int
readSltBlock(void)
{
	int which;
	long disp;
	int addr, remain, count, ch;
	assert(InSlt && sltFile && sltBlocks);
	if (READ_ONLY(HL))
	{
		Msg(M_WARN, "attempt to read level into ROM - ignored");
		return -1;
	}

#ifdef NO_FPOS_T
	if (fseek(sltFile, sltPos, SEEK_SET) == -1)
#else
	if (fsetpos(sltFile, &sltPos))
#endif
	{
		Msg(M_PERR, "couldn't load level %d", A);
		return -1;
	}

	for (which = 0, disp = 0; sltBlocks[which].type; which++)
	{
		if (sltBlocks[which].identifier == A)
		{
			break;
		}
		disp += sltBlocks[which].length;
	}

	if (sltBlocks[which].identifier != A)
	{
		Msg(M_WARN, "requested level %d doesn't exist", A);
		return -1;
	}

	if (fseek(sltFile, disp, SEEK_CUR) == -1)
	{
		Msg(M_PERR, "couldn't load level %d", A);
		return -1;
	}

	addr = HL;
	remain = (int)sltBlocks[which].length;
	while ((ch = fgetc(sltFile)) != EOF && remain)
	{
		remain--;
		if (ch != 0xed)
		{
			WR_BYTE(addr, ch);
			addr++;
		}
		else
		{
			if ((ch = fgetc(sltFile)) != 0xed)
			{
				WR_BYTE(addr, 0xed);
				addr++;
				(void)ungetc(ch, sltFile);
			}
			else
			{
				count = fgetc(sltFile);
				ch = fgetc(sltFile);
				remain -= 3;
				while (count)
				{
					WR_BYTE(addr, ch);
					addr++;
					count--;
				}
			}
		}
	}
	if (remain)
	{
		Msg(M_PWARN, "error while loading level %d", A);
		return -1;
	}
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		Msg(M_DEBUG, "reading level %2d of length %5d",
			sltBlocks[which].identifier, sltBlocks[which].length);
	}
#endif
	return 0;
}

int
WriteSnapshot(char *fname)
{
	int type, byte, ret;
	FILE *ofp;
	type = GetFileType(fname);

	if (GETCFG(machine) != 48 && type != FT_SLT)
	{
		type = FT_Z80;
	}
	switch(type)
	{
		default:
		case FT_UNKNOWN:
		case FT_SNA:
			ret = writeSnaFormat(fname);
			break;
		case FT_Z80:
			ret = writeZ80Format(fname, NULL);
			break;
		case FT_SLT:
			if (!InSlt)
			{
				Msg(M_WARN,
					"attempt to write SLT snapshots in non-SLT mode - ignored");
				return -1;
			}

			if (!strcmp(LastSnapshotName, fname))
			{
				Msg(M_WARN, "attempt to overwrite current SLT - ignored");
				return -1;
			}

			if ((ret = writeZ80Format(fname, &ofp)) != -1)
			{

				(void)putc(0, ofp);
				(void)putc(0, ofp);
				(void)putc(0, ofp);
				(void)putc('S', ofp);
				(void)putc('L', ofp);
				(void)putc('T', ofp);

#ifdef NO_FPOS_T
				if (fseek(sltFile, blkPos, SEEK_SET) == -1)
#else
				if (fsetpos(sltFile, &blkPos))
#endif
				{
					Msg(M_PWARN, "couldn't write SLT levels");
					ret = -1;
				}
				else
				{

					while ((byte = fgetc(sltFile)) != EOF)
					{
						if (putc(byte, ofp) == -1)
						{
							ret = -1;
							break;
						}
					}
				}
				(void)fclose(ofp);
			}
			break;
	}
	return ret;
}

int
ReadSnapshot(char *fname)
{
	int ret = -1;
#ifdef DEBUG
	if (GETCFG(debug) & D_SNAP)
	{
		Msg(M_DEBUG, "last snap was <%s>", LastSnapshotName);
		Msg(M_DEBUG, "now loading <%s>", fname);
	}
#endif
	Z80_Reset();
	(void)strcpy(LastSnapshotName, fname);
	switch(GetFileType(fname))
	{
		case FT_UNKNOWN:
		case FT_SNA:
			ret = readSnaFormat(fname);
			break;
		case FT_Z80:
			ret = readZ80Format(fname, NULL);
			break;
		case FT_SLT:
			ret = readSltFormat(fname);
			break;
	}

	ForceScreenRefresh();
	return ret;
}

void
LevelLoaderTrap(void)
{
	FILE *in;
	char *ptr;
	char name[MAXPATHLEN];
	int i, ch;

	if (!LastSnapshotName[0])
	{
		Msg(M_WARN, "no previous snapshot name - multiload ignored");
		return;
	}
	if (InSlt)
	{
		(void)readSltBlock();
	}
	else
	{

		if (READ_ONLY(HL))
		{
			Msg(M_WARN,
				"attempt to read multiload overlay into ROM - ignored");
			return;
		}

		(void)strcpy(name, LastSnapshotName);
		if (!(ptr = strrchr(name, '.')))
		{
			ptr = name + strlen(name);
		}

		(void)sprintf(ptr, "%d.dat", A);
		if (!(in = Fopen(name,"rb")))
		{

			(void)sprintf(ptr, "%d.DAT", A);
			if (!(in = Fopen(name,"rb")))
			{

				if (!(ptr = FileSelector("Load Level", True, "dat", NULL, NULL))
					|| !(in = Fopen(name,"rb")))
				{
					Msg(M_PWARN, "couldn't find level <%s>", name);

					CCF();
					return;
				}
			}
		}
#ifdef DEBUG
		if (GETCFG(debug) & D_SNAP)
		{
			Msg(M_DEBUG,
				"reading overlay <%s> at address <%04x>", name, HL);
		}
#endif

		for (i = HL; i < 65536; i++)
		{
			if ((ch = fgetc(in)) == EOF)
			{
				break;
			}
			WR_BYTE(i, ch);
		}
		(void)fclose(in);
	}
}

