 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                                                                      */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Mon Mar 10 1997                              */
 /* -------------------------------------------------------------------- */
 /*                                                                      */
 /* The common interface for all unit databases is defined here.         */
 /*                                                                      */
 /************************************************************************/

#include "UnitDatabase.h"

EST_String  SubtypedObject<UnitDatabase>::type_name(void) const { return "UnitDatabase";}

#if defined(INCLUDE_LISP)
int  SubtypedObject<UnitDatabase>::s_lisp_type = tc_festival_unitdatabase;
#endif

UnitDatabase::UnitDatabase(void) : p_unit_cache(UnitCacheHashtableSize)
{
  cwarn << "create UnitDatabase\n";
  p_name = ""; p_description = "";
  p_copyright = ""; p_filename = ""; 
  p_unit_type = "";

  p_phoneset = NULL; p_catalogue = NULL; 
  p_num_indexes=0; p_indexes = NULL;
}

static void free_cached_unit(EST_String &s, Unit *&u)
{
  (void)s;
  delete u;
}

UnitDatabase::~UnitDatabase(void)
{ 
  cwarn << "destroy UnitDatabase\n";
  if (p_indexes)
    delete[] p_indexes;

  p_unit_cache.map(free_cached_unit);
}


EST_read_status UnitDatabase::fill_from(FILE *stream) 
{(void)stream; return misc_read_error;}
EST_write_status UnitDatabase::write_to(FILE *stream) 
{(void)stream; return misc_write_error;}
void UnitDatabase::fill_from(UnitDatabase &source)
{(void)source;  }

#if defined(DONT_INLINE_EVERYTHING)
void UnitDatabase::set_description(EST_String description) {p_description=description;};
void UnitDatabase::set_copyright(EST_String copyright) {p_copyright = copyright;};
void UnitDatabase::set_filename(EST_String filename) {p_filename = filename;};
void UnitDatabase::set_unit_type(EST_String unit_type) {p_unit_type = unit_type;};
void UnitDatabase::set_catalogue(UnitCatalogue *catalogue) {p_catalogue = catalogue;};
void UnitDatabase::set_phoneset(PhoneSet *phones) {p_phoneset = phones;};
#endif


void UnitDatabase::property_names(EST_TList<EST_String> &list) const
{
  SubtypedObject<UnitDatabase>::property_names(list);
  list.append("description");
  list.append("copyright");
  list.append("unit_type");
  list.append("phoneset");

  EST_Litem *item;

  for(item=p_properties.head(); item; item=next(item))
    list.append(p_properties.key(item));
}

ValueType UnitDatabase::property(EST_String property) const
{

  if (property == "description")
    return string_as_value(p_description);
  else if (property == "copyright")
    return string_as_value(p_copyright);
  else if (property == "unit_type")
    return string_as_value(p_unit_type);
  else if (property == "phoneset")
    {
      const EST_String name = p_phoneset->phone_set_name();
      return string_as_value(name);
    }
  else if (p_properties.present(property))
    return string_as_value(p_properties.val(property));
  else
    return SubtypedObject<UnitDatabase>::property(property);
}

int UnitDatabase::set_property(EST_String property, ValueType value)
{
  if (property == "description")
    set_description(value_as_c_string(value));
  else if (property == "copyright")
    set_copyright(value_as_c_string(value));
  else if (property == "unit_type")
    {
      if (!Unit::legal_type(value_as_c_string(value)))
	return 0;
      set_unit_type(value_as_c_string(value));
    }
  else if (property == "phoneset")
    {
      PhoneSet *phones = phoneset_name_to_set(value_as_string(value));

      if (!phones)
	return 0;
      set_phoneset(phones);
    }
  else if(!SchemeObject::set_property(property, value))
      p_properties.add_item(property, value_as_c_string(value));

  return 1;
}

void UnitDatabase::add_index(EST_String name, EST_String type, UnitIndex *index)
{
  int i;

  for(i=0; i<p_num_indexes; i++)
    if (!p_indexes[i].index)
      break;

  cerr << "add index "<< i << "\n";

  if (i >= p_num_indexes)
    {
      UnitDatabase::index_information_s *new_info = new UnitDatabase::index_information_s[p_num_indexes+5];
      for (int j=0; j< p_num_indexes; j++)
	new_info[j] = p_indexes[j];
      
      for (int jj=p_num_indexes; jj< p_num_indexes+5; jj++)
	new_info[jj].index = NULL;

      if (p_indexes)
	delete[] p_indexes;

      p_indexes = new_info;
      p_num_indexes +=5;
    }

  p_indexes[i].name = name;
  p_indexes[i].type = type;
  p_indexes[i].index = index;
}

UnitDatabase::index_information_s *UnitDatabase::index_information(EST_String type)
{
  int i;
  for(i=0; i< p_num_indexes; i++)
    if (!p_indexes[i].index)
      break;
    else if (p_indexes[i].type == type)
      return p_indexes + i;

  return NULL;
}

const UnitIndex *UnitDatabase::index(const EST_String name)
{
  UnitDatabase::index_information_s *info = index_information(name);

  return info?info->index:(const UnitIndex *)NULL;
}

EST_StrVector UnitDatabase::index_names(void) const
{ 
  EST_StrVector names(p_num_indexes);
  int i;

  for(i=0; i< p_num_indexes; i++)
    if (p_indexes[i].index)
      names(i) = p_indexes[i].type;
  
return names;
}

UnitDatabase::ContentType UnitDatabase::contents_name_to_number(EST_String name)
{
  UnitDatabase::ContentType type = UnitDatabase::ContentTypeMap.token(name);

  int start_of_bracket[EST_Regex_max_subexpressions];
  int end_of_bracket[EST_Regex_max_subexpressions];
  static EST_Regex RxContent("\\(wave\\|coef\\|coefficients\\):\\([0-9]+\\)");
  
  if (type ==   UnitDatabase::ct_none && name.matches(RxContent, 0, start_of_bracket, end_of_bracket))
    {
      if (name[start_of_bracket[1]] == 'w')
	type =  UnitDatabase::ct_waveform(atoi(name.at(start_of_bracket[2], end_of_bracket[2]-start_of_bracket[2])));
      else
	type =  UnitDatabase::ct_coefficient(atoi(name.at(start_of_bracket[2], end_of_bracket[2]-start_of_bracket[2])));
    }

  return type;
}

const Unit *UnitDatabase::unit(const EST_String name)
{
  Unit *it = p_unit_cache.val(name);

  if (!it)
    {
      it = Unit::create(p_unit_type);

      it->associate_database(this);
      it->set_name(name);

      p_unit_cache.add_item(name, it);
    }

  return it;
}

void UnitDatabase::print_description(FILE *stream)
{
  fprintf(stream, "Name: %s\n", (const char *)name());
  fprintf(stream, "Description: %s\n", (const char *)description());
  fprintf(stream, "Copyright: %s\n", (const char *)copyright());
  fprintf(stream, "Unit Format: %s\n", (const char *)unit_type());
  if (p_phoneset)
    fprintf(stream, "Phoneme Set: %s\n", (const char *)p_phoneset->phone_set_name());
  if (p_catalogue)
    fprintf(stream, "Unit Count: %d\n", p_catalogue->num_entries());
  UnitDatabase::index_information_s *ip = p_indexes;
  if (ip)
    for(ip=p_indexes; ip->name != ""; ip++)
      fprintf(stream, "Index: %s %s\n", (const char *)ip->type, (const char *)ip->name);
}

#if !defined(UNITDATABASE_ABSTRACT)

UnitCatalogue::Itterator::~Itterator(void)
{ }

UnitName UnitCatalogue::Itterator::next(void)
{ return (const char *)NULL; }

int UnitCatalogue::Itterator::at_end(void) 
{ return 1; }
int UnitCatalogue::num_entries(void)
{ return 0; }
int UnitCatalogue::has_entry(EST_String name)
{ (void)name; return 0; }
EST_StrVector UnitDatabase::wave_names(void) const
{ return EST_StrVector(0); }
EST_StrVector UnitDatabase::coefficient_names(void) const
{ return EST_StrVector(0); }

UnitDatabase::wave_information_s *UnitDatabase::wave_information(ContentType which)
{
  (void)which;
  return NULL;
}

UnitDatabase::coefficient_information_s *UnitDatabase::coefficient_information(ContentType which)
{
  (void)which;
  return NULL;
}

EST_Wave *UnitDatabase::get_wave(UnitDatabase::ContentType content,
				 Unit &unit)
{ 
  (void)content; (void)unit; 
  cwarn << "trying to get wave from untyped database\n";
  return NULL; 
}

EST_Track *UnitDatabase::get_coefficients(UnitDatabase::ContentType content, 
					  EST_ContourType format,
					  Unit &unit)
{ 
  (void)content; (void)format; (void)unit; 
  cwarn << "trying to get coefficients from untyped database\n";
  return NULL; 
}


EST_TVector<float> *UnitDatabase::get_segments(Unit &unit)
{
  (void)unit; 
  cwarn << "trying to get coefficientssegments from untyped database\n";
  return NULL; 
}

#endif // UNITDATABASE_ABSTRACT

#if defined(INCLUDE_LISP)

LISP UnitDatabase::lisp_print_description(LISP ldatabase, LISP file)
{
  FILE *stream=get_c_file(file,stdout);
  if (ldatabase == NIL || TYPE(ldatabase) != tc_festival_unitdatabase)
    {
      fputs("[not a database]\n", stream);
      return NIL;
    }

  UnitDatabase *unitdatabase = (UnitDatabase *)PTRVAL(ldatabase);

  unitdatabase->print_description(stream);
  return ldatabase;
}

void UnitDatabase::lisp_declare(void)
{
  long gc_kind;
  set_gc_hooks(tc_festival_unitdatabase,
	       1,
	       NULL,
	       SchemeObject::lisp_gc_mark,
	       NULL,
	       SchemeObject::lisp_gc_free,
	       SchemeObject::lisp_gc_clear_mark,
	       &gc_kind);
  set_print_hooks(tc_festival_unitdatabase,
		  UnitDatabase::lisp_print,
		  UnitDatabase::lisp_print_string
		  );

  init_subr_2("make_unitdatabase", UnitDatabase::lisp_make,
 "(make_unitdatabase TYPE PROPERTIES)\n\
  Make an empty unitdatabase of the given type and properties.");

  init_subr_1("open_unitdatabase", UnitDatabase::lisp_open,
 "(open_unitdatabase FILENAME)\n\
  Return a database object which accesses the named unit database.");

  init_subr_2("describe_unitdatabase", UnitDatabase::lisp_print_description,
 "(describe_unitdatabase DATABASE FILE)\n\
  Print a description of the database.");
}
#endif // INCLUDE_LISP

static EST_TValuedEnumDefinition<UnitDatabase::AccessStrategy, const char *,NO_INFO> AccessStrategyNames[] = 
    {
      { UnitDatabase::as_direct,	{"direct" }},
      { UnitDatabase::as_dynamic,	{"dynamic" }},
      { UnitDatabase::as_ondemand,	{"on-demand", "ondemand"}},
      { UnitDatabase::as_direct,	{NULL }}
    };
EST_TNamedEnum<UnitDatabase::AccessStrategy> UnitDatabase::AccessStrategyMap(AccessStrategyNames);

static EST_TValuedEnumDefinition<UnitDatabase::ContentType, const char *,NO_INFO> ContentTypeNames[] = 
    {
      { UnitDatabase::ct_none,		{"nothing"}},
      { UnitDatabase::ct_signal,	{"original signal", "signal" }},
      { UnitDatabase::ct_residual,	{"residual" }},
      { UnitDatabase::ct_pitchmarks,	{"pitchmarks"}},
      { UnitDatabase::ct_analysis,	{"analysis of waveform", "analysis"}},
      { UnitDatabase::ct_experimentalwave1, {"experimentalwave1"}},
      { UnitDatabase::ct_experimentalwave2, {"experimentalwave2"}},
      { UnitDatabase::ct_experimentalcoef1, {"experimentalcoef1"}},
      { UnitDatabase::ct_experimentalcoef2, {"experimentalcoef2"}},
      { UnitDatabase::ct_none,	{NULL }}
    };
EST_TNamedEnum<UnitDatabase::ContentType> UnitDatabase::ContentTypeMap(ContentTypeNames);

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TNamedEnum.cc"
template class EST_TNamedEnum<UnitDatabase::SignalPreprocessing>;
template class EST_TNamedEnumI<UnitDatabase::SignalPreprocessing,char>;
template class EST_TValuedEnum<UnitDatabase::SignalPreprocessing, const char *>;
template class EST_TValuedEnumI<UnitDatabase::SignalPreprocessing, const char *,char>;
template class EST_TNamedEnum<UnitDatabase::ContentType>;
template class EST_TNamedEnumI<UnitDatabase::ContentType,char>;
template class EST_TValuedEnum<UnitDatabase::ContentType, const char *>;
template class EST_TValuedEnumI<UnitDatabase::ContentType, const char *,char>;
template class EST_TNamedEnum<UnitDatabase::AccessStrategy>;
template class EST_TNamedEnumI<UnitDatabase::AccessStrategy,char>;
template class EST_TValuedEnumI<UnitDatabase::AccessStrategy, const char *,char>;
template class EST_TValuedEnum<UnitDatabase::AccessStrategy, const char *>;

#include "SubtypedObject.cc"
template class SubtypedObject<UnitDatabase>;

#include "../base_class/EST_THash.cc"
template class EST_TStringHash<Unit *>;
template class EST_THash<EST_String,Unit *>;
template class EST_Hash_Pair<EST_String,Unit *>;

#endif
Unit *EST_THash<EST_String,Unit *>::Dummy_Value = NULL;
