/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1995,1996                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  May 1994                                  */
/*-----------------------------------------------------------------------*/
/*                  EST_Track Class header file                          */
/*                                                                       */
/*=======================================================================*/

class EST_Track;

#ifndef __Track_H__
#define __Track_H__

#include "EST_String.h"
#include "EST_FMatrix.h"
#include "EST_types.h"
#include "EST_ChannelType.h"
#include "EST_ContourType.h"
#include "EST_TrackMap.h"

#define TRACK_DEFAULT_SR 16000 // occasionally needed for xmg files
#define TRACK_DEF_FS 5.0 // default frame spacing in ms.

typedef enum EST_InterpType {
  it_nearest,			// nearest time point
  it_linear,			// linerar interpolation
  it_linear_nz			// .. unless one end near zero
} EST_InterpType;

class EST_Track {
 private:
    
    EST_FMatrix pa;			// float x array 
    EST_FVector pt;			// float y array 
    EST_IVector pv;			// for breaks and non-breaks
    int p_num_frames;		// size of contour
    int p_num_channels;		// "width" of contour
    EST_TrackMap *p_map;
    
    void default_vals();
    void default_field_names();
    void clear_arrays();
    void pad_breaks();		     // put in extra breaks 
    
    int interp_value(float x, float f);
    float interp_amp(float x, int c, float f);
    float estimate_shift(float x);
    void copy(const EST_Track& a);

    EST_String p_name;              // name of whole thing
    EST_StrVector p_field_names;      // name of each track

    EST_String p_space_type;     // Equal or variable spacing between points 
    EST_String p_file_type;	// type of input file
    EST_String p_break_type;	// single break value or lots between data 
    EST_ContourType p_contour_type;	// eg F0, "default" etc.
    
public:
    EST_Track();
    EST_Track(const EST_Track &a);
    EST_Track(int num_frames, int num_channels);

    void copy_setup(const EST_Track& a); // copy everything but data
    ~EST_Track();

    EST_String name() const {return p_name;}
    void set_name(EST_String n) {p_name = n;}

  // TrackMap manipulation.

  void assign_map(EST_TrackMap *map);
  void assign_map(EST_TrackMap &map) { assign_map(&map); };

  void create_map(EST_ChannelNameMap &names);
  void create_map(void) { create_map(EST_default_channel_names); };

  int channel_position(EST_ChannelType type, int offset=0) const
    { return p_map?((*p_map)(type) + offset):-1; };

  bool has_channel(EST_ChannelType type) const 
    { return (bool)(p_map && p_map->get(type) != NO_SUCH_CHANNEL); };
  bool has_channel(const char *name) const;
  bool has_channel(EST_String name) const 
    { return has_channel((const char *)name); };

  // Basic value access

  // return amplitude of point i, channel c
  float &a(int i, int c=0);
  float  a(int i, int c=0) const 
    { return ((EST_Track *)this)->a(i,c); };

  // return amplitude of point i, channel type c (plus offset)
  float &a(int i, EST_ChannelType c, int offset=0);
  float  a(int i, EST_ChannelType c, int offset=0) const
    { return ((EST_Track *)this)->a(i,c, offset); };

  // return amplitude of point i, channel named name (plus offset)
  float &a(int i, const char *name, int offset=0);
  float  a(int i, const char *name, int offset=0) const
    { return ((EST_Track *)this)->a(i, name, offset); };
  float &a(int i, EST_String name, int offset=0) 
    { return a(i, (const char *)name, offset); };
  float  a(int i, EST_String name, int offset=0) const
    { return ((EST_Track *)this)->a(i, (const char *)name, offset); };

  // return amplitude at time t, channel c
  float &a(float t, int c=0, EST_InterpType interp=it_nearest);	
  float  a(float t, int c=0, EST_InterpType interp=it_nearest) const
    { return ((EST_Track *)this)->a(t, c, interp); };	

  // return amplitude at time t, channel type c
  float &a(float t, EST_ChannelType c, EST_InterpType interp=it_nearest);
  float  a(float t, EST_ChannelType c, EST_InterpType interp=it_nearest) const
    { return ((EST_Track *)this)->a(t, c, interp); };

  // return amplitude at point i
  float &operator() (int i, int c)       { return a(i,c); };	
  float &operator() (int i)              { return a(i,0); };
  float  operator() (int i, int c) const { return a(i,c); };	
  float  operator() (int i) const        { return a(i,0); };	

  float &operator() (int i, EST_ChannelType c)       { return a(i,c); };
  float  operator() (int i, EST_ChannelType c) const { return a(i,c); };
  
  // return amplitude at time t
  float &operator() (float t, int c)       {return a(t,c); };	
  float &operator() (float t)              {return a(t,0);};
  float  operator() (float t, int c) const {return a(t,c); };	
  float  operator() (float t) const        {return a(t,0);};

  // return time position of point i
  float &t(int i);      
  float  t(int i) const                    { return t(i); };      

  // return position at i in milli-seconds.
  float ms_t(int i) const;    

    EST_read_status load(const EST_String name, float ishift = 0.0);
    EST_write_status save(const EST_String name, const EST_String EST_filetype = "");

    EST_read_status load_channel_names(const EST_String name);
    EST_write_status save_channel_names(const EST_String name);

    float amax;			// max amplitude
    float amin;			// min amplitude
    EST_String color;		// Color of output track

    void clear() {;}
    int empty() const;
    
    void set_break(int i);	// set point i to be a break
    void set_value(int i);	// set point i to be a value
    int val(int i) const;	// is point i a value?
    int track_break(int i) const;	// is point i a break?
    int prev_non_break(int i) const;
    int next_non_break(int i) const;
    int index(float x) const; // return index of time value x in the time array
    int index_below(float x) const; // return index below time value x in the time array
    
    int num_frames() const {return p_num_frames;}
    int num_channels() const {return p_num_channels;}
    // REORG confirm this is reasonable
    void set_num_channels(int n) { p_num_channels = n;}
    void set_num_frames(int n);

    const EST_String field_name(int channel, const EST_ChannelNameMap &map, int strings_override=1) const;
    const EST_String field_name(int channel, int strings_override=1) const 
		{ return field_name(channel, EST_default_channel_names, strings_override); };

    void set_field_name(const EST_String &name, int channel);
    
    void fill_time(float t, int start =1);// Fill time axis at t intervals
    void fill_amp(float f); // Fill all amplitudes values with f
    void sample(float shift);    // sample contour at "shift" spacing

    void change_type(float nshift, const EST_String &break_type);

    // take one of the channels as the timeline
    void channel_to_time(int channel, float scale=1.0);
    void channel_to_time(EST_ChannelType c,float  scale=1.0);
    void channel_to_time(const EST_String c_name, float scale=1.0);
    
    void channel_to_time_lengths(int channel, float scale=1.0);
    void channel_to_time_lengths(EST_ChannelType c,float  scale=1.0);
    void channel_to_time_lengths(const EST_String c_name, float scale=1.0);
    
    float shift() const;	//return frame spacing.
    float start() const;	//return time of first meaningful point.
    float end() const;		//return time value of last meaningful point.
    
    EST_Track& operator = (const EST_Track& a);
    EST_Track& operator+=(const EST_Track &a); // add to existing track
    EST_Track& operator|=(const EST_Track &a); // add to existing track in parallel
    friend ostream& operator << (ostream& s, const EST_Track &tr);

    void add_trailing_breaks();
    void rm_trailing_breaks();
    void rm_excess_breaks();	     // reduce to a single break

    void resize(int num_frames, int num_channels);

    EST_String space_type() {return p_space_type;}
    EST_String file_type() {return p_file_type;}
    EST_String break_type() {return p_break_type;}
    EST_ContourType contour_type() {return p_contour_type;}
    void set_space_type(const EST_String &t) {p_space_type = t;}
    void set_file_type(const EST_String &t) {p_file_type = t;}
    void set_break_type(const EST_String &t) {p_break_type = t;}
    void set_contour_type(const EST_ContourType &t) {p_contour_type = t;}
};

// list of tracks in serial
typedef EST_TList<EST_Track> EST_TrackList;


#endif /* __Track_H__ */
