/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1994                                 */
/*-----------------------------------------------------------------------*/
/*             Pitch Detection Algorithm Main routine                    */
/*                                                                       */
/*=======================================================================*/

#include "EST.h"
#include "EST_pda.h"
#include "EST_ContourType.h"
#include "EST_cmd_line_options.h"
#include "fstream.h"

void default_options(EST_Option &al);
int save_pm(EST_String filename, EST_Track fz);

EST_String options_pda_general(void);
EST_String options_pda_srpd(void);

int main (int argc, char *argv[])
{
    EST_Track fz;
    EST_Wave sig;
    EST_Option al, op;
    EST_String out_file("-");
    EST_StrList files;

    parse_command_line2(argc, argv, 
       EST_String("Usage:   ")+
       "pda  <input file> -o <output file> <options>\n"+
       "Summary: pitch track waveform files\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_wave_input()+ "\n"+
       options_pda_general()+ "\n"+
       options_pda_srpd()+ "\n"+
       "-color <string> Color of output line\n" +
       "-o <ofile>       Output filename, if not specified output is\n"+
       "                 to stdout\n"+
       "-otype <string>  Output file type, (optional).  If no type is\n"+
       "                 Specified the type of the input file is assumed\n"+
       "                 types are: "+options_track_filetypes()+"\n",
			files, al);

    default_pda_options(op);
    init_lib_ops(al, op);

    if (read_wave(sig, files.first(), al) == -1)
	exit(-1);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    fz = pda(sig, op);		// do f0 tracking

    fz.color = op.val("color", 0);
    if (al.present("-pm"))
	save_pm(out_file, fz);
    else
	fz.save(out_file, op.val("f0_file_type", 0));

    if (al.present("-sf")) // "simple f0 file", no esps fancy stuff
	fz.set_contour_type(EST_ContourType::SIMPLE_F0);

    if (al.present("-diff"))
    {
	fz = differentiate(fz);
	fz.set_contour_type(EST_ContourType::DIFF_F0);
	fz.save(out_file + ".diff", op.val("f0_file_type", 0));
    }
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    // general options
    a_list.override_val("sample_rate", al.val("-f", 0));
    a_list.override_val("min_pitch",  al.val("-fmin", 0));
    a_list.override_val("max_pitch",  al.val("-fmax", 0));
    a_list.override_val("pda_frame_shift", al.val("-s", 0));
    a_list.override_val("pda_frame_length",al.val("-l", 0));
    
    // low pass filtering options.
    a_list.override_val("lpf_cutoff",al.val("-u", 0));
    a_list.override_val("lpf_order",al.val("-forder", 0));
    
    //sprd options
    a_list.override_val("decimation", al.val("-d", 0));
    a_list.override_val("noise_floor",   al.val("-n", 0));
    a_list.override_val("min_v2uv_coef_thresh", al.val("-m", 0));
    a_list.override_val("v2uv_coef_thresh_ratio", al.val("-r", 0));
    a_list.override_val("v2uv_coef_thresh", al.val("-H", 0));
    a_list.override_val("anti_doubling_thresh", al.val("-t", 0));
    a_list.override_val("peak_tracking", al.val("-P", 0));
    a_list.override_val("join_track_points", al.val("-j", 0));
    
    if (al.val("-L", 0) == "true")
	a_list.override_val("do_low_pass", "true");
    if (al.val("-R", 0) == "true")
	a_list.override_val("do_low_pass", "false");
    a_list.override_val("color", al.val("-color", 0));    
    a_list.override_val("f0_file_type", al.val("-otype", 0));
    a_list.override_val("wave_file_type", al.val("-itype", 0));
}

int save_pm(EST_String filename, EST_Track fz)
{
    ostream *outf;
    float position, period;

    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);

    *outf << "XAO1\n\n";	// xmg header identifier.
    *outf << "LineType        bars \n";
    *outf << "LineStyle       solid \n";
    *outf << "LineWidth       0 \n";
    *outf << "Freq 16\n";
    *outf << "Format  Binary \n";
    if (fz.color != "")
	*outf << "LineColor  " << fz.color << endl;
    *outf << char(12) << "\n";	// control L character

    position = 0.0;
    int gap = 0;
    for (int i = 0; i < fz.num_frames(); ++i)
    {
	if (fz.val(i))
	{
	    if (gap)
	    {
		position = fz.t(i);
		gap = 0;
	    }
	    period = 1.0 / fz.a(i);
	    *outf << (position + period) * 1000.0 << endl;
	    position += period;
	}
	else
	    gap = 1;
    }
    
    if (outf != &cout)
	delete outf;

    return 0;
}


