/*
 *  Sarien AGI :: Copyright (C) 1999 Dark Fiber 
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sarien.h"
#include "agi.h"
#include "words.h"
#include "objects.h"
#include "picture.h"
#include "view.h"
#include "logic.h"
#include "sound.h"
#include "gfx.h"
#include "console.h"

UINT16	agi_v2_init (void);
UINT16	agi_v2_deinit (void);
UINT16	agi_v2_detect_game (UINT8 *);
UINT16	agi_v2_load_resource (UINT8, UINT16);
UINT16	agi_v2_unload_resource (UINT8, UINT16);
UINT16	agi_v2_load_dir (AGI_DIR *, UINT8 *);
UINT8	*agi_v2_load_vol_res (AGI_DIR *);

AGI_LOADER agi_v2= {
	2,
	0,
	agi_v2_init,
	agi_v2_deinit,
	agi_v2_detect_game,
	agi_v2_load_resource,
	agi_v2_unload_resource
};


UINT16 agi_v2_detect_game(UINT8 *gn)
{
	UINT16	ec=err_Unk;

	gdir=gn;

	fixpath(NO_GAMEDIR, (UINT8*)LOGDIR);
	if(__file_exists(path)!=0)
		return err_InvalidAGIFile;

	fixpath(NO_GAMEDIR, (UINT8*)PICDIR);
	if(__file_exists(path)!=0)
		return err_InvalidAGIFile;

	fixpath(NO_GAMEDIR, (UINT8*)SNDDIR);
	if(__file_exists(path)!=0)
		return err_InvalidAGIFile;

	fixpath(NO_GAMEDIR, (UINT8*)VIEWDIR);
	if(__file_exists(path)!=0)
		return err_InvalidAGIFile;

	loader->int_version=0x2917;		/* setup for 2.917 */
	ec=v2id_game();

	return err_OK;
}



UINT16 agi_v2_init(void)
{
	UINT16	ec=err_OK;

	/* load directory files */
	ec=agi_v2_load_dir(dir_logic, (UINT8*)LOGDIR);
	if(ec==err_OK)
		ec=agi_v2_load_dir(dir_pic, (UINT8*)PICDIR);
	if(ec==err_OK)
		ec=agi_v2_load_dir(dir_view, (UINT8*)VIEWDIR);
	if(ec==err_OK)
		ec=agi_v2_load_dir(dir_sound, (UINT8*)SNDDIR);

	return ec;
}

UINT16 agi_v2_deinit(void)
{
	UINT16	ec=err_OK;

	/* unload words */
	/*agi_v2_unload_words();*/

	/* unload objects */
	/*agi_v2_unload_objects();*/


	return ec;
}


UINT16	agi_v2_load_dir(AGI_DIR *agid, UINT8 *fname)
{
	FILE	*fp;
	UINT8	*mem;
	UINT16	ec=err_OK;
	UINT32	flen;
	UINT16	count;

	fixpath(NO_GAMEDIR, fname);
	report ("Loading directory: %s\n", path);

	if((fp=fopen((char*)path, "rb"))!=NULL)
	{
		fseek(fp, 0x0L, SEEK_END);
		flen=ftell(fp);
		fseek(fp, 0x0L, SEEK_SET);

		mem=(UINT8*)malloc(flen+32);
		if(mem!=NULL)
		{
			fread(mem, 1, flen, fp);

			/* set all directory resources to gone */
			for(count=0; count<MAX_DIRS; count++)
			{
				agid[count].volume=0xFF;
				agid[count].offset=_EMPTY;
			}

			/* build directory entries */
			for(count=0; count<flen; count+=3)
			{
				agid[count/3].volume=hilo_getbyte(mem+count)>>4;
				agid[count/3].offset=hilo_getpword(mem+count)&_EMPTY;
			}

			free(mem);
		}
		else
			ec=err_NotEnoughMemory;
		fclose(fp);
	}
	else
		ec=err_BadFileOpen;

	return ec;
}


UINT16	agi_v2_load_objects(UINT8 *fname)
{
	return load_objects(fname);
}

UINT16 agi_v2_load_words(UINT8* fname)
{
	return load_words(fname);
}

UINT16 agi_v2_unload_objects(void)
{
	unload_objects();
	return err_OK;
}

UINT16 agi_v2_unload_words(void)
{
	unload_words();
	return err_OK;
}

UINT16	agi_v2_unload_resource(UINT8 restype, UINT16 resnum)
{
	UINT16	ec=err_OK;

	switch(restype)
	{
	case rLOGIC:
		if((dir_logic[resnum].flags&RES_LOADED)==RES_LOADED)
		{
			if(logics[resnum].data!=NULL)
				free(logics[resnum].data);
			logics[resnum].data=NULL;

			if(logics[resnum].texts!=NULL)
				free(logics[resnum].texts);
			logics[resnum].texts=NULL;

			dir_logic[resnum].flags&=~RES_LOADED;	/* strip loaded flag */
		}

		/* if cached, we end up here */
		logics[resnum].sIP=2;
		logics[resnum].cIP=2;
		break;

	case rPICTURE:
	    /* remove visual buffer & priority buffer if they exist */

		if((dir_pic[resnum].flags&RES_LOADED)==RES_LOADED)
		    if((dir_pic[resnum].flags&0x80)==0)
				unload_picture(resnum);

		if((dir_pic[resnum].flags&RES_LOADED)==RES_LOADED)
		{
			if(pictures[resnum].rdata!=NULL)
			    free(pictures[resnum].rdata);	/* free raw data */
			pictures[resnum].rdata=NULL;
			dir_pic[resnum].flags&=~RES_LOADED;	/* remove loaded flag */
		}
		break;

	case rVIEW:
		if (dir_view[resnum].flags & RES_LOADED) {
			/* unload view removes ALL data for a view */
			unload_view(resnum);

			if (views[resnum].rdata)
				free(views[resnum].rdata);
			views[resnum].rdata = NULL;
		}
		dir_view[resnum].flags &= ~RES_LOADED;
		break;
	case rSOUND:
		if(dir_view[resnum].flags & RES_LOADED) {
			unload_sound(resnum);

			if (sounds[resnum].rdata)
				free(sounds[resnum].rdata);
			sounds[resnum].rdata = NULL;
		}

		dir_sound[resnum].flags &= ~RES_LOADED;
		break;
	}
	return ec;
}


/*

This function does noting but load a raw resource into memory,
if further decoding is required, it must be done by another
routine.

NULL is returned if unsucsessfull.

*/

UINT8* agi_v2_load_vol_res(AGI_DIR *agid)
{
	UINT8	x[256];
	UINT8	*data;
	FILE	*fp;

	sprintf((char*)x, "vol.%i", agid->volume);
	fixpath(NO_GAMEDIR, x);

	data=NULL;
	if(agid->offset!=_EMPTY &&  (fp=fopen((char*)path, "rb"))!=NULL)
	{
		fseek(fp, agid->offset, SEEK_SET);
		fread(&x, 1, 5, fp);
		if(hilo_getword(x)==0x1234)
		{
			agid->len=lohi_getword(x+3);
			data=(UINT8*)calloc(1, agid->len+32);
			if(data!=NULL)
				fread(data, 1, agid->len, fp);
		}
		else
		{
			gfx->deinit_video_mode();
			fprintf(stderr, "ACK! BAD RESOURCE!!!\n");
			exit(0);
		}
		fclose(fp);
	}
	else
	{
		/* we have a bad volume resource */
		/* set that resource to NA */
		agid->offset=_EMPTY;
	}

	return	data;
}

/*
 * Loads a resource into memory, a raw resource is loaded in
 * with above routine, then further decoded here.
 */

UINT16	agi_v2_load_resource (UINT8 restype, UINT16 resnum)
{
	UINT16	ec=err_OK;
	UINT8	*data=NULL;

	if(resnum > MAX_DIRS)
		return err_BadResource;

	switch(restype)
	{
	case rLOGIC:
		if (~dir_logic[resnum].flags & RES_LOADED)
		{
			loader->unload_resource (rLOGIC, resnum);
			/* load raw resource into data */
			data=agi_v2_load_vol_res(&dir_logic[resnum]);

			if ((logics[resnum].data = data))
				ec = decode_logic (resnum);
			else
       				ec = err_BadResource;

			logics[resnum].sIP = 2;
       		}

    		/* if logic was cached, we get here */
    		/* reset code pointers incase it was cached */

		/*logics[resnum].sIP=2;*/	/* saved IP = 2 */
   		/*logics[resnum].cIP=2;*/	/* current IP = 2 */

   		logics[resnum].cIP=logics[resnum].sIP;

		break;

	case rPICTURE:
		/* if picture is currently NOT loaded *OR* cacheing is off,
		   unload the resource (caching==off) and reload it */

		if((dir_pic[resnum].flags&RES_LOADED)==RES_LOADED)
			break;

		/* if loaded but not cached, unload it */
		/* if cached but not loaded, etc */
		loader->unload_resource(rPICTURE, resnum);
		data=agi_v2_load_vol_res(&dir_pic[resnum]);
		if(data!=NULL)
		{
			pictures[resnum].rdata=data;
			dir_pic[resnum].flags|=RES_LOADED;
		}
		else
			ec=err_BadResource;
		break;

	case rSOUND:
		if((dir_sound[resnum].flags&RES_LOADED)==RES_LOADED)
			break;

		data=agi_v2_load_vol_res(&dir_sound[resnum]);
		if(data!=NULL)
		{
			sounds[resnum].rdata=data;
			dir_sound[resnum].flags|=RES_LOADED;
			decode_sound(resnum);
		}
		else
			ec=err_BadResource;
		break;

	case rVIEW:
		/* Load a VIEW resource into memory...
		 * Since VIEWS alter the view table ALL the time
		 * can we cache the view? or must we reload it all
		 * the time?
		 */

		/* load a raw view from a VOL file into data */
		if ((dir_view[resnum].flags&RES_LOADED)!=RES_LOADED)
		{
    			loader->unload_resource (rVIEW, resnum);
    			data = agi_v2_load_vol_res (&dir_view[resnum]);

    			if (data) {
    				views[resnum].rdata = data;
    				dir_view[resnum].flags |= RES_LOADED;
    				ec = decode_view (resnum);
    			}
    			else
    				ec=err_BadResource;
    		}
		break;

	default:
		ec = err_BadResource;
		break;
	}


	return ec;
}

