/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import com.kvisco.xml.XMLPrinter;
import java.util.Hashtable;
import java.io.PrintWriter;
import org.w3c.dom.*;

/**
 * The Default class for processing Formatting Objects.
 * This class simply prints out the result tree and will
 * not perform any special formatting for FOs.
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class DefaultFormatter extends Formatter {

    public  static final String DOCTYPE_PI = "xsl:result-dtd";
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new DefaultFormatter
    **/
    public DefaultFormatter() {
        super();
    } //-- DefaultFormatter

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Defined by Formatter. Processes the given set of nodes.
     * @param nodeList the set of nodes to process
     * @param output the PrintStream for IO during processing.
     * @see org.w3c.dom.NodeList
    **/
    public void process(Document document, PrintWriter output) {

        if (document == null) return;
        
        XMLPrinter printer = new XMLPrinter(output);
        printer.setUnescapeCDATA(true);
        boolean indent = getIndentResult();
        if (indent) printer.setIndentSize(getIndentSize());
        printer.setUseFormat(indent);
        
        printer.print(document);
    } //-- process

    /**
     * Processes the given set of nodes
     * @param document the Document to process
     * @param output the PrintStream for IO during processing.
     * @see org.w3c.dom.Document
     * @see com.kvisco.xsl.Formatter
    **/
    public void process(Document document, PrintWriter output, OutputFormat format) 
    {
        
        if (document == null) return;
        if (format == null) process(document, output);
        
        XMLPrinter printer = new XMLPrinter(output);
        printer.setUnescapeCDATA(true);
        
        if (format.getIndent())
            printer.setIndentSize(getIndentSize());
            
        printer.setUseFormat(format.getIndent());
        
        String publicId = format.getDoctypePublicId();
        String systemId = format.getDoctypeSystemId();
        
        
        if (!format.getOmitXMLDeclaration()) {
            printer.printXMLDeclaration(format.getVersion(), "UTF-8");
        }
        
        Element root = document.getDocumentElement();
        
        if (root != null) {
            output.print("<!DOCTYPE ");
            output.print(root.getNodeName());
            if (publicId != null) {
                output.print(" PUBLIC \"");
                output.print(publicId);
                output.print('"');
            }
            if (systemId!= null) {
                output.print(" SYSTEM \"");
                output.print(systemId);
                output.print('"');
            }
            //-- handle internal subset
            //-- end DOCTYPE
            output.println('>');
        }
        
        NodeList nl = document.getChildNodes();
        for (int i = 0; i < nl.getLength(); i++) {
            printer.print(nl.item(i));
        }
        
        
        printer.print(document);
        
        process(document, output);
    } //-- process
} //-- DefaultFormatter