/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

import java.io.PrintWriter;
import java.util.Hashtable;
import java.util.StringTokenizer;
import com.kvisco.util.List;
import com.kvisco.util.QuickStack;



/**
 * This class represents a FilterExpr as defined by the
 * XSL Working Draft
 * <PRE>
 * [8] FilterExpr ::= NodeExpr ( '[' BooleanExpr ']' )?
 * </PRE>
 * @author Keith Visco (kvisco@ziplink.net)
**/
class FilterExpr extends FilterBase
{

    
    PrimaryExpr    primaryExpr = null;

      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new FilterExpr
    **/
    protected FilterExpr() {
        super();
    } //-- FilterExpr
    
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    public NodeSet evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        if (primaryExpr == null) return new NodeSet(0);
        
        ExprResult exprResult = primaryExpr.evaluate(context, ps);
        NodeSet nodeSet = null;
        
        switch (exprResult.getResultType()) {
            case ExprResult.NODE_SET:
                nodeSet = (NodeSet)exprResult;
                break;
            case ExprResult.TREE_FRAGMENT:
                nodeSet = new NodeSet(1);
                nodeSet.add(((TreeFragmentResult)exprResult).getValue());
                break;
            default:
                throw new InvalidExprException
                    ("expecting NodeSet or TreeFragment as the result of the "+
                        "expression: " + primaryExpr);
        }
        
        //-- filter nodes
        QuickStack nsStack = ps.getNodeSetStack();
        nsStack.push(nodeSet);
        evaluatePredicates(nodeSet, ps);
        nsStack.pop();
        return nodeSet;
        
    } //-- NodeSet
    
    
    /**
     * Determines the priority of a PatternExpr as follows:
     * <PRE>
     *  - From the 19990421 XSL Working Draft -
     *  + If the Pattern has the form of a QName optionally preceded by
     *    the @ character, then the priority is 0.
     *  + Otherwise if the pattern consists of just a NodeTest then the
     *    priority is -1
     *  + Otherwise the priority is 1
     * </PRE>
     * @return the priority for this PatternExpr
    **/
    public int getDefaultPriority() {
        return 1;
    } //-- getDefaultPriority
    
    
    public boolean matches(Node node, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodes = evaluate(node, ps);
        evaluatePredicates(nodes, ps);
        return nodes.contains(node);
    } //-- matches
    
    /**
     * Returns the String representation of this FilterExpr
     * @return the String representation of this FilterExpr
    **/
    public String toString() {
        if (primaryExpr == null)
            return super.toString();
        else {
            StringBuffer strbuf = new StringBuffer();
            strbuf.append(primaryExpr.toString());    
            strbuf.append(super.toString());
            return strbuf.toString();
        }
    } //-- toString
    
    
    public PrimaryExpr getPrimaryExpr() {
        return this.primaryExpr;
    } //-- getPrimaryExpr

    /*
    protected boolean isIDExpr() {
        if (nodeExpr != null) {
            return (nodeExpr.getNodeExprType() == NodeExpr.ID_EXPR);
        }
        return false;
    } //-- isIDExpr
    */
    
    protected void setPrimaryExpr(PrimaryExpr primaryExpr) {
        this.primaryExpr = primaryExpr;
    } //-- setPrimaryExpr

} //-- FilterExpr