/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xsl;

import org.w3c.dom.*;
import java.io.*;
import java.util.*;
import com.kvisco.xsl.functions.ErrorFunctionCall;

/**
 * TemplateRule
 * Represents an XSL Template Rule
 * Section 2.5 of the W3C XSL Working Draft 1.0 (19980818)
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class TemplateRule extends XSLObject {
    
	/**
	 * the MatchPatterns for this Rule
	**/
	private UnionExpr unionExpr = null;
	
	/**
	 * the name for this Rule.
	 * Peter Ciuffetti. added for WD-xslt-19990421
	**/
	private String templateName = null;
	
    private ErrorFunctionCall errorCall = null;
    
    private boolean allowParamVars = true;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new TemplateRule.<BR>
     * By default the new rule will not match any elements.
    **/
	public TemplateRule(XSLStylesheet parentStylesheet) {
	    super(parentStylesheet,XSLObject.TEMPLATE);
    } //-- TemplateRule

      //------------------/
     //- Public Methods -/
    //------------------/
    
    
	public XSLObject copy() {
	    TemplateRule tr = null;
	    tr = new TemplateRule(getParentStylesheet());
	    tr.setName(this.templateName);
	    tr.copyActions(this);
	    tr.copyAttributes(this);
	    return tr;
	} //-- copy

    public PathExpr getMatchingExpr
        (Node node, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        return unionExpr.getMatchingExpr(node, context, ps);
	} //-- getMatchingExpr
	
    /**
     * @returns the UnionExpr for this template
    **/
	public UnionExpr getMatchExpr() {
	    return unionExpr;
	} //--getMatchExpr

    public String getMode() {
        return getAttribute(Names.MODE_ATTR);
    } //-- getMode
    
    /**
     * @returns the name for this rule
    **/
	public String getName() {
	    return templateName;
	} //--getName

    /**
     * @return the priority for this TemplateRule
    **/
    public float calculatePriority(Node node, ProcessorState ps) { 
        
        String attValue = getAttribute(Names.PRIORITY_ATTR);
        if ((attValue != null) && (attValue.length() > 0)) {
            try {
                return Float.valueOf(attValue).floatValue();
            }
            catch(NumberFormatException nfe) {};
        }
        
        if (unionExpr == null) return -2;
        else {
            try {
                PathExpr pExpr = unionExpr.getMatchingExpr(node, node, ps);
                return (float) pExpr.getDefaultPriority();
            }
            catch (InvalidExprException iee) {};
        }
        
        return -2;
    } //-- calculatePriority

    /**
     * Determines if the given node is matched by this MatchExpr with
     * respect to the given context node.
     * @param node the node to determine a match for
     * @param context the Node which represents the current context
     * @param ps the current ProcessorState
     * @return true if the given node is matched by this MatchExpr
    **/
	public boolean matches(Node node, Node context, ProcessorState ps) 
	    throws InvalidExprException
    {
        
        if (errorCall != null) errorCall.evaluate(node, ps);
        if (unionExpr == null) return false;
	    return unionExpr.matches(node, context, ps);
	} //-- matches
	
    public void setAttribute(String name, String value) 
        throws XSLException
    {
        if (Names.MATCH_ATTR.equals(name)) {
        
            try {
                this.unionExpr = ExpressionParser.createUnionExpr(value);
            }
            catch (InvalidExprException iee) {
                ErrorFunctionCall efc = new ErrorFunctionCall();
                efc.setError("invalid 'match' attribute: " + value);
                this.errorCall = efc;
            }
            
        }
        super.setAttribute(name, value);
    }
    
    /**
     * Sets the MatchExpr for this TemplateRule
     * @param matchPattern the desired Match Pattern to use for this 
     * template
    **/
    public void setMatchAttr(String matchPattern) {
        if (matchPattern == null) matchPattern = "";
        try {
            setAttribute(Names.MATCH_ATTR, matchPattern);
        }
	    // an XSLException will never be thrown here
	    // since we are setting a valid attribute
        catch(XSLException xsle) {};
        
    } //-- setMatchExpr
    
    public void setModeAttr(String mode) {
        try {
            setAttribute(Names.MODE_ATTR,mode);
        }
	    // an XSLException will never be thrown here
	    // since we are setting a valid attribute
	    catch(XSLException xslException) {};
    } //-- setMode
    
    /**
     * sets the name for this TemplateRule
     * @param theName the desired name to use for this template
    **/
    public void setName(String theName) {
        this.templateName = theName;
        if (unionExpr == null) unionExpr = new UnionExpr();
    } //-- setName

    /**
     * Sets the priority for this TemplateRule
     * @param priority the desired priority of this rule.
     * 0 by default
    **/
	public void setPriority(float priority) {
	    try {
	        setAttribute(Names.PRIORITY_ATTR, String.valueOf(priority));
	    }
	    // an XSLException will never be thrown here
	    // since we are setting a valid attribute
	    catch(XSLException xslException) {};
	} //-- setPriority
    
    /**
     * Overrides handleAction in XSLObject
    **/
    protected boolean handleAction (XSLObject xslObject) {
        if (xslObject == null) return true;
        switch (xslObject.getType()) {
            case XSLObject.PARAM_VARIABLE:
                if (allowParamVars) break;
                else return true;
            default:
                allowParamVars = false;
                break;
        }
        return false;
    } //-- handleAction
    
} //-- TemplateRule