/* $Revision: 1.2 $ */
/*
 * "gpgp" Gnome/GTK Front for PGP
 * Copyright (C) 1998  Max Valianskiy
 * Copyright (c) 1999 tftp@netscape.net
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 */
#include <assert.h>

#include "../config.h"
#include "../version.h"
#include <gnome.h>
#include "gpgp.h"
#include "pgman.h"

enum 
{
    GNOME_PGP_ERROR_SIGNAL,
    LAST_SIGNAL
};

static gchar* KeyListTitles[] =
{
    N_("Type"),
    N_("Date"),
    N_("User ID"),
    N_("Validity")
};

struct add_pack
{
   GtkFileSelection* s;
   char* ring;
   GnomePgpKeyring* page;
};

struct export_pack
{
    GnomePgpKeyring* page;
    GtkFileSelection* w;
    GnomeDialog* di;
};

static int keylist_locale=0;
static gint gnome_pgp_keyring_signals[LAST_SIGNAL] = { 0 };

static void sign_del_cb (GtkWidget *widget, void *data);
static void sign_add_cb (GtkWidget *widget, void *data);
static void sign_export_cb (GtkWidget *widget, void *data);
static void destroy_cb (GtkWidget *widget, gpointer data);
static void sign_prop_cb (GtkWidget *widget, GnomePgpKeyring *page);
static void read_key_list (GnomePgpKeyring* page);
static void add2_cancel_cb(GtkWidget *widget, struct add_pack *data);

static void gnome_pgp_keyring_class_init(GnomePgpKeyringClass *klass);
static void gnome_pgp_keyring_init(GnomePgpKeyring      *ttt);

guint gnome_pgp_keyring_get_type()
{
  static guint gnome_pgp_keyring_type = 0;

  if (!gnome_pgp_keyring_type)
  {
      GtkTypeInfo gnome_pgp_keyring_info =
      {
          "GnomePgpKeyring",
          sizeof (GnomePgpKeyring),
          sizeof (GnomePgpKeyringClass),
          (GtkClassInitFunc) gnome_pgp_keyring_class_init,
          (GtkObjectInitFunc) gnome_pgp_keyring_init,
          (GtkArgSetFunc) NULL,
          (GtkArgGetFunc) NULL
      };
      gnome_pgp_keyring_type = gtk_type_unique(
          gtk_vbox_get_type(),
          &gnome_pgp_keyring_info);
  }
  return gnome_pgp_keyring_type;
}

static void gnome_pgp_keyring_class_init(
    GnomePgpKeyringClass *class)
{
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass*) class;
    gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL] =
        gtk_signal_new (
            "error",
            GTK_RUN_FIRST,
            object_class->type,
            GTK_SIGNAL_OFFSET (GnomePgpKeyringClass,error),
            gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
    gtk_object_class_add_signals(
        object_class, gnome_pgp_keyring_signals, LAST_SIGNAL);
    class->error = NULL;
}

static void secret_cb(
    GtkCList* list,
    gint row, gint column,
    GdkEvent* event,
    GnomePgpKeyring* data)
{
    char* text;
    int is_secret;

    gtk_clist_get_text( 
        GTK_CLIST(list),
        GPOINTER_TO_INT(GTK_CLIST(list)->selection->data),
        0,
        &text);
    is_secret = (text == NULL) || (strcmp(text,_("secret")) == 0);
    gtk_widget_set_sensitive(GTK_WIDGET(data->properties), !is_secret);
}

static void gnome_pgp_keyring_init(GnomePgpKeyring *page)
{
    GtkWidget* box2;
    GtkWidget* button2;
    GtkWidget* button3;
    GtkWidget* button5;

    /* Keylist i18n */
    if (!keylist_locale)
    {
        int i;
        for (i=0; i<4; i++)
            KeyListTitles[i] = gettext(KeyListTitles[i]);
        keylist_locale=1;
    }
  
    /* Create box2 */
    box2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start(GTK_BOX(page), box2, FALSE, TRUE, 0); 

    /* create button3 */
    button3 = gtk_button_new_with_label ( _("Add...") );
    gtk_signal_connect (GTK_OBJECT (button3), "clicked",
                        GTK_SIGNAL_FUNC (sign_add_cb), page);
    gtk_box_pack_start(GTK_BOX(box2), button3, FALSE, TRUE, 0); 

    /* create button1 */
    page->properties = GTK_BUTTON (gtk_button_new_with_label (_("Properties")));
    gtk_signal_connect (GTK_OBJECT (page->properties), "clicked",
                        GTK_SIGNAL_FUNC (sign_prop_cb), page);
    gtk_box_pack_start(GTK_BOX(box2), GTK_WIDGET(page->properties), FALSE, TRUE, 0); 

    /* create button2 */
    button2 = gtk_button_new_with_label ( _("Delete") );
    gtk_signal_connect (GTK_OBJECT (button2), "clicked",
                        GTK_SIGNAL_FUNC (sign_del_cb), page);
    gtk_box_pack_start(GTK_BOX(box2), button2, FALSE, TRUE, 0); 

    /* create button5 */
    button5 = gtk_button_new_with_label ( _("Export...") );
    gtk_signal_connect (GTK_OBJECT (button5), "clicked",
                        GTK_SIGNAL_FUNC (sign_export_cb), page);
    gtk_box_pack_start(GTK_BOX(box2), button5, FALSE, TRUE, 0); 

    /* add keylist */
    page->keylist = GTK_CLIST(gtk_clist_new_with_titles(4, KeyListTitles));
    gtk_clist_set_selection_mode (GTK_CLIST(page->keylist), GTK_SELECTION_BROWSE);

    /* FIXME! but how ? */
    gtk_clist_set_column_width(GTK_CLIST(page->keylist), 0, 60); 
    gtk_clist_set_column_width(GTK_CLIST(page->keylist), 1, 120); 
    gtk_clist_set_column_width(GTK_CLIST(page->keylist), 2, 270); 
    gtk_box_pack_start(GTK_BOX(page), GTK_WIDGET(page->keylist), TRUE, TRUE, 0); 
    gtk_signal_connect(GTK_OBJECT(page->keylist), "select_row",
                       GTK_SIGNAL_FUNC(secret_cb), page);

    gtk_widget_show_all(box2);
    gtk_widget_show_all(GTK_WIDGET(page->keylist));

    page->error_info=NULL;
    page->name=NULL;
    page->list=NULL;
}

/*
 * gnome_pgp_keyring_select()
 *
 * Procedure assigns new file name to the given keyring, re-reads
 * the key data from that file and redraws the GUI control. When we
 * assign new keyring filename we have to figure out if this a secret
 * keyring because it may need some different handling later on.
 *
 * History:
 * 07-May-99 tftp       Added comments.
 */
void gnome_pgp_keyring_select(GnomePgpKeyring* page, char* ring)
{
    assert(page != NULL);
    assert(ring != NULL);

    page->error_info=NULL;
    page->name=g_strdup(ring);
    page->is_secret = (strstr(ring, "sec") != NULL); /* FIXME stupid! */
    read_key_list(page);
    gtk_widget_queue_draw(GTK_WIDGET(page));
}

/*
 * gnome_pgp_keyring_new()
 *
 * Procedure creates (and returns pointer to) a new GnomePgpKeyring object.
 *
 * History:
 * 07-May-99 tftp       Added comments.
 */
GtkWidget *gnome_pgp_keyring_new(void)
{
    GnomePgpKeyring* page;
    page = gtk_type_new (gnome_pgp_keyring_get_type ());
    return GTK_WIDGET (page);
}

static void list_func(char** elm, void* clist)
{
 if (elm)
     gtk_clist_append(GTK_CLIST(clist), elm);
}

/*
 * read_key_list()
 *
 * Procedure reads specified keyring and refills the GTK_CLIST
 * associated with this keyring.
 *
 * History:
 * 07-May-99 tftp       Added 'is_secret' parameter to get_keylist() call.
 */
static void read_key_list(GnomePgpKeyring *page)
{
    GList* list;
 
    gtk_clist_clear( GTK_CLIST(page->keylist) );
    list=get_keylist(page->name, page->is_secret);
    if (pgman_error)
    { 
        page->error_info = pgman_error;
        gtk_signal_emit(
            GTK_OBJECT(page),
            gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL]);
        return;
    }
    g_list_foreach(list, (GFunc) list_func, page->keylist);
    g_list_free(list);
}

/* callbacks */

static int error=0;

static void report_error_cb(
    GnomePgpKeyringProperties *widget, GnomePgpKeyring* page)
{
    page->error_info = widget->error_info;
    gtk_signal_emit(
        GTK_OBJECT(page),
        gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL]);
    error=1;
}

static void sign_prop_cb(GtkWidget *widget, GnomePgpKeyring *page)
{
    GtkWidget* properties;
    char *text;

    gtk_clist_get_text( 
        GTK_CLIST(page->keylist),
        GPOINTER_TO_INT(GTK_CLIST(page->keylist)->selection->data),
        2,
        &text);
    properties=gnome_pgp_keyring_properties_new();
    error=0;
    gtk_signal_connect(
        GTK_OBJECT(properties),
        "error",
        GTK_SIGNAL_FUNC(report_error_cb),
        page);
    gnome_pgp_keyring_properties_select(
        GNOME_PGP_KEYRING_PROPERTIES(properties),
        page->name,text);
    if (error)
        gtk_widget_destroy(properties);
    else
        gtk_widget_show(properties);
}

static void del2_cb(gint reply, GnomePgpKeyring* page)
{
    if (reply==0)
    {
        char* text;
        gtk_clist_get_text( 
            GTK_CLIST(page->keylist),
            GPOINTER_TO_INT(GTK_CLIST(page->keylist)->selection->data),
            2,
            &text);
        pgp_del_key(page->name, text);
        gnome_pgp_keyring_refresh(page);
        if (pgman_error)
        {
            page->error_info=pgman_error;
            gtk_signal_emit(
                GTK_OBJECT(page),
                gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL]);
        }
    }
}

static void sign_del_cb(GtkWidget *widget, void *data)
{
    GtkWidget* msgbox;
    msgbox = gnome_question_dialog(
        _("Are you sure?"),
        (GnomeReplyCallback) del2_cb,
        data);
    gtk_widget_show(msgbox); 
}

static void
add2_cb(GtkWidget *widget, struct add_pack *data)
{
 pgp_add_key(gtk_file_selection_get_filename(data->s),data->ring); 

 if (pgman_error)
   {
    data->page->error_info=pgman_error;
    gtk_signal_emit(GTK_OBJECT(data->page), gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL]);
    add2_cancel_cb(widget,data);
    return;
   } 
 
 gnome_pgp_keyring_refresh(data->page);
 add2_cancel_cb(widget, data);
}

static void add2_cancel_cb(GtkWidget *widget, struct add_pack *data)
{
    gtk_widget_destroy(GTK_WIDGET(data->s));
    g_free(data);
}

static void sign_add_cb (GtkWidget *widget, void *data)
{
    GtkWidget* filesel;
    struct add_pack* pack=g_malloc(sizeof(struct add_pack));
 
    filesel=gtk_file_selection_new(_("Choose key file"));
    pack->s=GTK_FILE_SELECTION(filesel);
    pack->ring=((GnomePgpKeyring*)data)->name;
    pack->page=(GnomePgpKeyring*) data;
 
    gtk_signal_connect(
        GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
        "clicked",
        GTK_SIGNAL_FUNC(add2_cancel_cb),
        pack); 
    gtk_signal_connect(
        GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
        "clicked",
        GTK_SIGNAL_FUNC(add2_cb),
        pack); 
    gtk_widget_show(filesel);
}

static void destroy_cb(GtkWidget *widget, gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(data));
}

static void export2_cancel_cb(GtkWidget *widget, struct export_pack* data)
{
    if (data->di)
        gtk_widget_destroy(GTK_WIDGET(data->di));
    gtk_widget_destroy(GTK_WIDGET(data->w));
    g_free(data);
}

static void export2_append_cb(GtkWidget *widget, struct export_pack* data)
{
    char* text;
    gtk_clist_get_text( 
        GTK_CLIST(data->page->keylist),
        GPOINTER_TO_INT(GTK_CLIST(data->page->keylist)->selection->data),
        2,
        &text);
    pgp_export_key(
        data->page->name,
        text,
        gtk_file_selection_get_filename(data->w));

    if (pgman_error)
    {
        data->page->error_info=pgman_error;
        gtk_signal_emit(
            GTK_OBJECT(data->page),
            gnome_pgp_keyring_signals[GNOME_PGP_ERROR_SIGNAL]);
    }

    /* Destroy dialogs */
    export2_cancel_cb(widget,data);
} 

static void export2_over_cb(GtkWidget *widget, struct export_pack* data)
{
    unlink(gtk_file_selection_get_filename(data->w));
    export2_append_cb(widget,data);
}
 
static void
export2_cb (GtkWidget *widget, struct export_pack* data)
{
 if (g_file_exists(gtk_file_selection_get_filename(data->w)))
  {
   GtkWidget* msgbox;
   GtkWidget* bcancel;
   GtkWidget* bover;
   GtkWidget* bappend;
 
   /* create exist dialog */
   msgbox = gnome_message_box_new (_("File already exist"),
	             GNOME_MESSAGE_BOX_QUESTION,
		     NULL);

   bover = gtk_button_new_with_label(_("Overwrite"));
   bappend = gtk_button_new_with_label(_("Append"));
   bcancel = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);

   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bover,
                              FALSE, TRUE, GNOME_PAD_SMALL);
   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bappend,
                              FALSE, TRUE, GNOME_PAD_SMALL);
   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bcancel,
                              FALSE, TRUE, GNOME_PAD_SMALL);

   data->di=GNOME_DIALOG(msgbox);

   gtk_signal_connect (GTK_OBJECT (bover), "clicked",GTK_SIGNAL_FUNC (export2_over_cb), data);
   gtk_signal_connect (GTK_OBJECT (bappend), "clicked",GTK_SIGNAL_FUNC (export2_append_cb), data);
   gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",GTK_SIGNAL_FUNC (export2_cancel_cb), data);
   
   gtk_widget_show(bover);
   gtk_widget_show(bappend);
   gtk_widget_show(bcancel);
  
   gtk_widget_hide(GTK_WIDGET(data->w));
   
   gtk_widget_show(msgbox); 
   return;
  } 

  export2_append_cb(widget,data);
}

static void
sign_export_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;
 struct export_pack* ep=g_malloc(sizeof(struct export_pack));

 filesel=gtk_file_selection_new(_("Export To..."));
 gtk_file_selection_set_filename( GTK_FILE_SELECTION(filesel), "key.asc" );

 ep->w=GTK_FILE_SELECTION(filesel);
 ep->page=(GnomePgpKeyring*) data;
 ep->di=NULL;
 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel); 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(export2_cb),ep);
 gtk_widget_show(filesel);
}

void gnome_pgp_keyring_refresh(GnomePgpKeyring* page)
{
  read_key_list(page); 
  gtk_widget_queue_draw(GTK_WIDGET(page->keylist));
}
