/*
    ext2_inode_relocator.c -- ext2 inode relocator
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>	/* for S_ISDIR */
#include "ext2.h"







struct ext2_reference
{
	blk_t			 block;
	off_t			 offset;
};

struct ext2_inode_entry
{
	ino_t			 num;
	ino_t			 dest;
	unsigned		 numreferences:16;
	unsigned		 isdir:1;
	struct ext2_reference	*ref;
};

struct ext2_inode_relocator_state
{
	ino_t			 usedentries;
	int			 newgroups;
	struct ext2_inode_entry	*inode;
	struct ext2_reference	*last;
};





static int compare_inode_entries(const void *x0, const void *x1)
{
	const struct ext2_inode_entry *b0;
	const struct ext2_inode_entry *b1;

	b0 = (const struct ext2_inode_entry *)x0;
	b1 = (const struct ext2_inode_entry *)x1;

	if (b0->num < b1->num)
		return -1;

	if (b0->num > b1->num)
		return 1;

	return 0;
}

static struct ext2_inode_entry *findit(struct ext2_inode_relocator_state *state, ino_t inode)
{
	struct ext2_inode_entry *i;
	struct ext2_inode_entry key;

	key.num = inode;

	i = bsearch(&key, state->inode, state->usedentries, sizeof(struct ext2_inode_entry), compare_inode_entries);

	return i;
}

static int addref(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, ino_t inode, blk_t block, off_t offset)
{
	struct ext2_inode_entry *ent;
	int i;

	if ((ent = findit(state, inode)) == NULL)
		return 1;

	for (i=0;i<ent->numreferences;i++)
		if (!ent->ref[i].block)
			break;

	if (i == ent->numreferences)
	{
		fprintf(stderr, "ext2_inode_relocator: found an inode which has more links to it than it's inode descriptor said\n");
		return 0;
	}

	ent->ref[i].block = block;
	ent->ref[i].offset = offset;

	return 1;
}

static int doblock(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	off_t                    offset;

	if (!blockno)
		return 1;

	bh = ext2_bread(fs, blockno);

	offset = 0;
	do
	{
		struct ext2_dir_entry *ptr;

		ptr = (struct ext2_dir_entry *)(bh->data + offset);

		if (ptr->name_len)
			if (!addref(fs, state, ptr->inode, blockno, offset))
				return 0;

		offset += ptr->rec_len;
	} while (offset < fs->blocksize);

	ext2_brelse(bh, 1);         /* FIXXXME */
	return 1;
}

static int doindblock(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	int                      i;

	if (!blockno)
		return 1;

	bh = ext2_bread(fs, blockno);

	for (i=0;i<fs->blocksize>>2;i++)
		if (!doblock(fs, state, ((__u32 *)bh->data)[i]))
			return 0;

	ext2_brelse(bh, 1);        /* FIXXXME */
	return 1;
}

static int dodindblock(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	int                      i;

	if (!blockno)
		return 1;

	bh = ext2_bread(fs, blockno);

	for (i=0;i<fs->blocksize>>2;i++)
		if (!doindblock(fs, state, ((__u32 *)bh->data)[i]))
			return 0;

	ext2_brelse(bh, 1);      /* FIXXXME */
	return 1;
}

static int dotindblock(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, blk_t blockno)
{
	struct ext2_buffer_head *bh;
	int                      i;

	if (!blockno)
		return 1;

	bh = ext2_bread(fs, blockno);

	for (i=0;i<fs->blocksize>>2;i++)
		if (!dodindblock(fs, state, ((__u32 *)bh->data)[i]))
			return 0;

	ext2_brelse(bh, 1);        /* FIXXXME */
	return 1;
}

static int doinode(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, ino_t inode)
{
	struct ext2_inode buf;
	int		  i;

	ext2_read_inode(fs, inode, &buf);
	if (S_ISDIR(buf.i_mode))
	{
		for (i=0;i<EXT2_NDIR_BLOCKS;i++)
			if (!doblock(fs, state, buf.i_block[i]))
				return 0;

		if (!doindblock(fs, state, buf.i_block[EXT2_IND_BLOCK])
		    || !dodindblock(fs, state, buf.i_block[EXT2_DIND_BLOCK])
		    || !dotindblock(fs, state, buf.i_block[EXT2_TIND_BLOCK]))
			return 0;
	}

	return 1;
}

static int doscan(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;

	for (i=0;i<fs->numgroups;i++)
	{
		struct ext2_buffer_head *bh;
		int j;
		int offset;

		bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
		offset = i * fs->sb.s_inodes_per_group + 1;

		for (j=0;j<fs->sb.s_inodes_per_group;j++)
			if (bh->data[j>>3] & _bitmap[j&7])
				if (!doinode(fs, state, offset + j))
					return 0;

		ext2_brelse(bh, 0);
	}

	return 1;
}







static void ext2_inode_relocator_copy(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;

	for (i=0;i<state->usedentries;i++)
	{
		struct ext2_inode buf;
		struct ext2_inode_entry *entry;

		entry = &state->inode[i];

		if (!ext2_get_inode_state(fs, entry->num))
			fprintf(stderr, "inodebitmaperror\n");

		if (ext2_get_inode_state(fs, entry->dest))
			fprintf(stderr, "inodebitmaperror\n");

		ext2_read_inode(fs, entry->num, &buf);
		ext2_write_inode(fs, entry->dest, &buf);

		entry->isdir = S_ISDIR(buf.i_mode)?1:0;
	}
}

static void ext2_inode_relocator_finish(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;

	for (i=0;i<state->usedentries;i++)
	{
		struct ext2_inode_entry *entry;

		entry = &state->inode[i];
		ext2_set_inode_state(fs, entry->dest, 1, 1);
		ext2_set_inode_state(fs, entry->num, 0, 1);
		ext2_zero_inode(fs, entry->num);
	}
}

static void ext2_inode_relocator_ref(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;

	for (i=0;i<state->usedentries;i++)
	{
		struct ext2_inode_entry *entry;
		int			 j;
		__u32			 t;

		entry = &state->inode[i];
		t = entry->dest;

		for (j=0;j<entry->numreferences;j++)
		{
			struct ext2_buffer_head *bh;

			bh = ext2_bread(fs, entry->ref[j].block);
			bh->dirty = 1;

			if ((*((__u32 *)(bh->data + entry->ref[j].offset))) != entry->num)
				fprintf(stderr, "inodereferror\n");

			*((__u32 *)(bh->data + entry->ref[j].offset)) = t;
			ext2_brelse(bh, 0);
		}

		if (entry->isdir)
		{
			int oldgroup;
			int newgroup;

			oldgroup = (entry->num  - 1) / fs->sb.s_inodes_per_group;
			newgroup = (entry->dest - 1) / fs->sb.s_inodes_per_group;

			fs->gd[oldgroup].bg_used_dirs_count--;
			fs->gd[newgroup].bg_used_dirs_count++;

			fs->metadirty = 1;
		}
	}
}

static int ext2_inode_relocator_grab_inodes(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	int i;
	int ptr;

	ptr = 0;

	for (i=0;i<fs->numgroups;i++)
		if (fs->gd[i].bg_free_inodes_count)
		{
			struct ext2_buffer_head *bh;
			int j;
			int offset;

			bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
			offset = i * fs->sb.s_inodes_per_group + 1;

			j = i ? 0 : 13;
			for (;j<fs->sb.s_inodes_per_group;j++)
				if (!(bh->data[j>>3] & _bitmap[j&7]))
				{
					state->inode[ptr++].dest = offset + j;

					if (ptr == state->usedentries)
					{
						ext2_brelse(bh, 0);
						return 1;
					}
				}

			ext2_brelse(bh, 0);
		}

	return 0;
}

static int ext2_inode_relocator_flush(struct ext2_fs *fs, struct ext2_inode_relocator_state *state)
{
	if (!state->usedentries)
		return 1;

#if 1
	fprintf(stderr, "ext2_inode_relocator_flush\n");
#endif

	if (!doscan(fs, state))
		return 0;

	if (!ext2_inode_relocator_grab_inodes(fs, state))
		return 0;

	ext2_inode_relocator_copy(fs, state);
	ext2_inode_relocator_ref(fs, state);
	ext2_inode_relocator_finish(fs, state);

	state->usedentries = 0;
	state->last = (struct ext2_reference *)fs->relocator_pool_end;

	ext2_sync(fs);

	return 1;
}

static int ext2_inode_relocator_mark(struct ext2_fs *fs, struct ext2_inode_relocator_state *state, ino_t inode)
{
	struct ext2_inode	 buf;
	struct ext2_inode_entry *ent;
	int			 i;

	ext2_read_inode(fs, inode, &buf);

	{
		register void *adv;
		register void *rec;

		adv = state->inode + state->usedentries + 1;
		rec = state->last - buf.i_links_count;

		if (adv >= rec)
			ext2_inode_relocator_flush(fs, state);
	}

	state->last -= buf.i_links_count;

	ent = &state->inode[state->usedentries];
	ent->num = inode;
	ent->dest = 0;
	ent->numreferences = buf.i_links_count;
	ent->ref = state->last;

	for (i=0;i<ent->numreferences;i++)
	{
		ent->ref[i].block = 0;
		ent->ref[i].offset = 0;
	}

	state->usedentries++;

	return 1;
}


int ext2_inode_relocate(struct ext2_fs *fs, int newgroups)
{
	int i;
	struct ext2_inode_relocator_state state;

#if 1
	fprintf(stderr, "ext2_inode_relocate\n");
#endif

	state.usedentries = 0;
	state.newgroups = newgroups;
	state.inode = (struct ext2_inode_entry *)fs->relocator_pool;
	state.last = (struct ext2_reference *)fs->relocator_pool_end;

	for (i=newgroups;i<fs->numgroups;i++)
	{
		struct ext2_buffer_head *bh;
		int j;
		int offset;

		bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
		offset = i * fs->sb.s_inodes_per_group + 1;

		for (j=0;j<fs->sb.s_inodes_per_group;j++)
			if (bh->data[j>>3] & _bitmap[j&7])
				ext2_inode_relocator_mark(fs, &state, offset + j);

		ext2_brelse(bh, 0);
	}

	ext2_inode_relocator_flush(fs, &state);

	return 1;
}

