/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef GNOME_APT_DRAW_TREE_H
#define GNOME_APT_DRAW_TREE_H

#include <gtk/gtk.h>
#include <vector>
#include <string>

class DrawTree {
public:
  class Node;
  class NodeList;

  DrawTree();
  ~DrawTree();

  // must be done *first*
  void set_nodelist(NodeList* nl);

  GtkWidget* widget() { return da_; }
  
  // do this manually if the represented data changes
  //  or the display otherwise gets out of date.
  void queue_display_update();
  void queue_recalc_rows();

  void change_selection(Node* newnode);
  void change_selection(gint x, gint y); // to row at those coords, if any

  void focus_node(Node* node);

  bool openclose_node(Node* node);

  // move + or - steps, if there's no selection
  //  select an on-screen row; if the selection is
  //  off-screen jump to it before the move.
  void move_selection(gint steps);

  enum ColumnStyle {
    BoolCol,         // check item; no label drawn
    DrawCol,         // Node::draw_column is called
    TreeCol          // DrawCol which accounts for tree depth and draws expanders.
  };

  void set_visible(gushort col, bool visibility);
  bool is_visible(gushort col) const;

  void set_columns(const vector<ColumnStyle> & cols, 
                   const vector<string> & column_names,
                   const vector<guint> & shortcuts1,
                   const vector<guint> & shortcuts2);
  void set_column(gushort col, ColumnStyle style);
  
  void get_column_widths(vector<gint> & widths);
  void set_column_widths(const vector<gint> & widths);

  // We keep these here so every item doesn't 
  //  have to have one.
  GdkGC*   gc() {
    g_return_val_if_fail(gc_,0);
    return gc_;
  }

  GdkFont* font() { 
    g_return_val_if_fail(font_,0);
    return font_; 
  }

  Node* selected_node();

  void set_hadjustment(GtkAdjustment* a);
  void set_vadjustment(GtkAdjustment* a);

  // A major hack, but necessary without a GtkObject
  //  to attach accelerators to.
  gint consider_key_event(GdkEventKey* event);

private:
  // called by static callback
  gint configure(GdkEventConfigure* event);
  static gint configure_event(GtkWidget* w, GdkEventConfigure* event, gpointer data);

  gint expose(GdkEventExpose* event);
  static gint expose_event(GtkWidget* w, GdkEventExpose* event, gpointer data);
  
  gint button(GdkEventButton* event);
  static gint button_event(GtkWidget* w, GdkEventButton* event, gpointer data);

  gint key(GdkEventKey* event);
  static gint key_event(GtkWidget* w, GdkEventKey* event, gpointer data);

  gint motion(GdkEventMotion* event);
  static gint motion_event(GtkWidget* w, GdkEventMotion* event, gpointer data);

  gint leave(GdkEventCrossing* event);
  static gint leave_event(GtkWidget* w, GdkEventCrossing* event, gpointer data);

  static void realize_cb(GtkWidget* w, gpointer data);
  void realize();

  // for internal use (by idle callback), use queue version
  void do_display_update();
  static gint update_idle(gpointer data);

  GtkWidget* da_;
  GdkPixmap* pixmap_;

  bool update_queued_;

  gint row_gap_;
  gint col_gap_;
  gint row_height_;
  gint indent_; // for each tree depth
  gint margin_;
  gint titles_height_;

  struct Column {
    ColumnStyle style;

    string name;

    int width;

    // if the window is resized larger than we need, each column gets padded.
    //  but we need to save the user-defined size
    int extra;

    // cache x positions for speed
    int x;

    guint shortcut1;
    guint shortcut2;

    guint visible : 1;

  };

  vector<Column*> columns_;

  NodeList* nodes_;

  // Nodes on screen at any given time. Cached for speed.
  vector<Node*> visible_nodes_;
  vector<gushort> depths_;

  // recalc_rows sets focus_index to the index of a node
  //  matching this one. it's a hack so we can use recalc_rows
  //  rather than writing a find_row_number function
  Node* focus_node_; 
  guint focus_index_;

  void recalc_widths();
  gint total_width_;
  
  static gint recalc_rows_idle(gpointer data);
  void recalc_rows();
  bool recalc_rows_queued_;
  // row which begins current display
  guint row_; 

  // rows in the display
  guint nrows_; 

  // keeping this instead of an index forces us to occasionally
  //  
  Node* selected_node_;
  gint prelight_;    // -1 is magic for none

  GdkGC* gc_;
  GdkFont* font_;

  GtkAdjustment* hadjustment_;
  GtkAdjustment* vadjustment_;

  // so we can avoid changing things if we get a configure event
  //  with the same size we already have. This prevents 
  //  flicker on pane changes
  gint pixwidth_;
  gint pixheight_; 

  gint row_offset(guint row) { 
    return margin_*2 + 
      titles_height_ + 
      ((row - row_) * 
       (row_height_ + row_gap_)); }
  // returns -1 for none; returns offset from current row_
  gint row_from_coords(gint x, gint y);
  // -1 for none
  gint column_from_x  (gint x); 

  guint max_rows_onscreen();
  float exact_rows_onscreen();

  // arg can be negative
  void scroll_by(gint rows);

  void draw_expander(GdkRectangle* cell_rect, bool expanded);
  void update_row(guint visible_row_number, GtkStateType state, 
                  bool copy_to_window = true);

  gint row_index(Node* n);

  void setup_adjustments();
  static void hadjustment_changed_signal(GtkAdjustment* a, gpointer data);
  static void vadjustment_changed_signal(GtkAdjustment* a, gpointer data);
  void hadjustment_changed();
  void vadjustment_changed();

  // Column resize stuff
  class Resize {
  public:
    Resize(size_t left, size_t right) :
      left_(left), right_(right), xor_gc_(0) {}

    size_t left_;
    size_t right_;
    GdkGC* xor_gc_;
  };
  
  // Non-0 means we are resizing.
  Resize* resize_;

  guint update_idle_id_;
  guint recalc_idle_id_;

  void draw_resize_line();
};

// Subclass this to make a class you can stuff into the tree
class DrawTree::Node {
public:
  Node() : hidden_(false) {}
  // deletes children
  virtual ~Node();

  typedef vector<Node*>::iterator iterator;

  iterator begin() { return children_.begin(); }
  iterator end() { return children_.end(); }

  virtual Node* parent() = 0;

  virtual void expand() = 0;   // create children, unless you kept them around.
  virtual void collapse() = 0; // delete children, if convenient
  virtual bool expanded() = 0; // draw expansion?
  virtual bool expandable() = 0; // draw expander?
  virtual void refresh_expansion() = 0; // update child list if needed

  // currently visible rows (1 for item, rows() for each child)
  guint rows();

  // get current state for a check item
  virtual bool get_bool(gushort column) = 0;
  // set state of object due to check item being checked
  virtual void set_bool(gushort column, bool state) = 0;
  // whether this bool is interesting for this object
  virtual bool display_bool(gushort column) = 0;

  // Drawing starts after any tree expander thingies.
  //  x/y accounts for that, plus depth.
  // 'remaining' is how many more columns will be drawn 
  //   (in case the Node wants to keep some data cached)
  // Tree draws the background in the proper state, nodes
  // should just do the labels/pixmaps
  virtual void draw_column(gushort column, gushort remaining,
                           GtkStateType state,
                           GdkDrawable* drawable, 
                           GdkRectangle* rect) = 0;

  virtual void select() = 0;
  virtual void unselect() = 0;

  bool hidden() { return hidden_; }
  void hide() { hidden_ = true; }
  void show() { hidden_ = false; }

protected:
  // The thing about a vector is that it doesn't shrink when emptied. 
  //  Calling clear() might do it tho. Check 
  vector<Node*> children_;

  guint hidden_ : 1;

private:

};

class DrawTree::NodeList {
public:
  typedef vector<Node*>::iterator iterator;
  
  virtual iterator begin() = 0;
  virtual iterator end() = 0;

private:

  
};

#endif
