/* catsboot.c */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>

/* catsboot <output> <kernel> [<initrd>] */
/* catsboot <output> -r <device> */

#define GLUE "/usr/lib/catsboot/catsglue.bin"

void usage(void)
{
  fprintf(stderr, "Usage:\n  catsboot <output> <kernel> [<initrd>]\n"
	  "  catsboot <output> -r <device>\n");
  exit(1);
}

void do_rdev(char *image, char *dev)
{
  struct stat sb;
  int fd;
  if (stat(dev, &sb))
    {
      perror(dev);
      exit(1);
    }
  if (!(sb.st_mode & S_IFBLK))
    {
      fprintf(stderr, "%s: not a block device.\n", dev);
      exit(1);
    }
  fd = open(image, O_WRONLY);
  if (fd < 0)
    {
      perror(image);
      exit(1);
    }
  if (lseek(fd, 0x80, SEEK_SET))
    {
      perror("lseek");
      exit(1);
    }
  if (write(fd, &sb.st_dev, 2))
    {
      perror("write");
      exit(1);
    }
  if (close(fd))
    {
      perror("close");
      exit(1);
    }
  exit(0);
}

int main(int argc, char *argv[])
{
  char *output, *kernel, *initrd;
  char *buf;
  unsigned long initrd_size, kernel_size;
  int fd;
  struct stat sb;
  unsigned long size = 0x8000;
  if (argc < 3 || argc > 4)
    usage();

  output = argv[1];
  if (!strcmp(argv[2], "-r"))
    {
      if (argc < 4)
	usage();
      do_rdev(output, argv[3]);
    }

  kernel = argv[2];
  if (argc == 4)
    initrd = argv[3];
  else
    initrd = NULL;

  if (stat(kernel, &sb))
    {
      perror(kernel);
      exit(1);
    }
  kernel_size = (sb.st_size + 3) & ~3;
  size += kernel_size;

  if (initrd)
    {
      if (stat(initrd, &sb))
	{
	  perror(initrd);
	  exit(1);
	}
      initrd_size = (sb.st_size + 3) & ~3;
      size += initrd_size;
    }

  buf = malloc(size);
  if (!buf)
    {
      fprintf(stderr, "Memory exhausted.\n");
      exit(1);
    }

  memset(buf, 0, 0x8000);
  fd = open(GLUE, O_RDONLY);
  if (fd < 0)
    {
      perror(GLUE);
      exit(1);
    }
  read(fd, buf, 0x8000);
  close(fd);

  fd = open(kernel, O_RDONLY);
  if (fd < 0)
    {
      perror(kernel);
      exit(1);
    }
  read(fd, buf + 0x8000, kernel_size);
  close(fd);

  if (initrd)
    {
      fd = open(initrd, O_RDONLY);
      if (fd < 0)
	{
	  perror(initrd);
	  exit(1);
	}
      read(fd, buf + 0x8000 + kernel_size, initrd_size);
      close(fd);
    }

  fd = open(output, O_RDWR | O_CREAT, 0660);
  if (fd < 0)
    {
      perror(output);
      exit(1);
    }

  memcpy(buf + 4, &size, 4);
  if (initrd)
    {
      int i = 0x8000 + kernel_size;
      memcpy(buf + 0x84, &i, 4);
      memcpy(buf + 0x88, &initrd_size, 4);
      i = 0x100;
      memcpy(buf + 0x80, &i, 4);
    }
  else
    {
      if (stat("/", &sb))
	{
	  perror("/");
	  exit(1);
	}

      memcpy(buf + 0x80, &sb.st_dev, 4);
    }

  write(fd, buf, size);
  close(fd);

  exit(0);
}

