package Language::INTERCAL::Backend::PerlText;

# Back end for CLC-INTERCAL to generate Perl code

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION);
$VERSION = '0.05';

use Language::INTERCAL::Opcodes ':BC_IDS', '@BC_NAMES';
use Language::INTERCAL::Runtime::Library;

use Fcntl;
use Config;
use FileHandle;

sub backend {
    my ($name, $ptree, $filename, $line, @args) = @_;
    @args == 1 or die
	'SYNTAX IS "backend PARSE_TREE \'PerlText\', <program_name>" '
      . " at $filename line $line\n";
    my $dest = shift @args;
    my $prog_name;
    my $close;
    if (ref $dest) {
	$prog_name = '_clc_intercal_';
	$close = '';
    } else {
	$prog_name = $dest;
	$prog_name =~ s!^.*/!!;
	$prog_name =~ s!\.[^\.]*$!!;
	$prog_name =~ s/\W|^\d/_/g;
	$close = $dest;
	$dest = new FileHandle('> ' . $dest) or
	    die "012 $dest\: $! (at $filename line $line)\n";
    }
    my $intercal = 'CLC-INTERCAL ' . $VERSION;
    my $perl = _generate_perl($ptree, $prog_name);
    my $gotos = $perl =~ s/\bgoto\b/goto/g;
    output($dest, _default_init($prog_name, $intercal, $gotos));
    output($dest, 'i_initialise ', $perl, ';', "\n");
    output($dest, _default_pod($prog_name, $intercal));
    if ($close ne '') {
	close $dest;
	chmod 0777 & ~ umask, $close;
    }
}

sub suffix { '.pl' }

my $codesize = 0;

sub _generate_perl {
    my ($ptree, $label) = @_;
    $label = 'intercal_' . $label . '_';

    my $flags = $ptree->{'flags'};
    my $dbhook = $flags->{'dbhook'};
    my $quantum = $dbhook || $flags->{'quantum'};
    my $post = $dbhook || $flags->{'postprocess'};

    # step 1: do some precalculation like initial abstain values etc.
    # Results are kept in $stmt->[7] for each statement.
    my @reinstate = ();
    $codesize = 0;
    $ptree->iterate(sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	$stmt->[7] = [[], $codesize++, -1, $stmt->[0]];
	my $can_change_reinstate = $dbhook;
	if ($post) {
	    # a future version will be more clever, for now:
	    $can_change_reinstate = 1;
	} else {
	    if ($stmt->[0]) {
		my $list;
		for $list (qw(abstain reinstate fork)) {
		    my $ab;
		    for $ab (@{$ptree->{$list}}) {
			my $st = $ptree->{'files'}[$ab->[1]][$ab->[2]][$ab->[3]];
			$can_change_reinstate = 1
			    if ($st->[3][1][0] == E_CONSTANT &&
				$stmt->[0] == atoi($st->[3][1][1]))
			    || ($st->[3][1][0] != E_CONSTANT);
		    }
		}
	    }
	    if (! $can_change_reinstate) {
		my $list;
		for $list (qw(gabstain greinstate gfork)) {
		    my $ab;
		    for $ab (@{$ptree->{$list}}) {
			my @st = @{$ptree->{'files'}
				 [$ab->[1]][$ab->[2]][$ab->[3]][3]};
			shift @st;
			$can_change_reinstate = 1
			    if grep {$_ == $stmt->[3][0]} @st;
		    }
		}
	    }
	}
	if ($can_change_reinstate) {
	    $stmt->[7][2] = scalar @reinstate;
	    push @reinstate, $stmt->[1];
	}
    });

    # step 2: generate actual code, and save it in $stmt->[7][0]
    my $thread = $quantum ? '[$thread]' : '';
    my $reinst = $quantum ? '[0]' : '';
    $ptree->{'ULAB'} = {};
    $ptree->{'EVENTS'} = [];
    $ptree->{'LOOPS'} = [];
    $ptree->iterate(sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	my ($_barf1, $thislabel, $reinstate, $_barf3) = @{$stmt->[7]};
	my $double_oh_seven = '';
	if (ref $stmt->[2]) {
	    $double_oh_seven = _expr($thread, $stmt->[2], 1, 0) . ' > rand(100)';
	} elsif ($stmt->[2] < 100) {
	    $double_oh_seven = $stmt->[2] . ' > rand(100)';
	}
	my $next = '';
	if ($fid + 1 < @{$ptree->{'files'}}) {
	    $next = $label . 'S' . ($fid + 1);
	}
	my @st = ();
	if ($reinstate >= 0 || $stmt->[1]) {
	    @st = _stmt($stmt, $label, $ptree, $next, $quantum, $post);
	}
	if (@st && $reinstate >= 0) {
	    # abstain can change at runtime
	    my $dos = $double_oh_seven ? ' && ' . $double_oh_seven : '';
	    for (@st) { s/^\s/    $&/ }
	    $stmt->[7][0] = ["    if (\$reinstate$thread\[$reinstate]$reinst$dos) {",
			     @st,
			     "    }"],
	} elsif (@st && $stmt->[1]) {
	    # statement always reinstated
	    if ($double_oh_seven) {
		for (@st) { s/^\s/    $&/ }
		$stmt->[7][0] = ["    if ($double_oh_seven) {",
				 @st,
				 "    }"],
	    } else {
		$stmt->[7][0] = [@st];
	    }
	} else {
	    $stmt->[7][0] = []; # this is effectively a NOP
	}
    });

    # finally, the code generation
    my ($use, $decl) =
	_declarations($ptree, $quantum, $post, $dbhook, \@reinstate);
    my $text = "{\n";
    for my $u (@$use) {
	$text .= "    use $u;\n";
    }
    for my $name (@$decl) {
	my $my = $name->[0] =~ /\[/ ? '' : 'my ';
	$text .= "    $my$name->[0] = $name->[1];\n";
    }
    if ($dbhook) {
	$text .= "    \$program_counter[0] = '${label}C0';\n";
	$text .= "    my \%retval = ();\n";
	for my $name (@$decl) {
	    next if $name->[0] =~ /\[/;
	    $text .= "    \$retval{'$name->[0]'} = \\$name->[0];\n";
	}
	$text .= "    \$retval{'step'} = sub {\n";
	$text .= "        my \$stop_now = eval {\n";
	$text .= '            goto $program_counter[$thread];' . "\n";
    } elsif ($quantum) {
	$text .= "    eval {\n";
    }
    my $s = $dbhook ? '        ' : $quantum ? '    ' : '';
    $ptree->iterate(sub {
	my ($p, $fid, $bid, $sid, $stmt) = @_;
	my @l = ();
	push @l, $label . 'S' . $fid . ':' if $fid && $bid == 1 && $sid == 0;
	push @l, $label . 'L' . $stmt->[7][1] . ':' if $stmt->[7][3];
	if (@{$stmt->[7][0]} || $stmt->[0]) {
	    if ($dbhook) {
		my $csw = $label . 'C' . $stmt->[7][1];
		push @l, "    \$program_counter[\$thread] = '$csw';",
			 '    $thread++;',
			 '    $thread = 0 if $thread >= @program_counter;',
			 '    return 0;',
			 "$csw\:";
	    } elsif ($quantum) {
		# "context switch"
		my $csw = $label . 'C' . $stmt->[7][1];
		push @l, "    \$program_counter[\$thread] = '$csw';",
			 '    $thread++;',
			 '    $thread = 0 if $thread >= @program_counter;',
			 '    goto $program_counter[$thread];',
			 "$csw\:";
	    }
	    if (@{$ptree->{'EVENTS'}}) {
		my $events = $quantum ? '@{$events[$thread]}' : '@events';
		my $eret = $quantum ? '@{$eret[$thread]}' : '@eret';
		my $lab = $label . 'A' . $stmt->[7][1];
		push @l, "$lab\:",
			 "    _events(\\$eret, '$lab', \\$events);";
	    }
	    if (@{$ptree->{'LOOPS'}}) {
		my $loops = $quantum ? '@{$loops[$thread]}' : '@loops';
		my $loop = $quantum ? '$loop[$thread]' : '$loop';
		my $lret = $quantum ? '@{$lret[$thread]}' : '@lret';
		if ($ptree->{'flags'}{'loop_swap'}) {
		    my $lab = $label . 'B' . $stmt->[7][1];
		    push @l, "    if (\$doing_loops$thread) {",
			     "        \$doing_loops$thread = 0;",
		    	     "        $loop = -1;",
			     "$lab\:",
			     "        $loop ++;",
			     "        if ($loop < $loops) { push $lret, '$lab'; goto \$loops$thread\[$loop][\$loops$thread\[$loop][2]] }",
			     "        \$doing_loops$thread = 1;",
			     "    }";
		} else {
		    my $lab = $label . 'B' . $stmt->[7][1];
		    push @l, "    $loop = -1;",
			     "$lab\:",
			     "    $loop ++;",
			     "    if ($loop < $loops) { push $lret, '$lab'; goto \$loops$thread\[$loop] }";
		}
	    }
	}
	push @l, @{$stmt->[7][0]};
	if ($quantum || $dbhook) {
	    for (@l) { s/^\s/$s$&/ }
	}
	$text .= join("\n", @l, '');
	if ($stmt->[0]) {
	    my @cf = ();
	    my @ccf = ();
	    my %records = ('come_froms' => S_COME);
	    if ($post) {
		$records{'nexts'} = S_NEXT;
		$records{'abstain'} = S_ABSTAIN;
		$records{'reinstate'} = S_REINSTATE;
		$records{'fork'} = S_FORK;
	    }
	    for my $record (keys %records) {
		my $c = $ptree->{$record};
		my $opcode = $records{$record};
		my $cf;
		for $cf (@$c) {
		    my $st = $ptree->{'files'}[$cf->[1]][$cf->[2]][$cf->[3]];
		    my $ex = $st->[3][1];
		    if ($ex->[0] == E_CONSTANT) {
			push @cf, [$cf->[1], $cf->[2], $cf->[3], 0, $opcode]
			    if $stmt->[0] == atoi($ex->[1]);
		    } else {
			push @ccf, [$cf->[1], $cf->[2], $cf->[3], $ex, $opcode];
		    }
		}
	    }
	    my $count = @cf + @ccf;
	    if ($count) {
		$text .= "$s    \@come_from = ();\n" if $count > 1;
		my $push = $count > 1 ? 'push @come_from, \'' : 'goto ';
		my $p = $count > 1 ? "'" : '';
		for my $cf (@cf) {
		    my $st = $ptree->{'files'}[$cf->[0]][$cf->[1]][$cf->[2]];
		    my $d07 = '';
		    if (ref $st->[2]) {
			$d07 = ' if ' . _expr($thread, $st->[2], 1, 0) . ' > rand(100)';
		    } elsif ($st->[2] < 100) {
			$d07 = ' if ' . $st->[2] . ' > rand(100)';
		    }
		    if ($post) {
			$d07 =~ s/if/&&/;
			$d07 = " if \$opcodes$thread\[$cf->[4]][0] == "
			     . S_COME . $d07;
		    }
		    if ($st->[7][2] >= 0) {
			$d07 =~ s/if/&&/;
			$text .= "$s    $push${label}L$st->[7][1]$p if \$reinstate$thread\[$st->[7][2]]$reinst$d07;\n";
		    } elsif ($st->[1]) {
			$text .= "$s    $push${label}L$st->[7][1]$p$d07;\n";
		    } else {
			# this COME FROM is always abstained from
		    }
		}
		for my $cf (@ccf) {
		    my $st = $ptree->{'files'}[$cf->[0]][$cf->[1]][$cf->[2]];
		    my $d07 = '';
		    if (ref $st->[2]) {
			$d07 = ' && ' . _expr($thread, $st->[2], 1, 0) . ' > rand(100)';
		    } elsif ($st->[2] < 100) {
			$d07 = ' && ' . $st->[2] . ' > rand(100)';
		    }
		    my $op = '';
		    if ($post) {
			$op = " && \$opcodes$thread\[$cf->[4]][0] == " . S_COME;
		    }
		    my $v = "$stmt->[0] == " . _expr($thread, $cf->[3], 1, 0);
		    if ($st->[7][2] >= 0) {
			$text .= "$s    $push${label}L$st->[7][1]$p if \$reinstate$thread\[$st->[7][2]]$reinst && $v$op$d07;\n";
		    } elsif ($st->[1]) {
			$text .= "$s    $push${label}L$st->[7][1]$p if $v$op$d07;\n";
		    } else {
			# this COME FROM is always abstained from
		    }
		}
		if ($count > 1) {
		    if ($ptree->{'flags'}{'thick'}) {
			my $qent = '_qenter($thread, \\@regs, \\@reinstate, '
				 . '\\@program_counter, shift(@come_from), '
				 . '\\@loop_bc, \\@loop_cb, \\@lecture_stack, '
				 . '\\@next_stack, \\@loops, \\@loop, '
				 . '\\@doing_loops, \\@eret, \\@events'
				 . ($post ? ', \\@opcodes' : '') . ');';
			$text .= "$s    while (\@come_from > 1) { \n"
			       . "$s        $qent\n"
			       . "$s    }\n";
		    } else {
			$text .= "$s    die \"555 MULTIPLE COME FROM ($stmt->[0])\\n\" if \@come_from > 1;\n";
		    }
		    $text .= "$s    goto \$come_from[0] if \@come_from;\n";
		}
	    }
	}
    });
    $text .= "$s    die \"633 FALLING OFF THE EDGE OF PROGRAM\\n\";\n";
    my $ul;
    for $ul (keys %{$ptree->{'ULAB'}}) {
	$text .= "${label}U$ul:\n"
	       . "$s    die \"129 LABEL ($ul) NOT DEFINED\\n\";\n";
    }
    for $ul (@{$ptree->{'EVENTS'}}) {
	my ($lab, @code) = @$ul;
	for (@code) { s/^\s/$s$&/ }
	my $eret = $quantum ? '@{$eret[$thread]}' : '@eret';
	$text .= join("\n", $lab . ':', @code, $s . '    goto (pop ' . $eret . ');', '');
    }
    for $ul (@{$ptree->{'LOOPS'}}) {
	my $lret = $quantum ? '@{$lret[$thread]}' : '@lret';
	if ($ptree->{'flags'}{'loop_swap'}) {
	    my ($l1, $l2, $c1, $c2) = @$ul;
	    for (@$c1, @$c2) { s/^\s/$s$&/ }
	    $text .= join("\n", $l1 . ':', @$c1, $s . '    goto (pop ' . $lret . ');', '');
	    $text .= join("\n", $l2 . ':', @$c2, $s . '    goto (pop ' . $lret . ');', '');
	} else {
	    my ($lab, @code) = @$ul;
	    for (@code) { s/^\s/$s$&/ }
	    $text .= join("\n", $lab . ':', @code, $s . '    goto (pop ' . $lret . ');', '');
	}
    }
    if ($dbhook) {
	$text .= $label . "END:\n            return 1;\n";
    } elsif ($quantum || $post) {
	$text .= $label . "END:\n";
    }
    if ($dbhook) {
	my $qexit = '_qexit($thread, \\@program_counter, \'' . $label . 'END\', '
		  . '\\@splats, \\@regs, \\@lecture_stack, \\@next_stack, '
		  . '\\@loops, \\@loop, \\@reinstate, \\@loop_bc, \\@loop_cb, '
		  . '\\@doing_loops, \\@eret, \\@events'
		  . ', \\@opcodes);';
	$text .= "        };\n"
	       . "        if (\$\@) {\n"
	       . "            chomp(\$\@);\n"
	       . "            push \@splats, \$\@ if \$\@;\n"
	       . "            \$lastsplat = \$1 if \$\@ =~ /^(\\d{3}) /;\n"
	       . "            $qexit\n"
	       . "        }\n"
	       . "        return \$stop_now;\n"
	       . "    };\n"
	       . "    return \\\%retval;\n";
    } elsif ($quantum) {
	my $qexit = '_qexit($thread, \\@program_counter, \'' . $label . 'END\', '
		  . '\\@splats, \\@regs, \\@lecture_stack, \\@next_stack, '
		  . '\\@loops, \\@loop, \\@reinstate, \\@loop_bc, \\@loop_cb, '
		  . '\\@doing_loops, \\@eret, \\@events'
		  . ($post ? ', \\@opcodes' : '') . ');';
	$text .= "    };\n"
	       . "    if (\$\@) {\n"
	       . "        chomp(\$\@);\n"
	       . "        push \@splats, \$\@ if \$\@;\n"
	       . "        \$lastsplat = \$1 if \$\@ =~ /^(\\d{3}) /;\n"
	       . "        $qexit\n"
	       . "    }\n";
    }
    $text .= "}\n";
    while ($text =~ s/\n($label\w+):\n($label\w+):/\n$1:/) {
	my $gone = $2;
	my $stay = $1;
	$text =~ s/$gone/$stay/g;
    }
    delete $ptree->{'ULAB'};
    delete $ptree->{'EVENTS'};
    delete $ptree->{'LOOPS'};
    $text;
}

sub _stmt {
    my ($stmt, $label, $ptree, $next, $quantum, $post) = @_;
    my $q = $quantum ? 'q' : '';
    my $t = $quantum ? '[$thread]' : '';
    my $s = $stmt->[3];
    my $op = $s->[0];
    my $nx = $label . 'F' . $stmt->[7][1];
    my $qenter = '_qenter($thread, \\@regs, \\@reinstate, \\@program_counter, '
	       . '\'' . $nx . '\', \\@loop_bc, \\@loop_cb, \\@lecture_stack, '
	       . '\\@next_stack, \\@loops, \\@loop, \\@doing_loops, '
	       . '\\@eret, \\@events'
	       . ($post ? ', \\@opcodes' : '') . ');';
    if ($op == S_NOP) {
	();
    } elsif ($op == S_COME) {
	$stmt->[7][3] = 1;
	if ($post) {
	    _post_label($ptree, $label, $t, $stmt);
	} else {
	    ();
	}
    } elsif ($op == S_STOP) {
	if ($post) {
	    $next = $label . 'END' if ! $quantum && ! $next;
	    ("    &{\$opcodes$t\[$op][2]}('$next');");
	} elsif ($next) {
	    ("    goto $next;");
	} elsif ($quantum) {
	    ('    die "\\n";');
	} else {
	    ("    return 0;");
	}
    } elsif ($op == S_ILLEGAL) {
	my $err = $s->[1];
	$err =~ s/[\000-\037\177-\377\$\@\"\\]/sprintf('\\%03o', ord($&))/ge;
	('    die "' . $err . '\\n";');
    } elsif ($op == S_BUG || $op == S_UBUG) {
	my $err = ($op == S_BUG ? '' : 'UNEXPLAINABLE ') . 'COMPILER ERROR';
	('    die "774 ' . $err . '\\n";');
    } elsif ($op == S_ABSTAIN || $op == S_REINSTATE || $op == S_FORK) {
	if ($post) {
	    $stmt->[7][3] = 1;
	    return _post_label($ptree, $label, $t, $stmt);
	}
	my $lab = $s->[1];
	my @pre = ();
	my @post = ();
	my $reinst = '';
	my @change = ();
	if ($quantum) {
	    $reinst = '[0]';
	}
	if ($op == S_FORK) {
	    push @pre, '    $new_thread = ' . $qenter;
	    push @post, $nx . ':';
	    push @change, '    $reinstate[$thread][STMT] = [1, 1];';
	    push @change, '    $reinstate[$new_thread][STMT] = [0, 1];';
	} elsif ($op == S_ABSTAIN) {
	    push @change, '    $reinstate' . $t . '[STMT]' . $reinst . ' = 0;';
	} else {
	    push @change, '    $reinstate' . $t . '[STMT]' . $reinst . ' = 1;';
	}
	if ($lab->[0] == E_CONSTANT) {
	    my $l = atoi($lab->[1]);
	    if (exists $ptree->{'labels'}{$l}) {
		my @l = @{$ptree->{'labels'}{$l}};
		my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		return () if $stmt->[3][0] == S_STOP;
		(@pre,
		 (map { my $x = $_; $x =~ s/STMT/$stmt->[7][2]/g; $x} @change),
	         @post);
	    } else {
		();
	    }
	} else {
	    my @c = ('    $label = ' . _expr($t, $lab, 1, 0) . ';');
	    my $lab = $ptree->{'labels'};
	    my $l;
	    for $l (keys %$lab) {
		my @l = @{$lab->{$l}};
		my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		next if $stmt->[3][0] == S_STOP;
		push @c, '    if (' . $stmt->[0] . ' == $label) {',
		         (map { my $x = '    ' . $_; $x =~ s/STMT/$stmt->[7][2]/g; $x} @change),
			 '    }';
	    }
	    (@pre, @c, @post);
	}
    } elsif ($op == S_GABSTAIN || $op == S_GREINSTATE || $op == S_GFORK) {
	my @pre = ();
	my @post = ();
	my $reinst = '';
	my @change = ();
	if ($quantum) {
	    $reinst = '[0]';
	}
	my %opcodes = ();
	for (my $i = 1; $i < @$s; $i++) {
	    $opcodes{$s->[$i]} = 1;
	}
	delete $opcodes{&S_STOP};
	if ($post) {
	    push @pre, '    $new_value = $opcodes' . $t . '[' . $op . '][0] == ' . S_GABSTAIN . ' ? 0 : 1;';
	    if ($quantum) {
		push @pre, '    if ($opcodes' . $t . '[' . $op . '][0] == ' . S_GFORK . ') {';
		push @pre, '        $new_thread = ' . $qenter;
		push @pre, '    } else {';
		push @pre, '        $new_thread = -1;';
		push @pre, '    }';
	    }
	    push @pre, '    %opcodes = (' . join(', ', map {($_ => 1)} keys %opcodes) . ');';
	    push @change, '    if (exists $opcodes{$opcodes' . $t . '[OPCODE][0]}) {';
	    if ($quantum) {
		push @change, '        $reinstate' . $t . '[STMT] = [$new_value, 1];';
		push @change, '        $reinstate[$new_thread][STMT] = [0, 1] if $new_thread >= 0;';
	    } else {
		push @change, '        $reinstate' . $t . '[STMT] = $new_value;';
	    }
	    push @change, '    }';
	    push @post, $nx . ':';
	} elsif ($op == S_GFORK) {
	    push @pre, '    $new_thread = ' . $qenter;
	    push @post, $nx . ':';
	    push @change, '    $reinstate[$thread][STMT] = [1, 1];';
	    push @change, '    $reinstate[$new_thread][STMT] = [0, 1];';
	} elsif ($op == S_GABSTAIN) {
	    push @change, '    $reinstate' . $t . '[STMT]' . $reinst . ' = 0;';
	} else {
	    push @change, '    $reinstate' . $t . '[STMT]' . $reinst . ' = 1;';
	}
	my @stmt = ();
	$ptree->iterate(sub {
	    my ($p, $fid, $bid, $sid, $stmt) = @_;
	    push @stmt, [$stmt->[7][2], $stmt->[3][0]]
		if $post || exists $opcodes{$stmt->[3][0]};
	});
	return () if ! @stmt;
	(@pre,
	 (map { my ($s, $o) = @$_;
		map { my $x = $_; $x =~ s/STMT/$s/g; $x =~ s/OPCODE/$o/g; $x }
		    @change }
	      @stmt),
	 @post);
    } elsif ($op == S_IGNORE || $op == S_REMEMBER || $op == S_VFORK) {
	if ($post) {
	    my @rp = ();
	    for (my $i = 1; $i < @$s; $i++) {
		push @rp, _reg($t, $s->[$i]);
	    }
	    my $rp = join(', ', @rp);
	    return ("    &{\$opcodes$t\[$op][2]}('$nx', $rp);",
		    $nx . ':');
	}
	my @pre = ();
	my @post = ();
	my @change = ();
	if ($op == S_VFORK) {
	    push @pre, '    $new_thread = ' . $qenter;
	    push @post, $nx . ':';
	    push @change, '    _qvfork(\\@regs, \\%shregs, REG, $thread, $new_thread);';
	} else {
	    my $remember = $op == S_IGNORE ? 0 : 1;
	    push @change, '    if (exists $regs' . $t . '{REG}) {';
	    push @change, '        $regs' . $t . '{REG}[1] = ' . $remember . ';';
	    if ($quantum) {
		push @change, '    } elsif (exists $shregs{REG}) {';
		push @change, '        $shregs{REG}[1] = $remember;';
		push @change, '        $regs[$thread]{REG} = \$shregs{REG};';
	    }
	    push @change, '    } else {';
	    push @change, '        $regs' . $t . '{REG} = [0, ' . $remember . ', [], 0, 0, [], []];';
	    if ($quantum) {
		push @change, '        $shregs{REG} = $regs[$thread]{REG};';
	    }
	    push @change, '    }';
	}
	my @c = ();
	for (my $i = 1; $i < @$s; $i++) {
	    my $rp = _reg($t, $s->[$i]);
	    push @c, map { my $x = $_; $x =~ s/REG/$rp/g; $x } @change;
	}
	(@pre, @c, @post);
    } elsif ($op == S_STASH || $op == S_RETRIEVE) {
	if ($post) {
	    my @rp = ();
	    for (my $i = 1; $i < @$s; $i++) {
		push @rp, _reg($t, $s->[$i]);
	    }
	    my $rp = join(', ', @rp);
	    return ("    &{\$opcodes$t\[$op][2]}('$nx', $rp);",
		    $nx . ':');
	}
	my @c = ();
	my $func = $op == S_STASH ? "_${q}stash" : "_${q}retrieve";
	for (my $i = 1; $i < @$s; $i++) {
	    my $rp = _reg($t, $s->[$i]);
	    if ($quantum) {
		push @c, "    $func(\$regs$t, \\%shregs, $rp);";
	    } else {
		push @c, "    $func(\\%regs, $rp);";
	    }
	}
	@c;
    } elsif ($op == S_ENSLAVE) {
	my $slave = _reg($t, $s->[1]);
	my $owner = _reg($t, $s->[2]);
	return ("    &{\$opcodes$t\[$op][2]}($slave, $owner);") if $post;
	my @c = ();
	push @c, "    if (exists \$regs$t\{$slave}) {";
	push @c, "        unshift \@{\$regs$t\{$slave}[2]}, $owner if \$regs$t\{$slave}[1];";
	if ($quantum) {
	    push @c, "    } elsif (exists \$shregs{$slave}) {";
	    push @c, "        \$regs$t\{$slave} = \$shregs{$slave};";
	    push @c, "        unshift \@{\$regs{$slave}[2]}, $owner if \$regs{$slave}[1];";
	}
	push @c, "    } else {";
	push @c, "        \$regs$t\{$slave} = [0, 1, [$owner], 0, 0, [], []];";
	if ($quantum) {
	    push @c, "        \$shregs{$slave} = \$regs$t\{$slave};";
	}
	push @c, "    }";
	@c;
    } elsif ($op == S_FREE) {
	my $slave = _reg($t, $s->[1]);
	my $owner = _reg($t, $s->[2]);
	if ($post) {
	    ("    &{\$opcodes$t\[$op][2]}($slave, $owner);");
	} elsif ($quantum) {
	    ("    _qfree(\$regs$t, \\%shregs, $slave, $owner);");
	} else {
	    ("    _free(\\%regs, $slave, $owner);");
	}
    } elsif ($op == S_ASSIGN) {
	my $rp = _reg($t, $s->[1]);
	my @val = ();
	for (my $i = 2; $i < @$s; $i++) {
	    push @val, _expr($t, $s->[$i], 0, 0);
	}
	my $val = join(', ', $rp, @val);
	if ($post) {
	    my $ret = $label . 'R' . $stmt->[7][1];
	    ("    &{\$opcodes$t\[$op][2]}('$ret', $val);",
	     $ret . ':');
	} elsif ($quantum) {
	    ('    _qassign($regs[$thread], \\%shregs, ' . $val . ');');
	} else {
	    ('    _assign(\\%regs, ' . $val . ');');
	}
    } elsif ($op == S_STUDY) {
	my @c = ();
	my $lecture = '';
	if ($s->[2][0] == E_CONSTANT) {
	    my $lab = atoi($s->[2][1]);
	    if (exists $ptree->{'labels'}{$lab}) {
		my @l = @{$ptree->{'labels'}{$lab}};
		my $st = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		$lecture = "'${label}L$st->[7][1]'";
	    } else {
		$lecture = "'${label}U$lab'";
		$ptree->{'ULAB'}{$lab} = 1;
	    }
	} else {
	    push @c, '    $lecture = ' . _expr($t, $s->[2], 1, 0) . ';';
	    push @c, '    die "999 NO LECTURES BEFORE (1000)\\n" if $lecture < 1000;';
	    my $els = '';
	    my $l;
	    for $l (keys %{$ptree->{'labels'}}) {
		my @l = @{$ptree->{'labels'}{$l}};
		my $st = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		push @c, "    ${els}if (\$lecture == $st->[0]) {",
			 "        \$lecture = '${label}L$st->[7][1]';";
		$els = '} els';
	    }
	    if ($els) {
		push @c, '    } else {';
		push @c, "        die \"129 LABEL (\$lecture) NOT DEFINED\\n\";";
		push @c, "    }";
	    } else {
		push @c, "    die \"129 LABEL (\$lecture) NOT DEFINED\\n\";";
	    }
	    $lecture = '$lecture';
	}
	push @c, '    $lectures{'
			 . _reg($t, $s->[3])
			 . '}{'
			 . _expr($t, $s->[1], 1, 0)
			 . '} = ' . $lecture . ';';
	@c;
    } elsif ($op == S_ENROL) {
	my $rp = _reg($t, $s->[1]);
	my @subj = ();
	for (my $i = 2; $i < @$s; $i++) {
	    push @subj, _expr($t, $s->[$i], ! $post, 0);
	}
	my $subj = join(', ', @subj);
	if ($post) {
	    my $ret = $label . 'R' . $stmt->[7][1];
	    return ("    &{\$opcodes$t\[$op][2]}('$ret', $rp, $subj);",
		    $ret . ':');
	}
	my @c = ();
	push @c, "    \$class = _enrol(\\%lectures, $subj);";
	push @c, "    if (exists \$regs$t\{$rp}) {";
	push @c, "        push \@{\$regs$t\{$rp}[6]}, \$class if \$regs$t\{$rp}[1];";
	if ($quantum) {
	    push @c, "    } elsif (exists \$shregs{$rp}) {";
	    push @c, "        \$regs$t\{$rp} = \$shregs{$rp};";
	    push @c, "        push \@{\$regs$t\{$rp}[6]}, \$class if \$regs$t\{$rp}[1];";
	}
	push @c, "    } else {";
	push @c, "        \$regs$t\{$rp} = [0, 1, [], 0, 0, [], [\$class]];";
	if ($quantum) {
	    push @c, "        \$shregs{$rp} = \$regs$t\{$rp};";
	}
	push @c, "    }";
	@c;
    } elsif ($op == S_LEARN) {
	my @c = ();
	my $rp = _reg($t, $s->[1]);
	my $subj = _expr($t, $s->[2], ! $post, 0);
	my $ret = $label . 'R' . $stmt->[7][1];
	if ($post) {
	    push @c, "    &{\$opcodes$t\[$op][2]}('$ret', $rp, $subj);";
	} elsif ($quantum) {
	    push @c, "    goto (_qlearns(\$regs$t, \\%shregs, $rp, \\%lectures, $subj, \$lecture_stack$t, '$ret'));";
	} else {
	    push @c, "    goto (_learns(\\%regs, $rp, \\%lectures, $subj, \\\@lecture_stack, '$ret'));";
	}
	push @c, $ret . ':';
	@c;
    } elsif ($op == S_FINISH) {
	if ($post) {
	    $next = $label . 'END' if ! $quantum && ! $next;
	    ("    &{\$opcodes$t\[$op][2]}('$next');");
	} elsif ($quantum) {
	    ('    _qfinish($regs[$thread], \\%shregs, $lecture_stack[$thread]);');
	} else {
	    ('    _finish(\\%regs, \\@lecture_stack);');
	}
    } elsif ($op == S_GRADUATES) {
	my $rp = _reg($t, $s->[1]);
	return ("    &{\$opcodes$t\[$op][2]}($rp);") if $post;
	my @c = ();
	push @c, "    if (exists \$regs$t\{$rp}) {";
	push @c, "        \$regs$t\{$rp}[6] = [] if \$regs$t\{$rp}[1];";
	if ($quantum) {
	    push @c, "    } elsif (exists \$shregs{$rp}) {";
	    push @c, "        \$shregs{$rp}[6] = [] if \$shregs{$rp}[1];";
	    push @c, "        \$regs$t\{$rp} = \$shregs{$rp};";
	}
	push @c, "    }";
	@c;
    } elsif ($op == S_WHILE_E) {
	# in a non-quantum program, SPLAT is never valid
	return () if $s->[2][0] == E_SPLAT && ! $quantum;
	my $expr = _expr($t, $s->[2], 0, 0);
	my @st = (0, 1, 100, $s->[1], 0, 0, 0, [[], $codesize++, -1, 0]);
	my @code = _stmt(\@st, $label, $ptree, $next, $quantum, $post);
	my $lab = $label . 'E' . $stmt->[7][1];
	push @{$ptree->{'EVENTS'}}, [$lab, @code];
	my $events = $quantum ? '@{$events[$thread]}' : '@events';
	("    push $events, [sub { $expr }, '$lab'];");
    } elsif ($op == S_READ) {
	my @rp = ();
	for (my $i = 1; $i < @$s; $i++) {
	    push @rp, _reg($t, $s->[$i]);
	}
	my $rp = join(', ', @rp);
	if ($post) {
	    ("    &{\$opcodes$t\[$op][2]}('$nx', $rp);", "$nx\:");
	} elsif ($quantum) {
	    ("    _qread(\$regs$t, \\%shregs, \$output, \$roman, \\\$outval, \$arrayio, $rp);");
	} else {
	    ("    _read(\\%regs, \$output, \$roman, \\\$outval, \$arrayio, $rp);");
	}
    } elsif ($op == S_WRITE) {
	my @rp = ();
	for (my $i = 1; $i < @$s; $i++) {
	    push @rp, _reg($t, $s->[$i]);
	}
	my $rp = join(', ', @rp);
	if ($post) {
	    ("    &{\$opcodes$t\[$op][2]}('$nx', $rp);", "$nx\:");
	} elsif ($quantum) {
	    ("    _qwrite(\$regs$t, \\%shregs, \$input, \\\$inval, \$arrayio, $rp);");
	} else {
	    ("    _write(\\%regs, \$input, \\\$inval, \$arrayio, $rp);");
	}
    } elsif ($op == S_WHILE_CB || $op == S_WHILE_BC) {
	if ($ptree->{'flags'}{'loop_swap'}) {
	    my $index = ($op == S_WHILE_CB ? '$loop_cb' : '$loop_bc') . $t;
	    $index .= '[0]' if $quantum;
	    my @s1 = (0, 1, 100, $s->[1], 0, 0, 0, [[], $codesize++, -1, 0]);
	    my @c1 = _stmt(\@s1, $label, $ptree, $next, $quantum, $post);
	    my @s2 = (0, 1, 100, $s->[2], 0, 0, 0, [[], $codesize++, -1, 0]);
	    my @c2 = _stmt(\@s2, $label, $ptree, $next, $quantum, $post);
	    my $l1 = $label . 'E' . $stmt->[7][1];
	    my $l2 = $label . 'F' . $stmt->[7][1];
	    my $lr1 = $label . 'R' . $stmt->[7][1];
	    my $lr2 = $label . 'S' . $stmt->[7][1];
	    push @{$ptree->{'LOOPS'}}, [$l1, $l2, \@c1, \@c2];
	    my $push = $post || grep { /\bgoto\b|\@loops\b/ } @c1, @c2;
	    my $loops = $quantum ? '@{$loops[$thread]}' : '@loops';
	    my $lret = $quantum ? '@{$lret[$thread]}' : '@lret';
	    ("    push $lret, '$lr1';",
	     "    goto (('$l1', '$l2')[$index]);",
	     "$lr1\:",
	     ($push ? ("    push $loops, ['$l1', '$l2', $index];") : ()),
	     "    push $lret, '$lr2';",
	     "    goto (('$l2', '$l1')[$index]);",
	     "$lr2\:",
	     ($push ? ("    pop $loops;") : ()));
	} else {
	    my $body = $s->[$op == S_WHILE_CB ? 2 : 1];
	    my @st = (0, 1, 100, $body, 0, 0, 0, [[], $codesize++, -1, 0]);
	    my @body = _stmt(\@st, $label, $ptree, $next, $quantum, $post);
	    my $cond = $s->[$op == S_WHILE_CB ? 1 : 2];
	    my @tt = (0, 1, 100, $cond, 0, 0, 0, [[], $codesize++, -1, 0]);
	    my @cond = _stmt(\@tt, $label, $ptree, $next, $quantum, $post);
	    my $loops = $quantum ? '@{$loops[$thread]}' : '@loops';
	    if ($post || grep { /\bgoto\b|\@loops\b/ } @cond) {
		# condition involves several steps
		my $lab1 = $label . 'E' . $stmt->[7][1];
		my $lab2 = $label . 'F' . $stmt->[7][1];
		push @{$ptree->{'LOOPS'}}, [$lab1, @body];
		my $lret = $quantum ? '@{$lret[$thread]}' : '@lret';
		("    push $lret, '$lab2';",
		 "    goto $lab1;",
		 "$lab2\:",
		 "    push $loops, '$lab1';",
		 @cond,
		 "    pop $loops;");
	    } else {
		# condition is just one step, so body would execute once
		(@body, @cond);
	    }
	}
    } elsif ($op == S_NEXT) {
	if ($post) {
	    $stmt->[7][3] = 1;
	    return _post_label($ptree, $label, $t, $stmt);
	}
	my $lab = $s->[1];
	my $nxt = $label . 'N' . $stmt->[7][1];
	my @c = ('    die "401 YOUR PROGRAM IS OBSOLETE\\n" if ! $next_ok;');
	if ($quantum) {
	    push @c, "    unshift \@{\$next_stack$t}, '$nxt';";
	} else {
	    push @c, "    unshift \@next_stack, '$nxt';";
	}
	if ($lab->[0] == E_CONSTANT) {
	    my $lab = atoi($lab->[1]);
	    if (exists $ptree->{'labels'}{$lab}) {
		my @l = @{$ptree->{'labels'}{$lab}};
		my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		push @c, "    goto ${label}L$stmt->[7][1];";
	    } else {
		push @c, "    die \"129 LABEL ($lab) NOT DEFINED\\n\";";
	    }
	} else {
	    push @c, '    $label = ' . _expr($t, $lab, 1, 0) . ';';
	    my $lab = $ptree->{'labels'};
	    my $l;
	    for $l (keys %$lab) {
		my @l = @{$lab->{$l}};
		my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
		push @c, '    goto ' . $label . 'L' . $stmt->[7][1] .
			 ' if $label == ' . $l . ';';
	    }
	    push @c, '    die "129 LABEL ($label) NOT DEFINED\\n";';
	}
	push @c, $nxt . ':';
	@c;
    } elsif ($op == S_FORGET || $op == S_RESUME) {
	my $val = _expr($t, $s->[1], 1, 0);
	my @c = ('    die "401 YOUR PROGRAM IS OBSOLETE\\n" if ! $next_ok;');
	push @c, '    $label = ' . $val . ';';
	push @c, '    die "621 FORGETTING OR RESUMING TOO LITTLE\\n" if $label < 1;';
	push @c, '    die "623 FORGETTING OR RESUMING TOO MUCH\\n" if $label > @next_stack;';
	if ($post) {
	    push @c, '    $rlabel = $next_stack' . $t . '[$label - 1] if ' . S_RESUME . ' == $opcodes' . $t . '[' . $op . '][0];';
	} elsif ($op == S_RESUME) {
	    push @c, '    $rlabel = $next_stack' . $t . '[$label - 1];';
	}
	if ($quantum) {
	    push @c, '    splice(@{$next_stack[$thread]}, 0, $label);';
	} else {
	    push @c, '    splice(@next_stack, 0, $label);';
	}
	if ($post) {
	    push @c, '    goto $rlabel if ' . S_RESUME . ' == $opcodes' . $t . '[' . $op . '][0];';
	} elsif ($op == S_RESUME) {
	    push @c, '    goto $rlabel;';
	}
	@c;
    } elsif ($op == S_SWAP || $op == S_SFORK) {
	my $op1 = $s->[1][0];
	my $op2 = $s->[2][0];
	my @c = ();
	my @e = ();
	if ($op == S_SFORK) {
	    push @c, '    ' . $qenter;
	    push @e, $nx . ':';
	}
	return (@c, @e) if $op1 == $op2;
	my $lt = $quantum ? ($t . '[0]') : '';
	if (($op1 == S_WHILE_BC && $op2 == S_WHILE_CB) ||
	    ($op1 == S_WHILE_CB && $op2 == S_WHILE_BC))
	{
	    if ($op == S_SFORK) {
		push @c, '    ($loop_bc[$thread], $loop_cb[$thread]) = ([$loop_cb[$thread][0], 1], [$loop_bc[$thread][0], 1]);';
	    } else {
		push @c, '    ($loop_bc' . $lt . ', $loop_cb' . $lt . ') = ($loop_cb' . $lt . ', $loop_bc' . $lt . ');';
	    }
	} elsif ($post) {
	    @c = ();
	    push @c, "    &{\$opcodes$t\[$op][2]}('$nx', $op1, $op2);";
	    push @c, $nx . ':';
	} else {
	    die "Internal error - this code should have never been reached (ops=[$op1, $op2], code=$op)\n";
	}
	(@c, @e);
    } elsif ($op == S_CONVERT || $op == S_CFORK) {
	my $op1 = $s->[1][0];
	my $op2 = $s->[2][0];
	my @c = ();
	my @e = ();
	if ($op == S_SFORK) {
	    push @c, '    ' . $qenter;
	    push @e, $nx . ':';
	}
	return (@c, @e) if $op1 == $op2;
	my $lt = $quantum ? ($t . '[0]') : '';
	if ($op1 == S_WHILE_BC && $op2 == S_WHILE_CB) {
	    if ($op == S_CFORK) {
		push @c, '    $loop_bc[$thread] = [$loop_cb[$thread][0], 1];';
	    } else {
		push @c, '    $loop_bc' . $lt . ' = $loop_cb' . $lt . ';';
	    }
	} elsif ($op1 == S_WHILE_CB && $op2 == S_WHILE_BC) {
	    if ($op == S_CFORK) {
		push @c, '    $loop_cb[$thread] = [$loop_bc[$thread][0], 1];';
	    } else {
		push @c, '    $loop_cb' . $lt . ' = $loop_bc' . $lt . ';';
	    }
	} elsif ($post) {
	    @c = ();
	    push @c, "    &{\$opcodes$t\[$op][2]}('$nx', $op1, $op2);";
	    push @c, $nx . ':';
	} else {
	    die "Internal error - this code should have never been reached (ops=[$op1, $op2], code=$op)\n";
	}
	(@c, @e);
    } else {
	die "Internal compiler error (opcode = $op in function 37)\n";
    }
}

sub _post_label {
    my ($ptree, $label, $t, $stmt) = @_;
    my $s = $stmt->[3];
    my $lab = $s->[1];
    my $reinstate;
    my $opcode;
    my $go_label;
    my $back_label = $label . 'N' . $stmt->[7][1];
    my @c = ();
    if ($lab->[0] == E_CONSTANT) {
	my $l = atoi($lab->[1]);
	if (exists $ptree->{'labels'}{$l}) {
	    my @l = @{$ptree->{'labels'}{$l}};
	    my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
	    $reinstate = $stmt->[7][2];
	    $opcode = $stmt->[3][0];
	    $go_label = "'${label}L$stmt->[7][1]'";
	} else {
	    $reinstate = -1;
	    $opcode = 0;
	    $go_label = "''";
	}
    } else {
	push @c, '    $label = ' . _expr($t, $lab, 1, 0) . ';',
		 '    $reinstate = -1;',
		 '    $opcode = 0;';
	my $lb = $ptree->{'labels'};
	my $l;
	for $l (keys %$lb) {
	    my @l = @{$lb->{$l}};
	    my $stmt = $ptree->{'files'}[$l[0]][$l[1]][$l[2]];
	    push @c, '    $reinstate = ' . $stmt->[7][2] . ', '
		   . '$opcode = ' . $stmt->[3][0] . ', '
		   . '$clabel = \'' . $label . 'L' . $stmt->[7][1] . '\''
		   . ' if ' . $stmt->[0] . ' == $label;';
	}
	$reinstate = '$reinstate';
	$opcode = '$opcode';
	$go_label = '$clabel';
    }
    push @c, "    &{\$opcodes$t\[$s->[0]][2]}($reinstate, $go_label, '$back_label', $opcode);";
    push @c, $back_label . ':';
    @c;
}

sub _reg {
    my ($thread, $reg) = @_;
    my $op = $reg->[0];
    if ($op == E_REGISTER) {
	return "'$reg->[1]$reg->[2]'";
    } elsif ($op == E_SUBSCRIPT) {
	my $rp = _reg($thread, $reg->[1]);
	my @val = ();
	for (my $i = 2; $i < @$reg; $i++) {
	    push @val, _expr($thread, $reg->[$i], 1, 0);
	}
	if ($thread ne '') {
	    my $val = join(', ', $rp, @val);
	    return '_qsubscript($regs' . $thread . ', \\%shregs, ' . $val . ')';
	} else {
	    my $val = join(', ', $rp, @val);
	    return '_subscript(\\%regs, ' . $val . ')';
	}
    } elsif ($op == E_OWNER) {
	my @path = ();
	while ($op == E_OWNER) {
	    unshift @path, $reg->[1];
	    $reg = $reg->[2];
	    $op = $reg->[0];
	}
	my $rp = _reg($thread, $reg);
	my $path = join(', ', @path);
	if ($thread eq '') {
	    return "_owner(\\%regs, $rp, $path)";
	} else {
	    return "_qowner(\$regs$thread, \\%shregs, $rp, $path)";
	}
    } elsif ($op == E_INDIRECT) {
	my $rtype = _reg($thread, $reg->[1]);
	my $rnumber = _reg($thread, $reg->[2]);
	"do { my \$rp = $rnumber; " .
	'$rp =~ s/^.//; ' .
	'$rp = $` if $rp =~ /\s/; ' .
	"substr($rtype, 0, 1) . \$rp " .
	'}';
    } else {
	die "Internal compiler error (opcode = $op in function 25)\n";
    }
}

sub _expr {
    my ($thread, $e, $int, $lv) = @_;
    my $op = $e->[0];
    my $a0 = $int ? 'atoi(' : '';
    my $a1 = $int ? ')' : '';
    if ($op == E_CONSTANT) {
	return 'die "275 ASSIGNMENT TO CONSTANT\\n";' if $lv;
	my $v = $e->[1];
	return atoi($v) if $int;
	$v =~ s/[\000-\377]/sprintf('\\%03o', ord($&))/ge;
	return '"' . $v . '"';
    } elsif ($op == E_SPLAT) {
	return 'die "275 ASSIGNMENT TO SPLAT\\n";' if $lv;
	return 'die "456 NO SPLAT\\n"' if $thread eq '';
	return "\$lastsplat >= 0 ? \$lastsplat : die \"456 NO SPLAT\\n\"" if $int;
	return "\$lastsplat >= 0 ? pack('n', \$lastsplat) : die \"456 NO SPLAT\\n\"";
    } elsif ($op == E_NUMBER) {
	return 'die "275 ASSIGNMENT TO CONSTANT\\n";' if $lv;
	my $rp = _reg($thread, $e->[1]);
	"do { my \$rp = $rp; " .
	'$rp =~ s/^.//; ' .
	'$rp = $` if $rp =~ /\s/; ' .
	($int ? '$rp' : 'pack($rp < 0x10000 ? "n" : "N", $rp)') .
	' }';
    } elsif ($op == E_REGISTER ||
	     $op == E_SUBSCRIPT ||
	     $op == E_OWNER ||
	     $op == E_INDIRECT)
    {
	my $rp = _reg($thread, $e);
	$int ||= 0;
	if ($thread eq '') {
	    return "_assign(\\%regs, $rp, \@_);" if $lv;
	    return "_value(\\%regs, $rp, $int)";
	} else {
	    return "_qassign(\$regs$thread, \\%shregs, $rp, \@_);" if $lv;
	    return "_qvalue(\$regs$thread, \\%shregs, $rp, $int)";
	}
    } elsif ($op == E_AND) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO UNARY AND\\n" if @_ != 1;' .
	    '@_ = (i_unand($_[0]));' .
	    _expr($thread, $e->[1], 0, 1)
		if $lv;
	return $a0 . 'i_and(' . _expr($thread, $e->[1], 0, 0) . ')' . $a1;
    } elsif ($op == E_OR) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO UNARY OR\\n" if @_ != 1;' .
	    '@_ = (i_unor($_[0]));' .
	    _expr($thread, $e->[1], 0, 1)
		if $lv;
	return $a0 . 'i_or(' . _expr($thread, $e->[1], 0, 0) . ')' . $a1;
    } elsif ($op == E_XOR) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO UNARY XOR\\n" if @_ != 1;' .
	    '@_ = (i_unxor($_[0]));' .
	    _expr($thread, $e->[1], 0, 1)
		if $lv;
	return $a0 . 'i_xor(' . _expr($thread, $e->[1], 0, 0) . ')' . $a1;
    } elsif ($op == E_BAND) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO AND\\n" if @_ != 1;' .
	    '{' .
		'my ($item1, $item2) = i_unband($_[0]);' .
		'@_ = ($item1);' .
		_expr($thread, $e->[1], 0, 1) .
		'@_ = ($item2);' .
		_expr($thread, $e->[2], 0, 1) .
	    '}'
		if $lv;
	return $a0 . 'i_band(' . _expr($thread, $e->[1], 0, 0) . ', '
			       . _expr($thread, $e->[2], 0, 0) . ')' . $a1;
    } elsif ($op == E_BOR) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO OR\\n" if @_ != 1;' .
	    '{' .
		'my ($item1, $item2) = i_unbor($_[0]);' .
		'@_ = ($item1);' .
		_expr($thread, $e->[1], 0, 1) .
		'@_ = ($item2);' .
		_expr($thread, $e->[2], 0, 1) .
	    '}'
		if $lv;
	return $a0 . 'i_bor(' . _expr($thread, $e->[1], 0, 0) . ', '
			      . _expr($thread, $e->[2], 0, 0) . ')' . $a1;
    } elsif ($op == E_BXOR) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO XOR\\n" if @_ != 1;' .
	    '{' .
		'my ($item1, $item2) = i_unbxor($_[0]);' .
		'@_ = ($item1);' .
		_expr($thread, $e->[1], 0, 1) .
		'@_ = ($item2);' .
		_expr($thread, $e->[2], 0, 1) .
	    '}'
		if $lv;
	return $a0 . 'i_bxor(' . _expr($thread, $e->[1], 0, 0) . ', '
			       . _expr($thread, $e->[2], 0, 0) . ')' . $a1;
    } elsif ($op == E_INTERLEAVE) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO INTERLEAVE\\n" if @_ != 1;' .
	    '{' .
		'my ($item1, $item2) = i_uninterleave($_[0]);' .
		'@_ = ($item1);' .
		_expr($thread, $e->[1], 0, 1) .
		'@_ = ($item2);' .
		_expr($thread, $e->[2], 0, 1) .
	    '}'
		if $lv;
	return $a0 . 'i_interleave(' . _expr($thread, $e->[1], 0, 0) . ', '
				     . _expr($thread, $e->[2], 0, 0) . ')' . $a1;
    } elsif ($op == E_SELECT) {
	return
	    'die "241 CANNOT ASSIGN ARRAYS TO SELECT\\n" if @_ != 1;' .
	    '{' .
		'my ($item1, $item2) = i_unselect($_[0]);' .
		'@_ = ($item1);' .
		_expr($thread, $e->[1], 0, 1) .
		'@_ = ($item2);' .
		_expr($thread, $e->[2], 0, 1) .
	    '}'
		if $lv;
	return $a0 . 'i_select(' . _expr($thread, $e->[1], 0, 0) . ', '
			         . _expr($thread, $e->[2], 0, 0) . ')' . $a1;
    } elsif ($op == E_OVERLOAD_REGISTER) {
	return _expr($thread, $e->[1], 0, 1) if $lv;
	my $reg = _reg($thread, $e->[1]);
	my $val = _expr($thread, $e->[2], 0, 0);
	my $set = _expr($thread, $e->[2], 0, 1);
	my @c = ();
	push @c, "my \$rp = $reg; ";
	push @c, "_qimport(\$regs$thread, \\%shregs, \$rp, 1); " if $thread;
	push @c, "if (exists \$regs{\$rp}) { " if ! $thread;
	push @c,     "\$regs$thread\{\$rp}[4] = [ sub { $val }, sub { $set } ] if \$regs$thread\{\$rp}[1]; ";
	push @c, "} else { " if ! $thread;
	push @c,     "\$regs$thread\{\$rp} = [0, 1, [], 0, [ sub { $val }, sub { $set } ], [], []]; " if ! $thread;
	push @c, "} " if ! $thread;
	if ($int) {
	    push @c, "atoi(\$regs$thread\{\$rp}[0])";
	} else {
	    push @c, "\$regs$thread\{\$rp}[0]";
	}
	return join('', 'do { ', @c, ' }');
    } elsif ($op == E_OVERLOAD_RANGE) {
	return _expr($thread, $e->[1], 0, 1) if $lv;
	my $range = _expr($thread, $e->[1], 0, 0);
	my $val = _expr($thread, $e->[2], 0, 0);
	my $set = _expr($thread, $e->[2], 0, 1);
	my @c = ();
	push @c, "my \$range = $range; ";
	push @c, 'my ($start, $end) = i_uninterleave($range); ';
	push @c, '$start = atoi($start); ';
	push @c, '$end = atoi($end); ';
	push @c, 'while ($start <= $end) {';
	push @c,     'my $rp = ".$start"; ';
	push @c,     "_qimport(\$regs$thread, \\%shregs, \$rp, 1); " if $thread;
	push @c,     "if (exists \$regs{\$rp}) { " if ! $thread;
	push @c,         "\$regs$thread\{\$rp}[4] = [ sub { $val }, sub { $set } ] if \$regs$thread\{\$rp}[1]; ";
	push @c,     "} else { " if ! $thread;
	push @c,         "\$regs$thread\{\$rp} = [0, 1, [], 0, [ sub { $val }, sub { $set } ], [], []]; " if ! $thread;
	push @c,     "} " if ! $thread;
	push @c,     '$rp = ":$start"; ';
	push @c,     "_qimport(\$regs$thread, \\%shregs, \$rp, 1); " if $thread;
	push @c,     "if (exists \$regs{\$rp}) { " if ! $thread;
	push @c,         "\$regs$thread\{\$rp}[4] = [ sub { $val }, sub { $set } ] if \$regs$thread\{\$rp}[1]; ";
	push @c,     "} else { " if ! $thread;
	push @c,         "\$regs$thread\{\$rp} = [0, 1, [], 0, [ sub { $val }, sub { $set } ], [], []]; " if ! $thread;
	push @c,     "} " if ! $thread;
	push @c,     '$start ++; ';
	push @c, "}";
	if ($int) {
	    push @c, "atoi(\$range)";
	} else {
	    push @c, "\$range";
	}
	return join('', 'do { ', @c, ' }');
    } else {
	die "Internal compiler error (opcode = $op in function 52)\n";
    }
}

sub _declarations {
    my ($ptree, $quantum, $post, $dbhook, $reinstate) = @_;
    my $r = $ptree->{'toggle'}{'roman'};
    my $n = $ptree->{'toggle'}{'next'};
    my $a = $ptree->{'toggle'}{'arrayio'};
    my $i = $a eq 'C' ? 0 : $ptree->{'toggle'}{'io'};
    my $o = $a eq 'C' ? 0 : $ptree->{'toggle'}{'io'};
    my @c = (
	['$input' => '@_ > 0 ? $_[0] : \*STDIN'],
	['$output' => '@_ > 1 ? $_[1] : \*STDOUT'],
	['$flags' => '@_ > 2 ? $_[2] : {}'],
	['$roman' => "exists \$flags->{'roman'} ? \$flags->{'roman'} : $r"],
	['$inval' => "exists \$flags->{'io'} ? \$flags->{'io'} : $i"],
	['$outval' => "exists \$flags->{'io'} ? \$flags->{'io'} : $o"],
	['$arrayio' => "exists \$flags->{'arrayio'} ? \$flags->{'arrayio'} : '$a'"],
	['$next_ok' => "exists \$flags->{'next'} ? \$flags->{'next'} : $n"],
	['%lectures' => '()'],
	['@come_from' => '()'],
	['$lecture' => 0],
	['$label' => 0],
	['$rlabel' => 0],
	['$class' => 0],
    );
    my @use = ();
    if ($quantum) {
	push @c,
	    ['@regs' => '({})'],
	    ['@loop_bc' => '([0, 0])'],
	    ['@loop_cb' => '([1, 0])'],
	    ['@lecture_stack' => '([])'],
	    ['@next_stack' => '([])'],
	    ['@loops' => '([])'],
	    ['@doing_loops' => '(1)'],
	    ['@loop' => '(0)'],
	    ['@lret' => '([])'],
	    ['@eret' => '([])'],
	    ['@events' => '([])'],
	    ['$thread' => 0],
	    ['$new_thread' => 0],
	    ['@splats' => '()'],
	    ['$lastsplat' => -1],
	    ['@program_counter' => '(0)'],
	    ['%shregs' => '()'],
	    ['@reinstate' => '([' . join(', ', map {"[$_, 0]"} @$reinstate) . ']);'];
	push @use,
	    'Language::INTERCAL::Runtime::Library',
	    'Language::INTERCAL::Runtime::QuantumLibrary';
    } else {
	push @c,
	    ['%regs' => '()'],
	    ['$loop_bc' => 0],
	    ['$loop_cb' => 1],
	    ['@lecture_stack' => '()'],
	    ['@next_stack' => '()'],
	    ['@loops' => '()'],
	    ['$doing_loops' => 1],
	    ['@lret' => '()'],
	    ['$loop' => 0],
	    ['@eret' => '()'],
	    ['@events' => '()'],
	    ['@reinstate' => '(' . join(', ', @$reinstate) . ');'];
	push @use, 'Language::INTERCAL::Runtime::Library';
    }
    if ($post) {
	push @c,
	    ['$reinstate' => 0],
	    ['$opcode' => 0],
	    ['$clabel' => 0],
	    ['$new_value' => 0],
	    ['%opcodes' => '()'],
	    ['@opcodes' => '(' . ($quantum ? '[]' : '') . ')'];
	my @o = _opcodes($quantum);
	for (my $o = 0; $o < @o; $o++) {
	    next unless $o[$o];
	    push @c, ['$opcodes[' . ($quantum ? '0][' : '') . $o . ']',
		      "[$o, 0, $o[$o]]"];
	}
    }
    (\@use, \@c);
}

sub _opcodes {
    my ($quantum) = @_;

    my @o = (0) x @BC_NAMES;

    my $qenter = '_qenter($thread, \\@regs, \\@reinstate, \\@program_counter, '
	       . '$_[2], \\@loop_bc, \\@loop_cb, \\@lecture_stack, '
	       . '\\@next_stack, \\@loops, \\@loop, \\@doing_loops, '
	       . '\\@eret, \\@events, \\@opcodes);';

    my $Qenter = '_qenter($thread, \\@regs, \\@reinstate, \\@program_counter, '
	       . '$_[0], \\@loop_bc, \\@loop_cb, \\@lecture_stack, '
	       . '\\@next_stack, \\@loops, \\@loop, \\@doing_loops, '
	       . '\\@eret, \\@events, \\@opcodes);';

    $o[S_COME] = 'sub {}';
    $o[S_FORGET] = 'sub {}';
    $o[S_RESUME] = 'sub {}';
    $o[S_GABSTAIN] = 'sub {}';
    $o[S_GREINSTATE] = 'sub {}';
    $o[S_GFORK] = 'sub {}';
    if ($quantum) {
	$o[S_NEXT] = 'sub { die "401 YOUR PROGRAM IS OBSOLETE\\n" if ! $next_ok; unshift @{$next_stack[$thread]}, $_[2]; goto $_[1] }';
	$o[S_ABSTAIN] = 'sub { $reinstate[$thread][$_[0]][0] = 0 if $_[0] >= 0 }';
	$o[S_REINSTATE] = 'sub { $reinstate[$thread][$_[0]][0] = 1 if $_[0] >= 0 }';
	$o[S_FORK] = 'sub { '
		   .     'my $new_thread = ' . $qenter
		   .     'if ($_[0] >= 0) { '
		   .         '$reinstate[$thread][$_[0]] = [1, 1]; '
		   .         '$reinstate[$new_thread][$_[0]] = [0, 1]; '
		   .     '} '
		   . '}';
	$o[S_ASSIGN] = 'sub { shift; my $rp = shift; _qassign($regs[$thread], \\%shregs, $rp, @_ ) }';
	$o[S_ENROL] = 'sub { '
		    .     'shift; '
		    .     'my $rp = shift; '
		    .     '$class = _enrol(\\%lectures, map { atoi($_) } @_); '
		    .     'if (exists $regs[$thread]{$rp}) { '
		    .         'push @{$regs[$thread]{$rp}[6]}, $class if $regs[$thread]{$rp}[1]; '
		    .     '} elsif (exists $shregs{$rp}) {'
		    .         '$regs[$thread]{$rp} = $shregs{$rp}; '
		    .         'push @{$regs[$thread]{$rp}[6]}, $class if $regs[$thread]{$rp}[1]; '
		    .     '} else {'
		    .         '$regs[$thread]{$rp} = $shregs{$rp} = [0, 1, [], 0, 0, [], [$class]]; '
		    .     '}'
		    . '}';
	$o[S_LEARN] = 'sub { goto (_qlearns($regs[$thread], \\%shregs, $_[1], \\%lectures, atoi($_[2]), $lecture_stack[$thread], $_[0])) }';
	$o[S_STOP] = 'sub { if ($_[0]) { goto $_[0] } else { die "\\n" } }';
	$o[S_FINISH] = 'sub { _qfinish($regs[$thread], \\%shregs, $lecture_stack[$thread]) }';
	$o[S_WRITE] = 'sub { shift; _qwrite($regs[$thread], \\%shregs, $input, \\$inval, $arrayio, @_); }';
	$o[S_READ] = 'sub { shift; _qread($regs[$thread], \\%shregs, $output, $roman, \\$outval, $arrayio, @_); }';
	$o[S_GRADUATES] = 'sub { '
			.     'my $rp = shift; '
			.     'if (exists $regs[$thread]{$rp}) { '
			.         '$regs[$thread]{$rp}[6] = [] if $regs[$thread]{$rp}[1]; '
			.     '} elsif (exists $shregs{$rp}) {'
			.         '$regs[$thread]{$rp} = $shregs{$rp}; '
			.         '$regs[$thread]{$rp}[6] = [] if $regs[$thread]{$rp}[1]; '
			.     '}'
			. '}';
	$o[S_IGNORE] = 'sub { '
		     .     'shift; '
		     .     'while (@_) { '
		     .         'my $rp = shift; '
		     .         'if (exists $regs[$thread]{$rp}) { '
		     .             '$regs[$thread]{$rp}[1] = 0; '
		     .         '} elsif (exists $shregs{$rp}) {'
		     .             '$regs[$thread]{$rp} = $shregs{$rp}; '
		     .             '$regs[$thread]{$rp}[1] = 0; '
		     .         '} else { '
		     .             '$regs[$thread]{$rp} = $shregs{$rp} = [0, 0, [], 0, 0, [], []]; '
		     .         '}'
		     .     '}'
		     . '}';
	$o[S_REMEMBER] = 'sub { '
		       .     'shift; '
		       .     'while (@_) { '
		       .         'my $rp = shift; '
		       .         'if (exists $regs[$thread]{$rp}) { '
		       .             '$regs[$thread]{$rp}[1] = 1; '
		       .         '} elsif (exists $shregs{$rp}) {'
		       .             '$regs[$thread]{$rp} = $shregs{$rp}; '
		       .             '$regs[$thread]{$rp}[1] = 1; '
		       .         '} else { '
		       .             '$regs[$thread]{$rp} = $shregs{$rp} = [0, 1, [], 0, 0, [], []]; '
		       .         '}'
		       .     '}'
		       . '}';
	$o[S_VFORK] = 'sub { '
		    .     'my $new_thread = ' . $Qenter
		    .     'shift; '
		    .     'while (@_) { '
		    .         'my $rp = shift; '
		    .         '_qimport($regs[$thread], \\%shregs, $rp, 1); '
		    .         '$regs[$thread]{$rp} = _clone_array($regs[$thread]{$rp}); '
		    .         '$regs[$thread]{$rp}[3] = 1; '
		    .         '$regs[$new_thread]{$rp} = _clone_array($regs[$thread]{$rp}); '
		    .         '$regs[$thread]{$rp}[1] = 1; '
		    .         '$regs[$new_thread]{$rp}[1] = 0; '
		    .     '}'
		    . '}';
	$o[S_STASH] = 'sub { shift; while (@_) { _qstash($regs[$thread], \\%shregs, shift @_) } }';
	$o[S_RETRIEVE] = 'sub { shift; while (@_) { _qretrieve($regs[$thread], \\%shregs, shift @_) } }';
	$o[S_FREE] = 'sub { _qfree($regs[$thread], \\%shregs, $_[0], $_[1]) }';
	$o[S_ENSLAVE] = 'sub { '
		      .     'if (exists $regs[$thread]{$_[0]}) { '
		      .         'unshift @{$regs[$thread]{$_[0]}[2]}, $_[1] if $regs[$thread]{$_[0]}[1]; '
		      .     '} elsif (exists $shregs{$_[0]}) { '
		      .         '$regs[$thread]{$_[0]} = $shregs{$_[0]}; '
		      .         'unshift @{$regs[$thread]{$_[0]}[2]}, $_[1] if $regs[$thread]{$_[0]}[1]; '
		      .     '} else {'
		      .         '$regs[$thread]{$_[0]} = $shregs{$_[0]} = [0, 1, [$_[1]], 0, 0, [], []]; '
		      .     '}'
		      . '}';
	$o[S_CONVERT] = 'sub { '
		      .     '$opcodes[$thread][$_[1]][0] = $opcodes[$thread][$_[2]][0]; '
		      .     '$opcodes[$thread][$_[1]][1] = $opcodes[$thread][$_[2]][1]; '
		      .     '$opcodes[$thread][$_[1]][2] = $opcodes[$thread][$_[2]][2]; '
		      . '}';
	$o[S_CFORK] = 'sub { '
		    .     $Qenter . ' '
		    .     '$opcodes[$thread][$_[1]][0] = $opcodes[$thread][$_[2]][0]; '
		    .     '$opcodes[$thread][$_[1]][1] = $opcodes[$thread][$_[2]][1]; '
		    .     '$opcodes[$thread][$_[1]][2] = $opcodes[$thread][$_[2]][2]; '
		    . '}';
	$o[S_SWAP] = 'sub { '
		   .     '($opcodes[$thread][$_[1]][0], $opcodes[$thread][$_[2]][0]) = ($opcodes[$thread][$_[2]][0], $opcodes[$thread][$_[1]][0]); '
		   .     '($opcodes[$thread][$_[1]][1], $opcodes[$thread][$_[2]][1]) = ($opcodes[$thread][$_[2]][1], $opcodes[$thread][$_[1]][1]); '
		   .     '($opcodes[$thread][$_[1]][2], $opcodes[$thread][$_[2]][2]) = ($opcodes[$thread][$_[2]][2], $opcodes[$thread][$_[1]][2]); '
		   . '}';
	$o[S_SFORK] = 'sub { '
		    .     $Qenter . ' '
		    .     '($opcodes[$thread][$_[1]][0], $opcodes[$thread][$_[2]][0]) = ($opcodes[$thread][$_[2]][0], $opcodes[$thread][$_[1]][0]); '
		    .     '($opcodes[$thread][$_[1]][1], $opcodes[$thread][$_[2]][1]) = ($opcodes[$thread][$_[2]][1], $opcodes[$thread][$_[1]][1]); '
		    .     '($opcodes[$thread][$_[1]][2], $opcodes[$thread][$_[2]][2]) = ($opcodes[$thread][$_[2]][2], $opcodes[$thread][$_[1]][2]); '
		    . '}';
    } else {
	$o[S_NEXT] = 'sub { die "401 YOUR PROGRAM IS OBSOLETE\\n" if ! $next_ok; unshift @next_stack, $_[2]; goto $_[1] }';
	$o[S_ABSTAIN] = 'sub { $reinstate[$_[0]] = 0 if $_[0] >= 0 }';
	$o[S_REINSTATE] = 'sub { $reinstate[$_[0]] = 1 if $_[0] >= 0 }';
	$o[S_ASSIGN] = 'sub { shift; my $rp = shift; _assign(\\%regs, $rp, @_ ) }';
	$o[S_ENROL] = 'sub { '
		    .     'shift; '
		    .     'my $rp = shift; '
		    .     '$class = _enrol(\\%lectures, map { atoi($_) } @_); '
		    .     'if (exists $regs{$rp}) { '
		    .         'push @{$regs{$rp}[6]}, $class if $regs{$rp}[1]; '
		    .     '} else {'
		    .         '$regs{$rp} = [0, 1, [], 0, 0, [], [$class]]; '
		    .     '}'
		    . '}';
	$o[S_LEARN] = 'sub { goto (_learns(\\%regs, $_[1], \\%lectures, atoi($_[2]), \\@lecture_stack, $_[0])) }';
	$o[S_STOP] = 'sub { goto $_[0] }';
	$o[S_FINISH] = 'sub { _finish(\\%regs, \\@lecture_stack) }';
	$o[S_WRITE] = 'sub { shift; _write(\\%regs, $input, \\$inval, $arrayio, @_); }';
	$o[S_READ] = 'sub { shift; _read(\\%regs, $output, $roman, \\$outval, $arrayio, @_); }';
	$o[S_GRADUATES] = 'sub { '
			.     'my $rp = shift; '
			.     'if (exists $regs{$rp}) { '
			.         '$regs{$rp}[6] = [] if $regs{$rp}[1]; '
			.     '}'
			. '}';
	$o[S_IGNORE] = 'sub { '
		     .     'shift; '
		     .     'while (@_) { '
		     .         'my $rp = shift; '
		     .         'if (exists $regs{$rp}) { '
		     .             '$regs{$rp}[1] = 0; '
		     .         '} else { '
		     .             '$regs{$rp} = [0, 0, [], 0, 0, [], []]; '
		     .         '}'
		     .     '}'
		     . '}';
	$o[S_REMEMBER] = 'sub { '
		       .     'shift; '
		       .     'while (@_) { '
		       .         'my $rp = shift; '
		       .         'if (exists $regs{$rp}) { '
		       .             '$regs{$rp}[1] = 1; '
		       .         '} else { '
		       .             '$regs{$rp} = [0, 1, [], 0, 0, [], []]; '
		       .         '}'
		       .     '}'
		       . '}';
	$o[S_STASH] = 'sub { shift; while (@_) { _stash(\\%regs, shift @_) } }';
	$o[S_RETRIEVE] = 'sub { shift; while (@_) { _retrieve(\\%regs, shift @_) } }';
	$o[S_ENSLAVE] = 'sub { '
		      .     'if (exists $regs{$_[0]}) { '
		      .         'unshift @{$regs{$_[0]}[2]}, $_[1] if $regs{$_[0]}[1]; '
		      .     '} else {'
		      .         '$regs{$_[0]} = [0, 1, [$_[1]], 0, 0, [], []]; '
		      .     '}'
		      . '}';
	$o[S_FREE] = 'sub { _free(\\%regs, $_[0], $_[1]) }';
	$o[S_CONVERT] = 'sub { $opcodes[$_[1]] = $opcodes[$_[2]] }';
	$o[S_SWAP] = 'sub { ($opcodes[$_[1]], $opcodes[$_[2]]) = ($opcodes[$_[2]], $opcodes[$_[1]]) }';
    }

    pop @o while @o && ! $o[-1];

    @o;
}

sub _default_init {
    my ($prog_name, $intercal, $gotos) = @_;

    my $perl = $Config::Config{'perlpath'};

    my $g = $gotos > 10 ? " (AND CONTAINS $gotos GOTOS)" : '';

    '#!' . $perl . '

# THIS PROGRAM GENERATED BY ' . $intercal . '
# WARNING! THIS CODE IS NOT FOR THE FAINT OF HEART' . $g . '

eval \'exec ' . $perl . ' -S $0 ${1+"@"}\'
	if 0; # running under some shell

use Language::INTERCAL::Runtime::Library "i_initialise";

';
}

sub _default_pod {
    my ($prog_name, $intercal) = @_;
    '

__END__

=pod

=head1 NAME

' . $prog_name . ' - a program generated by ' . $intercal . '

=head1 SYNOPSIS

B<' . $prog_name . '> [options] B<files>...\n";

=head1 DESCRIPTION

This is just a description of the command-line options, as the compiler has
no way to figure out what the program does (can I<you> figure out what an
INTERCAL program does just by looking at it?). The programmer can change
this paragraph if this is a problem.

=over 4

=item B<-A>

Produce output in ASCII (default).

=item B<-B>

Produce output in Baudot.

=item B<-E>

Produce output in EBCDIC.

=item B<-a>

Accept input in ASCII.

=item B<-b>

Accept input in Baudot.

=item B<-e>

Accept input in EBCDIC (default).

=item B<-o> I<name>

Internally redirects the standard output to the named file.

=back

=head1 SEE ALSO

L<Language::INTERCAL>, and, most importantly, a qualified psychiatrist.

';
}

1;

__END__

=head1 NAME

Language::INTERCAL::Backend::PerlText - Perl back end for CLC-INTERCAL

=head1 SYNOPSIS

    use Language::INTERCAL;

    my $program = compile Language::INTERCAL 'program text';

    $program->backend('PerlText', 'fun.pl');

    system 'perl', 'fun.pl';

=head1 DESCRIPTION

I<Language::INTERCAL::Backend::PerlText> contains a back end for
I<Language::INTERCAL> which creates a Perl script.

The back end is normally invoked using the method I<backend> of package
I<Language::INTERCAL>, with the string 'PerlText' as first argument and
the name of the file to be created as second argument.

The program produced by this back end will contain the same Perl code as
internally generated by the 'Perl' back end (see
L<Language::INTERCAL::Backend::Perl>),
with a wrapper to parse some command-line options and fiddle with the input
and output alphabet. See the POD in the generated file for details.

=head1 COPYRIGHT

This module is part of CLC-INTERCAL.

Copyright (c) 1999 by Claudio Calvelli E<lt>C<lunatic@assurdo.com>E<gt>,
all (f)rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

=head1 SEE ALSO

A qualified psychiatrist.

