/* Authors:  Peter Holst Andersen (txix@diku.dk)
 *           Jens Peter Secher (jpsecher@diku.dk)
 *           Henning Makholm (makholm@diku.dk)
 * Content:  C-Mix system: C Parser
 *
 * Copyright  1999. The TOPPS group at DIKU, U of Copenhagen.
 * Redistribution and modification are allowed under certain
 * terms; see the file COPYING.cmix for details.
 */

/* Copyright (C) 1989,1990 James A. Roskind, All rights reserved.
    This grammar was developed  and  written  by  James  A.  Roskind.
    ...
    later the C-Mix folks FUBARed it
*/

%{

#include <cmixconf.h>
// magic incantation from the autoconf manual
#ifdef __GNUC__
#  define alloca __builtin_alloca
#else
#  if HAVE_ALLOCA_H
#    include<alloca.h>
#  else
#    ifdef _AIX
#      pragma alloca
#    else
#      ifndef alloca /* predefined by HP cc +Olibcalls */
         char *alloca();
#      endif
#    endif
#  endif
#endif

#include "Plist.h"
#include "symboltable.h"
#include "syntax.h"
#include "cpgm.h"
#include "liststack.h"
#include "directives.h"
#include "auxilary.h"
#include "options.h"
#include "parser.h"

    static void yyerror(const char*message) {
        Diagnostic(ERROR,ccchere) << message;
    }

#define YYERROR_VERBOSE

#define YYDEBUG 1

#ifdef YYDEBUG
    static void settheflag();
#else
    #define settheflag()
#endif

// trick bison into producing a member function rather
// than a freestanding C function.
#define cccparse(x) cccParser::ParseNow(x)

extern SymbolTable<Indirection> goto_labels;	// Goto labels
extern Scope<ObjectDecl> names;                 // Names
extern Scope<UserDecl> usernames;               // Names of structs/unions/enums.
extern ListStack<VarDecl> objects;             // List of real objects.
extern ListStack<UserDecl> userdecls;          // Stack of user declarations;

%}

%expect 1

%union{
    // Temporaries and switchboards (syntax.h)
    Parse_Type* pt;
    Parse_UserType* utype;
    UserTag uttag;
    Parse_TypedefType* tdtype;
    Plist<Parse_Postfix>* postfix;
    Parse_IdPostfix* idpost;
    Parse_MemberId* mempost;
    Parse_Decl* decl;
    Plist<Parse_Decl>* decls;
    char* str;
    // Permanent structures/classes (cpgm.h)
    FunDef* fundef;
    Type* type;
    Expr* expr;
    ConstExpr* cexpr;
    Stmt* stmt;
    Init* init;
    Plist<VarDecl>* vars;
    Plist<Expr>* exprs;
    Plist<MemberDecl>* membs;
    Plist<Type>* types;
    Plist<Stmt>* stmts;
    CProgram* cpgm;
    UnOp unop;
    AssignOp asgnop;
}

/* Define terminal tokens */

/* New Lexical element, whereas ANSI suggested non-terminal */

%token AUTO             DOUBLE          INT             STRUCT
%token BREAK            ELSE            LONG            SWITCH
%token CASE             ENUM            REGISTER        TYPEDEF
%token CHAR             EXTERN          RETURN          UNION
%token CONST            FLOAT           SHORT           UNSIGNED
%token CONTINUE         FOR             SIGNED          VOID
%token DEFAULT          GOTO            SIZEOF          VOLATILE
%token DO               IF              STATIC          WHILE

%token FLOATINGconstant         INTEGERconstant         CHARACTERconstant
%token LONGconstant             UINTEGERconstant        STRINGliteral
%token ULONGconstant            DOUBLEconstant

/* Multi-Character operators */
%token  ARROW                 /*    ->                              */
%token  ICR DECR              /*    ++      --                      */
%token  LEFTSHIFT RIGHTSHIFT  /*    <<      >>                      */
%token  LESSEQUAL GREATEQUAL  /*    <=      >=                      */
%token  EQUAL NOTEQUAL        /*    <=      >=      ==      !=      */
%token  ANDAND OROR           /*    &&      ||                      */
%token  ELLIPSIS              /*    ...                             */

/* modifying assignment operators */
%token MULTassign  DIVassign    MODassign   /*   *=      /=      %=      */
%token PLUSassign  MINUSassign              /*   +=      -=              */
%token LSassign    RSassign                 /*   <<=     >>=             */
%token ANDassign   ERassign     ORassign    /*   &=      ^=      |=      */

%token <str> IDENTIFIER
%token <str> TYPEDEFname /* Lexer will tell the difference between this and
    an  identifier!   An  identifier  that is CURRENTLY in scope as a
    typedef name is provided to the parser as a TYPEDEFname.*/

%token <str> CMIXTAG

/*************************************************************************/

%type <expr> constant_expression assignment_expression

%type <uttag> aggregate_key

%type <pt> declaration_qualifier_list type_qualifier_list
%type <pt> declaration_qualifier type_qualifier basic_declaration_specifier
%type <pt> basic_type_name storage_class basic_type_specifier

%type <decl> declaration_specifier
%type <utype> sue_type_specifier sue_declaration_specifier
%type <utype> elaborated_type_name aggregate_name enum_name
%type <vars> enumerator_list
%type <expr> enumerator_value_opt
%type <type> type_specifier type_name
%type <decl> member_declaring_list member_default_declaring_list
%type <decl> declaring_list default_declaring_list
%type <type> parameter_declaration

%type <tdtype> typedef_type_specifier typedef_declaration_specifier

/*%type <membs> member_declaration_list member_declaration*/
%type <expr> bit_field_size_opt bit_field_size
%type <str> identifier_or_typedef_name member_name
%type <init> initializer initializer_list initializer_opt
%type <stmt> statement, jump_statement, compound_statement, compound_statement1
%type <stmt> labeled_statement
%type <stmt> expression_statement, selection_statement, iteration_statement
%type <stmts> statement_list

%type <postfix> postfix_abstract_declarator unary_abstract_declarator
%type <postfix> parameter_type_list abstract_declarator
%type <postfix> postfixing_abstract_declarator  array_abstract_declarator

%type <idpost> postfix_identifier_declarator unary_identifier_declarator
%type <idpost> identifier_declarator paren_postfix_typedef_declarator
%type <idpost> paren_typedef_declarator clean_postfix_typedef_declarator
%type <idpost> clean_typedef_declarator parameter_typedef_declarator
%type <idpost> typedef_declarator declarator simple_paren_typedef_declarator
%type <idpost> paren_identifier_declarator

%type <types> parameter_list

%type <mempost> member_identifier_declarator member_declarator
%type <idpost> old_function_declarator postfix_old_function_declarator

%type <str> FLOATINGconstant INTEGERconstant CHARACTERconstant
%type <str> DOUBLEconstant UINTEGERconstant STRINGliteral
%type <str> LONGconstant ULONGconstant

%type <cexpr> constant string_literal_list
%type <expr> primary_expression postfix_expression comma_expression_opt
%type <expr> unary_expression comma_expression
%type <unop> unary_operator
%type <asgnop> assignment_operator
%type <expr> conditional_expression logical_OR_expression
%type <expr> logical_AND_expression inclusive_OR_expression
%type <expr> exclusive_OR_expression AND_expression equality_expression
%type <expr> relational_expression shift_expression additive_expression
%type <expr> multiplicative_expression cast_expression
%type <exprs> argument_expression_list argument_list
/*************************************************************************/

%start start_symbol

/*************************************************************************/

%%

/************************** CONSTANTS *********************************/

constant:
        INTEGERconstant
          {
              $$=new ConstExpr($1,Integer,ccchere);
          }
        | UINTEGERconstant
          {
              $$=new ConstExpr($1,UInteger,ccchere);
          }
        | LONGconstant
          {
              $$=new ConstExpr($1,Long,ccchere);
          }
        | ULONGconstant
          {
              $$=new ConstExpr($1,ULong,ccchere);
          }
        | FLOATINGconstant
          {
              $$=new ConstExpr($1,Floating,ccchere);
          }
        | DOUBLEconstant
          {
              $$=new ConstExpr($1,DoubleF,ccchere);
          }
        | CHARACTERconstant
          {
              // Remember that the pings are still present.
              $$=new ConstExpr($1,Character,ccchere);
          }
        ;

string_literal_list:
	  STRINGliteral
          {
              // Remember that the quotes are still present.
              $$=new ConstExpr($1,StrConst,ccchere);
          }
        | string_literal_list STRINGliteral
          {
              // Concatenate the two strings by removing the " at
              // the end of the first string and the beginning of
              // the second one.
              int s1 = strlen($1->literal);
              int s2 = strlen($2);
              char* str = new char[s1+s2-1];
              strcpy(str,$1->literal);
              strcpy(str+s1-1,$2+1);
              $1->literal = str;
              $$ = $1;
          }
        ;

/************************* EXPRESSIONS ********************************/

primary_expression:
          IDENTIFIER
          {
              // Find the declaration for this identifier.
              $$=add_var_ref($1,ccchere);
          }
        | CMIXTAG ')' IDENTIFIER
          {
            VarExpr *e = add_var_ref($3,ccchere);
            e->annos = directives.AnnoFromString($1,ccchere);
            $$ = e ;
          }
        | constant			{ $$=$1; }
        | string_literal_list		{ $$=$1; }
        | '(' comma_expression ')'	{ $$=$2; }
        ;

argument_list:
        '(' ')'  { $$ = new Plist<Expr>(); }
        | '(' argument_expression_list ')' { $$ = $2 ; }
        ;

postfix_expression:
        primary_expression { $$=$1; }
        | postfix_expression '[' comma_expression ']'
          {
              $$=new ArrayExpr($1,$3,$1->pos);
          }
        | postfix_expression argument_list
          {
              $$=new CallExpr($1,$1->pos,$2);
          }
        | postfix_expression {} '.'   member_name
          {
              $$=new DotExpr($1,$4,$1->pos);
          }
        | postfix_expression {} ARROW member_name
          {
              // Convert "a->m" into "(*a).m"
              UnaryExpr* p = new UnaryExpr($1);
              $$=new DotExpr(p,$4,p->pos);
          }
        | postfix_expression ICR
          {
              $$=new PostExpr($1,Inc,$1->pos);
          }
        | postfix_expression DECR
          {
              $$=new PostExpr($1,Decr,$1->pos);
          }
        ;

member_name:
          IDENTIFIER
        | TYPEDEFname
        ;

argument_expression_list:
          assignment_expression
          {
              $$=new Plist<Expr>($1);
          }
        | argument_expression_list ',' assignment_expression
          {
              $1->push_back($3); $$=$1;
          }
        ;

unary_expression:
          postfix_expression { $$=$1; }
        | ICR unary_expression { $$=new PreExpr($2,Inc,$2->pos); }
        | DECR unary_expression { $$=new PreExpr($2,Decr,$2->pos); }
        | unary_operator cast_expression
          {
              $$=new UnaryExpr($2,$1,$2->pos);
          }
        | SIZEOF unary_expression
          {
              $$=new ExprSize($2,$2->pos)
          }
        | SIZEOF '(' type_name ')'
          {
              $$=new TypeSize($3,ccchere)
          }
        ;

unary_operator:
        '&' { $$=Addr; }
        | '*' { $$=DeRef; }
        | '+' { $$=Pos; }
        | '-' { $$=Neg; }
        | '~' { $$=Not; }
        | '!' { $$=Bang; }
        ;

cast_expression:
        unary_expression
        | '(' type_name ')' cast_expression
        {
            $$=new CastExpr($2,$4,ccchere);
        }
        ;

multiplicative_expression:
          cast_expression
        | multiplicative_expression '*' cast_expression
          {
              $$=new BinaryExpr($1,$3,Mul,$1->pos);
          }
        | multiplicative_expression '/' cast_expression
          {
              $$=new BinaryExpr($1,$3,Div,$1->pos);
          }
        | multiplicative_expression '%' cast_expression
          {
              $$=new BinaryExpr($1,$3,Mod,$1->pos);
          }
        ;

additive_expression:
          multiplicative_expression
        | additive_expression '+' multiplicative_expression
          {
              $$=new BinaryExpr($1,$3,Add,$1->pos);
          }
        | additive_expression '-' multiplicative_expression
          {
              $$=new BinaryExpr($1,$3,Sub,$1->pos);
          }
        ;

shift_expression:
          additive_expression
        | shift_expression LEFTSHIFT additive_expression
          {
              $$=new BinaryExpr($1,$3,LShift,$1->pos);
          }
        | shift_expression RIGHTSHIFT additive_expression
          {
              $$=new BinaryExpr($1,$3,RShift,$1->pos);
          }
        ;

relational_expression:
          shift_expression
        | relational_expression '<' shift_expression
          {
              $$=new BinaryExpr($1,$3,LT,$1->pos);
          }
        | relational_expression '>' shift_expression
          {
              $$=new BinaryExpr($1,$3,GT,$1->pos);
          }
        | relational_expression LESSEQUAL shift_expression
          {
              $$=new BinaryExpr($1,$3,LEq,$1->pos);
          }
        | relational_expression GREATEQUAL shift_expression
          {
              $$=new BinaryExpr($1,$3,GEq,$1->pos);
          }
        ;

equality_expression:
          relational_expression
        | equality_expression EQUAL relational_expression
          {
              $$=new BinaryExpr($1,$3,Eq,$1->pos);
          }
        | equality_expression NOTEQUAL relational_expression
          {
              $$=new BinaryExpr($1,$3,NEq,$1->pos);
          }
        ;

AND_expression:
          equality_expression
        | AND_expression '&' equality_expression
          {
              $$=new BinaryExpr($1,$3,BAnd,$1->pos);
          }
        ;

exclusive_OR_expression:
          AND_expression
        | exclusive_OR_expression '^' AND_expression
          {
              $$=new BinaryExpr($1,$3,BEOr,$1->pos);
          }
        ;

inclusive_OR_expression:
          exclusive_OR_expression
        | inclusive_OR_expression '|' exclusive_OR_expression
          {
              $$=new BinaryExpr($1,$3,BOr,$1->pos);
          }
        ;

logical_AND_expression:
          inclusive_OR_expression
        | logical_AND_expression ANDAND inclusive_OR_expression
          {
              $$=new BinaryExpr($1,$3,And,$1->pos);
          }
        ;

logical_OR_expression:
          logical_AND_expression
        | logical_OR_expression OROR logical_AND_expression
          {
              $$=new BinaryExpr($1,$3,Or,$1->pos);
          }
        ;

conditional_expression:
          logical_OR_expression
        | logical_OR_expression '?' comma_expression ':' conditional_expression
          {
              $$=new CondExpr($1,$3,$5,$1->pos);
          }
        ;

assignment_expression:
          conditional_expression
        | unary_expression assignment_operator assignment_expression
          {
              $$=new AssignExpr($1,$3,$2,$1->pos);
          }
        ;

assignment_operator:
          '=' { $$=Asgn; }
        | MULTassign { $$=MulAsgn; }
        | DIVassign { $$=DivAsgn; }
        | MODassign { $$=ModAsgn; }
        | PLUSassign { $$=AddAsgn; }
        | MINUSassign { $$=SubAsgn; }
        | LSassign { $$=LSAsgn; }
        | RSassign { $$=RSAsgn; }
        | ANDassign { $$=AndAsgn; }
        | ERassign { $$=EOrAsgn; }
        | ORassign { $$=OrAsgn; }
        ;

comma_expression:
          assignment_expression
        | comma_expression ',' assignment_expression
          { $$=new CommaExpr($1,$3,$1->pos); }
        ;

constant_expression:
          conditional_expression
        ;

comma_expression_opt:
          /* nothing */
          { $$=NULL; }
        | comma_expression
        ;


/******************************* DECLARATIONS *********************************/

    /* The following is different from the ANSI C specified  grammar.
    The  changes  were  made  to  disambiguate  typedef's presence in
    declaration_specifiers (vs.  in the declarator for redefinition);
    to allow struct/union/enum tag declarations without  declarators,
    and  to  better  reflect the parsing of declarations (declarators
    must be combined with declaration_specifiers ASAP  so  that  they
    are visible in scope).

    Example  of  typedef  use  as either a declaration_specifier or a
    declarator:

      typedef int T;
      struct S { T T;}; /* redefinition of T as member name * /

    Example of legal and illegal statements detected by this grammar:

      int; /* syntax error: vacuous declaration * /
      struct S;  /* no error: tag is defined or elaborated * /

    Example of result of proper declaration binding:

        int a=sizeof(a); /* note that "a" is declared with a type  in
            the name space BEFORE parsing the initializer * /

        int b, c[sizeof(b)]; /* Note that the first declarator "b" is
             declared  with  a  type  BEFORE the second declarator is
             parsed * /

    */

declaration:
        sue_declaration_specifier ';'
          {
	    // A lone structure declaration. It has to be inserted if
	    // it is not in local scope.
	    lone_sue($1);
	  }
        | sue_type_specifier ';'
          {
	    lone_sue($1);
	  }
        | declaring_list ';' {}
        | default_declaring_list ';' {}
        ;

    /* Note that if a typedef were  redeclared,  then  a  declaration
    specifier must be supplied */

default_declaring_list: /* decl */ /* Can't redeclare typedef names */
        declaration_qualifier_list identifier_declarator
          {
            // Make a declaration.
            Parse_Decl* tmplt = make_decl($1);
            // Make a copy of the base declaration template.
            Parse_Decl* decl = new Parse_Decl(tmplt);
            // Complete this declaration and put it into scope.
            decl->combine_with_postfix($2);
            add_declaration(decl);
            // Wrap up the template and declaration.
            //$<tdecl>$=new Parse_DeclTemplate(tmplt,var);
            $<decl>$ = tmplt;
          }
          initializer_opt
          {
              // Put the initializer in the declaration.
              add_initializer($4);
              $$=$<decl>3;
          }
        | type_qualifier_list identifier_declarator
          {
            // (Same as above.)
            Parse_Decl* tmplt = make_decl($1);
            Parse_Decl* decl = new Parse_Decl(tmplt);
            decl->combine_with_postfix($2);
            add_declaration(decl);
            $<decl>$=tmplt;
          }
          initializer_opt
          {
              add_initializer($4);
              $$=$<decl>3;
          }
        | default_declaring_list ',' identifier_declarator
          {
              // (Same as above.)
              Parse_Decl* decl = new Parse_Decl($1);
              decl->combine_with_postfix($3);
              add_declaration(decl);
              $<decl>$=$1;
          }
          initializer_opt
          {
              add_initializer($5);
              $$=$<decl>4;
          }
        ;

declaring_list: /* tdecl */
        declaration_specifier declarator
          {
              // (Same as above.)
              Parse_Decl* tmplt = $1;
              Parse_Decl* decl = new Parse_Decl(tmplt);
              decl->combine_with_postfix($2);
              add_declaration(decl);
              $<decl>$=tmplt;
          }
          initializer_opt
          {
              add_initializer($4);
              $$=$<decl>3;
          }
        | type_specifier declarator
          {
              // (Same as above.)
              Parse_Decl* tmplt = new Parse_Decl($1,P_NoStorageClass);
              Parse_Decl* decl = new Parse_Decl(tmplt);
              decl->combine_with_postfix($2);
              add_declaration(decl);
              $<decl>$=tmplt;
          }
          initializer_opt
          {
              add_initializer($4);
              $$=$<decl>3;
          }
        | declaring_list ',' declarator
          {
              // (Same as above.)
              Parse_Decl* decl = new Parse_Decl($1);
              decl->combine_with_postfix($3);
              add_declaration(decl);
              $<decl>$=$1
          }
          initializer_opt
          {
              add_initializer($5);
              $$=$<decl>4;
          }
        ;

declaration_specifier:  /* Parse_Decl*  */
        basic_declaration_specifier          /* Arithmetic or void */
          { $$=make_decl($1); }
        | sue_declaration_specifier          /* struct/union/enum */
          { $$=make_decl($1); }
        | typedef_declaration_specifier      /* typedef*/
          { $$ = make_decl($1); }
        ;

type_specifier:  /* Type*  */
        basic_type_specifier                 /* Arithmetic or void */
          { $$=make_basetype($1); }
        | sue_type_specifier                 /* Struct/Union/Enum */
          { $$=make_usertype($1); }
        | typedef_type_specifier             /* Typedef */
          { $$=make_typedeftype($1); }
        ;


declaration_qualifier_list:  /* Parse_Type */ /* const/volatile, AND storage class */
        storage_class
        | type_qualifier_list storage_class
          { $$=($1->combine($2)); }
        | declaration_qualifier_list declaration_qualifier
          { $$=($1->combine($2)); }
        ;

type_qualifier_list:  /* Parse_Type*  */
        type_qualifier
        | type_qualifier_list type_qualifier
          { $$=($1->combine($2)); }
        ;

declaration_qualifier:  /*  Parse_Type*  */
        storage_class
        | type_qualifier                  /* const or volatile */
        ;

type_qualifier:  /*  Parse_Type*  */
        CONST { ($$=new Parse_Type())->qualifiers=constvol(true,false); }
        | VOLATILE { ($$=new Parse_Type())->qualifiers=constvol(false,true); }
        ;

basic_declaration_specifier: /* Parse_Type*  */
                             /*Storage Class+Arithmetic or void*/
        declaration_qualifier_list basic_type_name
          { $$=($1->combine($2)); }
        | basic_type_specifier  storage_class
          { $$=($1->combine($2)); }
        | basic_declaration_specifier declaration_qualifier
          { $$=($1->combine($2)); }
        | basic_declaration_specifier basic_type_name
          { $$=($1->combine($2)); }
        ;

basic_type_specifier:
        basic_type_name            /* Arithmetic or void */
        | type_qualifier_list  basic_type_name
          { $$=$1->combine($2); }
        | basic_type_specifier type_qualifier
          { $$=$1->combine($2); }
        | basic_type_specifier basic_type_name
          { $$=$1->combine($2); }
        ;

sue_declaration_specifier: /* Parse_UserType  */
                           /* Storage Class + struct/union/enum */
        declaration_qualifier_list  elaborated_type_name
	  { $$=$2->combine($1); }
        | sue_type_specifier        storage_class
	  { $$=$1->combine($2); }
        | sue_declaration_specifier declaration_qualifier
	  { $$=$1->combine($2); }
        ;

sue_type_specifier:  /* Parse_UserType*  */
        elaborated_type_name              /* struct/union/enum */
        | type_qualifier_list elaborated_type_name
	  { $$=$2->combine($1); }
        | sue_type_specifier  type_qualifier
	  { $$=$1->combine($2); }
        ;


typedef_declaration_specifier: /* Parse_TypedefType* */ /*Storage Class + typedef types */
        typedef_type_specifier          storage_class
          { $$=$1->combine($2); }
        | declaration_qualifier_list    TYPEDEFname
          { $$=make_typedef($2)->combine($1); }
        | typedef_declaration_specifier declaration_qualifier
          { $$=$1->combine($2); }
        ;

typedef_type_specifier:  /* Parse_TypedefType* */  /* typedef types */
        TYPEDEFname
          { $$=make_typedef($1); }
        | type_qualifier_list    TYPEDEFname
          { $$=make_typedef($2)->combine($1); }
        | typedef_type_specifier type_qualifier
          { $$=$1->combine($2); }
        ;

storage_class:
        TYPEDEF         { $$=new Parse_Type(P_Typedef); }
        | EXTERN        { $$=new Parse_Type(P_Extern); }
        | STATIC        { $$=new Parse_Type(P_Static); }
        | AUTO          { $$=new Parse_Type(P_Auto); }
        | REGISTER      { $$=new Parse_Type(P_Register); }
        ;

basic_type_name:
        INT             { $$=new Parse_Type(P_Int); }
        | CHAR          { $$=new Parse_Type(P_Char); }
        | SHORT         { $$=new Parse_Type(P_Short); }
        | LONG          { $$=new Parse_Type(P_Long); }
        | FLOAT         { $$=new Parse_Type(P_Float); }
        | DOUBLE        { $$=new Parse_Type(P_Double); }
        | SIGNED        { $$=new Parse_Type(P_Signed); }
        | UNSIGNED      { $$=new Parse_Type(P_Unsigned); }
        | VOID          { $$=new Parse_Type(P_Void); }
        ;

elaborated_type_name:  /* Parse_UserType */
        aggregate_name
        | enum_name
        ;

aggregate_name:  /* Parse_UserType */
        aggregate_key '{' { members_enter_scope(); } member_declaration_list
        '}'
          {
	    /* Insert into the right symboltable. */
	    char const *aname = fresh_anonymous_name() ;
            anonymous_struct_decl($1,aname);
	    /* Make a temporary type for it. */
            $$ = new Parse_UserType(aname,$1);
            members_leave_scope();
          }
        | aggregate_key identifier_or_typedef_name
          {
              members_enter_scope();
	      // Reserve name (incomplete declaration).
              introduce_struct_decl($1, $2);
          }
          '{'  member_declaration_list '}'
          {
	      // Complete declaration.
              update_struct_decl($2);
              $$ = new Parse_UserType($2,$1);
              members_leave_scope();
          }
        | aggregate_key identifier_or_typedef_name
          {
              find_struct_decl($1, $2);
              $$ = new Parse_UserType($2,$1);
          }
        ;

aggregate_key:
        STRUCT          { $$=Struct; }
        | UNION         { $$=Union; }
        ;

/* Returns the list of member declations */
member_declaration_list: /* Plist<MemberDecl>* */
        member_declaration {}
        | member_declaration_list member_declaration
          { }
        ;

member_declaration: /* (nothing) */
          member_declaring_list ';' {}
        | member_default_declaring_list ';' {}
        ;

member_default_declaring_list: /* decl */ /* doesn't redeclare typedef */
        type_qualifier_list member_identifier_declarator /* must be base type */
          {
              // Use a template to generate the rest members from.
              Parse_Decl* tpl = make_decl($1);
              // Create the member declaration and put the name in scope.
              // All members have to be inserted under
              // the struct when the present member scope is finished.
              Parse_Decl* d = new Parse_Decl(tpl);
              d->combine_with_postfix($2->p_id);
              add_member(d,$2->bits);
              $$=tpl;
          }
        | member_default_declaring_list ',' member_identifier_declarator
          {
              // (see above)
              Parse_Decl* d = new Parse_Decl($1);
              d->combine_with_postfix($3->p_id);
              add_member(d,$3->bits);
              $$=$1;
          }
        ;

member_declaring_list:  /* decl  */
        type_specifier member_declarator
          {
              // (Same as member_default_declaring_list).
              Parse_Decl* tpl = new Parse_Decl($1);
              Parse_Decl* d = new Parse_Decl(tpl);
              d->combine_with_postfix($2->p_id);
              add_member(d,$2->bits);
              $$=tpl;
          }
        | member_declaring_list ',' member_declarator
          {
              // (Same as member_default_declaring_list).
              Parse_Decl* d = new Parse_Decl($1);
              d->combine_with_postfix($3->p_id);
              add_member(d,$3->bits);
              $$=$1;
          }
        ;

member_declarator:  /* Parse_MemberId*  */
        declarator bit_field_size_opt
            { $$=new Parse_MemberId($1,$2); }
        | bit_field_size
            { $$=new Parse_MemberId(NULL,$1); }
        ;

member_identifier_declarator:  /* Parse_MemberId*  */
        identifier_declarator bit_field_size_opt
            { $$=new Parse_MemberId($1,$2); }
        | bit_field_size
            { $$=new Parse_MemberId(NULL,$1); }
        ;

bit_field_size_opt:  /* Expr* */
        /* nothing */
          { $$ = NULL; }
        | bit_field_size
        ;

bit_field_size:  /* Expr* */
        ':' constant_expression
          { $$ = $2; }
        ;

enum_name:  /* Parse_UserType  */
        ENUM '{' enumerator_list '}'
          {
	    // Insert it into the right symboltable.
            char const *aname = fresh_anonymous_name() ;
	    add_enum_decl(aname);
	    // Make a temporary type for it.
            $$ = new Parse_UserType(aname,Enum);
          }
        | ENUM identifier_or_typedef_name '{' enumerator_list '}'
          {
	    add_enum_decl($2);
            $$ = new Parse_UserType($2,Enum);
          }
        | ENUM identifier_or_typedef_name
          {
	    find_enum_decl($2);
            $$ = new Parse_UserType($2,Enum);
          }
        ;

enumerator_list:  /* (nothing) */
        identifier_or_typedef_name enumerator_value_opt
          {
            add_enumconst($1,ccchere,$2);
          }
        | enumerator_list ',' identifier_or_typedef_name enumerator_value_opt
          {
            add_enumconst($3,ccchere,$4);
          }
        ;

enumerator_value_opt:   /* Init* */
        /* Nothing */
          { $$ = NULL; }
        | '=' constant_expression
          { $$ = $2; }
        ;

// This is the full parameter list. Types are extracted later.
parameter_type_list:  /* Parse_Postfix* */
        parameter_list
          // Make a function pre-type.
          {
            if ( $1->empty() )
              $1->push_back(new EllipsisType());
            Parse_Postfix* pf = new Parse_Postfix($1,0);
            $$=new Plist<Parse_Postfix>(pf);
          }
        | parameter_list ',' ELLIPSIS
          // Make a function pre-type (signal varargs).
          {
              $1->push_back(new EllipsisType());
              Parse_Postfix* pf = new Parse_Postfix($1,1);
              $$=new Plist<Parse_Postfix>(pf);
          }
        ;

parameter_list: /* types  */
        parameter_declaration
          { $$=new Plist<Type>($1); }
        | parameter_list ',' parameter_declaration
          { $1->push_back($3); $$=$1; }
        ;

parameter_declaration:  /* Type*  */
        declaration_specifier
          { $$=$1->type; }
        | declaration_specifier abstract_declarator
	  { $1->combine_with_postfix($2); $$=$1->type; }
        | declaration_specifier identifier_declarator
          { $1->combine_with_postfix($2);
            add_declaration($1);
            $$=$1->type; }
        | declaration_specifier parameter_typedef_declarator
	  { $1->combine_with_postfix($2); $$=$1->type; }
        | declaration_qualifier_list
          { $$=(make_decl($1)->type); }
        | declaration_qualifier_list abstract_declarator
          { $$=((make_decl($1)->combine_with_postfix($2))->type); }
        | declaration_qualifier_list identifier_declarator
          { Parse_Decl *d = make_decl($1)->combine_with_postfix($2);
            add_declaration(d);
            $$=d->type;
          }
        | type_specifier
          { $$=(new Parse_Decl($1))->type; }
        | type_specifier abstract_declarator
          { $$=((new Parse_Decl($1))->combine_with_postfix($2))->type; }
        | type_specifier identifier_declarator
          { Parse_Decl *d = (new Parse_Decl($1))->combine_with_postfix($2);
            add_declaration(d);
            $$=d->type;
          }
        | type_specifier parameter_typedef_declarator
          { $$=((new Parse_Decl($1))->combine_with_postfix($2))->type; }
        | type_qualifier_list
          { $$=make_decl($1)->type; }
        | type_qualifier_list abstract_declarator
          { $$=(make_decl($1)->combine_with_postfix($2))->type; }
        | type_qualifier_list identifier_declarator
          { Parse_Decl *d = make_decl($1)->combine_with_postfix($2) ;
            add_declaration(d);
            $$=d->type;
          }
        ;

    /*  ANSI  C  section  3.7.1  states  "An identifier declared as a
    typedef name shall not be redeclared as a parameter".  Hence  the
    following is based only on IDENTIFIERs */

identifier_list: /* (only used for old style functions) */
        IDENTIFIER
          {
             // Just put the name in the current scope. Later on
             // it gets updated.
             add_declaration(new Parse_Decl($1,ccchere));
          }
        | identifier_list ',' IDENTIFIER
          {
             add_declaration(new Parse_Decl($3,ccchere));
          }
        ;

identifier_or_typedef_name:   /* str */
        IDENTIFIER
        | TYPEDEFname
        ;

type_name:  /* Type*  */
        type_specifier
        | type_specifier abstract_declarator
	  {
             Parse_Decl* d = new Parse_Decl($1);
             d->combine_with_postfix($2);
             $$=d->type;
          }
        | type_qualifier_list
          { $$=make_basetype($1); }
        | type_qualifier_list abstract_declarator
	  {
             Parse_Decl* d = new Parse_Decl(make_basetype($1));
             d->combine_with_postfix($2);
             $$=d->type;
          }
        ;

initializer_opt:  /* Init */
        /* nothing */
          { $$=NULL; }
        | '=' initializer
          { $$=$2; }
        ;

initializer: /* Init */
        '{' initializer_list '}'
          { $$=$2; }
        | '{' initializer_list ',' '}'
          { $$=$2; }
        | assignment_expression
          { $$=new Init($1); }
        ;

initializer_list: /* Init */
        initializer
          { $$=new Init($1);}
        | initializer_list ',' initializer
            { $$=$1->append($3); }
        ;


/*************************** STATEMENTS *******************************/

statement:
          labeled_statement
        | compound_statement
        | expression_statement
        | selection_statement
        | iteration_statement
        | jump_statement
        ;

labeled_statement:
          identifier_or_typedef_name ':' statement
          {
	      LabelStmt* s = new LabelStmt($1,$3,ccchere);
              add_Label(s);
	      $$=s;
          }
        | CASE constant_expression ':' statement
          {
              $$=new CaseStmt($2->pos,$2,$4);
          }
        | DEFAULT ':' statement
          {
              $$=new DefaultStmt($3->pos,$3);
          }
        ;

compound_statement:
          '{'
          {
              enter_scope();
	  }
          compound_statement1
          '}'
          {
	      $$ = $3;
	      leave_scope();
	  }
        ;

compound_statement1:
          /* nothing */
          {
              $$=new DummyStmt(ccchere);
          }
        | declaration_list
          {
              // Pick up the declarations.
              Plist<VarDecl>* dcls = objects.get_list();
              Plist<UserDecl>* usrdcls = userdecls.get_list();
              // Use declarations first line.
	      if (dcls->empty())
		if (usrdcls->empty())
		  $$=new CompoundStmt(ccchere,dcls,usrdcls,new Plist<Stmt>);
		else
		  $$=new CompoundStmt(usrdcls->front()->pos,dcls,usrdcls,new Plist<Stmt>);
	      else
		$$=new CompoundStmt(dcls->front()->pos,dcls,usrdcls,new Plist<Stmt>);
          }
        | statement_list
          {
              $$=new CompoundStmt($1->front()->pos,
				  new Plist<VarDecl>,new Plist<UserDecl>,$1);
          }
        | declaration_list statement_list
          {
              // Pick up the declarations.
              Plist<VarDecl>* dcls = objects.get_list();
              Plist<UserDecl>* usrdcls = userdecls.get_list();
              // Find a position.
	      if (dcls->empty())
		$$=new CompoundStmt($2->front()->pos,dcls,usrdcls,$2);
	      else
		$$=new CompoundStmt(dcls->front()->pos,dcls,usrdcls,$2);
          }
        ;

declaration_list:
          declaration
        | declaration_list declaration
        ;

statement_list:
          statement
          {
              $$=new Plist<Stmt>($1);
          }
        | statement_list statement
          {
              $1->push_back($2);
              $$=$1;
          }
        ;

expression_statement:
          comma_expression_opt ';'
          {
              // The epression could be NULL (nothing).
              if ($1==NULL)
                  $$=new DummyStmt(ccchere);
              else
                  $$=new ExprStmt($1->pos,$1);
          }
        ;

selection_statement:
          IF '(' comma_expression ')' statement
          {
              $$=new IfStmt($3->pos,$3,$5,new DummyStmt(ccchere));
          }
        | IF '(' comma_expression ')' statement ELSE statement
          {
              $$=new IfStmt($3->pos,$3,$5,$7);
          }
        | SWITCH '(' comma_expression ')' statement
          {
              $$=new SwitchStmt($3->pos,$3,$5);
          }
        ;

iteration_statement:
          WHILE '(' comma_expression ')' statement
          {
              $$=new WhileStmt($3->pos,$3,$5);
          }
        | DO statement WHILE '(' comma_expression ')' ';'
          {
              $$=new DoStmt($2->pos,$5,$2);
         }
        | FOR '(' comma_expression_opt ';' comma_expression_opt ';'
                comma_expression_opt ')' statement
          {
              // Expressions can be NULL;
              Position p = $9->pos;
              if ($3 != NULL)
                  p = $3->pos;
              else if ($5 != NULL)
                  p = $5->pos;
              else if ($7 != NULL)
                  p = $7->pos;
              $$=new ForStmt(p,$3,$5,$7,$9);
          }
        ;

jump_statement:
          GOTO identifier_or_typedef_name ';'
          {
              $$=new GotoStmt(ccchere,add_goto($2,ccchere));
          }
        | CONTINUE ';'
          {
              $$=new ContStmt(ccchere);
          }
        | BREAK ';'
          {
              $$=new BreakStmt(ccchere);
          }
        | RETURN comma_expression_opt ';'
          {
              $$=new ReturnStmt(ccchere,$2);
          }
        ;

/***************************** EXTERNAL DEFINITIONS *****************************/

start_symbol:
        { settheflag(); } translation_unit { clean_up_between_files(); }
        ;

translation_unit:
        external_definition
        | translation_unit external_definition
        ;

external_definition:
        // After every function definition we dispose all labels.
        function_definition { goto_labels.clear(); }
        | declaration {}
        | cmix_syntactic_ext ';' {}
        ;

cmix_syntactic_ext:
	  TYPEDEF CMIXTAG ')' TYPEDEFname { /* ignore attempt to redefine */ }
	| TYPEDEF CMIXTAG ')' IDENTIFIER {
	      Type *thetype = new AbsType($4,$2);
              VarDecl *decl
                  = new VarDecl($4,ccchere,thetype,AbstractType,VarMu);
              names.insert($4,decl);
              assert($2!=NULL);
              objects.push_back(decl);
              directives.add_taboo($4);
          }
        ;

function_definition:
        identifier_declarator
          {
              // No type is specified so its regarded as int.
              Parse_Decl* d = make_decl(new Parse_Type());
              d->combine_with_postfix($1);
              $<fundef>$=add_FunDef(d);
          }
          compound_statement
          { if ( $<fundef>2 ) $<fundef>2->stmts = $3; }
        | declaration_specifier      identifier_declarator
          {
              $1->combine_with_postfix($2);
              $<fundef>$=add_FunDef($1);
          }
          compound_statement
          { if ( $<fundef>3 ) $<fundef>3->stmts = $4; }
        | type_specifier             identifier_declarator
          {
              Parse_Decl* d = new Parse_Decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=add_FunDef(d);
          }
          compound_statement
          { if ( $<fundef>3 ) $<fundef>3->stmts = $4; }
        | declaration_qualifier_list identifier_declarator
          {
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=add_FunDef(d);
          }
          compound_statement
          { if ( $<fundef>3 ) $<fundef>3->stmts = $4; }
        | type_qualifier_list        identifier_declarator
          {
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=add_FunDef(d);
          }
          compound_statement
          { if ( $<fundef>3 ) $<fundef>3->stmts = $4; }
        |                            old_function_declarator
          {
              Parse_Decl* d = make_decl(new Parse_Type());
              d->combine_with_postfix($1);
              $<fundef>$=old_style_add_FunDef(d);
          }
          compound_statement
          { $<fundef>2->stmts = $3; }
        | declaration_specifier      old_function_declarator
          {
              $1->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef($1);
          }
           compound_statement
          { $<fundef>3->stmts = $4; }
        | type_specifier             old_function_declarator
          {
              Parse_Decl* d = new Parse_Decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
           compound_statement
          { $<fundef>3->stmts = $4; }
        | declaration_qualifier_list old_function_declarator
          {
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
           compound_statement
          { $<fundef>3->stmts = $4; }
        | type_qualifier_list        old_function_declarator
          {
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
           compound_statement
          { $<fundef>3->stmts = $4; }
        |                            old_function_declarator
          { old_style_decl_list(); }
           declaration_list
          {
              leave_scope();
              Parse_Decl* d = make_decl(new Parse_Type());
              d->combine_with_postfix($1);
              $<fundef>$=old_style_add_FunDef(d);
          }
                compound_statement
          { $<fundef>4->stmts = $5; }
        | declaration_specifier      old_function_declarator
          { old_style_decl_list(); }
           declaration_list
          {
              leave_scope();
              $1->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef($1);
          }
                compound_statement
          { $<fundef>5->stmts = $6; }
        | type_specifier             old_function_declarator
          { old_style_decl_list(); }
           declaration_list
          {
              leave_scope();
              Parse_Decl* d = new Parse_Decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
                compound_statement
          { $<fundef>5->stmts = $6; }
        | declaration_qualifier_list old_function_declarator
          { old_style_decl_list(); }
           declaration_list
          {
              leave_scope();
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
                compound_statement
          { $<fundef>5->stmts = $6; }
        | type_qualifier_list        old_function_declarator
          { old_style_decl_list(); }
           declaration_list
          {
              leave_scope();
              Parse_Decl* d = make_decl($1);
              d->combine_with_postfix($2);
              $<fundef>$=old_style_add_FunDef(d);
          }
                compound_statement
          { $<fundef>5->stmts = $6; }
        ;

declarator:  /* Parse_Identifier*  */
        identifier_declarator
        | typedef_declarator
        ;

typedef_declarator:  /* Parse_Identifier*  */
        paren_typedef_declarator          /* would be ambiguous as parameter*/
        | parameter_typedef_declarator    /* not ambiguous as param*/
        ;

parameter_typedef_declarator:   /* Parse_IdPostfix*  */
        TYPEDEFname
          {
              $$=new Parse_IdPostfix($1,ccchere);
          }
        | TYPEDEFname postfixing_abstract_declarator
          {
              $$=new Parse_IdPostfix($1,ccchere,$2);
          }
        | clean_typedef_declarator
        ;

    /*  The  following have at least one '*'. There is no (redundant)
    '(' between the '*' and the TYPEDEFname. */

clean_typedef_declarator:  /* Parse_IdPostfix*  */
        clean_postfix_typedef_declarator
        | '*' parameter_typedef_declarator
          {
              $2->postfix->push_front(new Parse_Postfix(constvol()));
              $$=$2;
          }
        | '*' type_qualifier_list parameter_typedef_declarator
          {
              $3->postfix->push_front(new Parse_Postfix($2));
              $$=$3;
          }
        ;

clean_postfix_typedef_declarator:  /* Parse_IdPostfix*  */
        '(' clean_typedef_declarator ')'
          { $$=$2; }
        | '(' clean_typedef_declarator ')' postfixing_abstract_declarator
          { $$=$2->append_postfixes($4); }
        ;

    /* The following have a redundant '(' placed immediately  to  the
    left of the TYPEDEFname */

paren_typedef_declarator:   /* Parse_IdPostfix*  */
        paren_postfix_typedef_declarator
        | '*' '(' simple_paren_typedef_declarator ')' /* redundant paren */
          {
              Parse_Postfix* ptr = new Parse_Postfix(constvol());
              Plist<Parse_Postfix>* lst = new Plist<Parse_Postfix>();
              lst->push_back(ptr);
              $3->append_postfixes(lst);
              $$=$3;
          }
        | '*' type_qualifier_list
                '(' simple_paren_typedef_declarator ')' /* redundant paren */
          {
              Parse_Postfix* ptr = new Parse_Postfix($2);
              Plist<Parse_Postfix>* lst = new Plist<Parse_Postfix>();
              lst->push_back(ptr);
              $4->append_postfixes(lst);
              $$=$4;
          }
        | '*' paren_typedef_declarator
          { $2->postfix->push_front(new Parse_Postfix(constvol())); $$=$2; }
        | '*' type_qualifier_list paren_typedef_declarator
          { $3->postfix->push_front(new Parse_Postfix($2)); $$=$3; }
        ;

paren_postfix_typedef_declarator:   /* Parse_Identifier*  *//* redundant paren to left of tname*/
        '(' paren_typedef_declarator ')'
          { $$=$2; }
        | '(' simple_paren_typedef_declarator postfixing_abstract_declarator ')' /* redundant paren */
          { $$=$2->append_postfixes($3);}
        | '(' paren_typedef_declarator ')' postfixing_abstract_declarator
          {
            $$=$2->append_postfixes($4);
          }
        ;

simple_paren_typedef_declarator:   /* Parse_IdPostfix*  */
        TYPEDEFname
          {
              $$=new Parse_IdPostfix($1,ccchere);
          }
        | '(' simple_paren_typedef_declarator ')'
          { $$=$2; }
        ;

identifier_declarator:   /* Parse_IdPostfix*  */
        unary_identifier_declarator
        | paren_identifier_declarator
        ;

unary_identifier_declarator:   /* Parse_IdPostfix*  */
        postfix_identifier_declarator
        | '*' identifier_declarator
          {
             $2->postfix->push_front(new Parse_Postfix(constvol()));
             $$=$2;
          }
        | '*' type_qualifier_list identifier_declarator
          {
             $3->postfix->push_front(new Parse_Postfix($2));
             $$=$3;
          }
        ;

postfix_identifier_declarator:  /* Parse_IdPostfix*  */
        paren_identifier_declarator postfixing_abstract_declarator
          {
              $1->append_postfixes($2);
              // The declarations are in the list `postfix'.
              $$=$1;
          }
        | '(' unary_identifier_declarator ')'
          { $$=$2; }
        | '(' unary_identifier_declarator ')' postfixing_abstract_declarator
          {
              $2->prepend_postfixes($4);
              $$=$2;
          }
        ;

paren_identifier_declarator:   /* Parse_IdPostfix*  */
        IDENTIFIER
          {
              $$=new Parse_IdPostfix($1,ccchere);
          }
        | '(' paren_identifier_declarator ')'
          { $$=$2; }
        ;

old_function_declarator: /* Parse_IdPostfix*  */
        postfix_old_function_declarator
        | '*' old_function_declarator
          { $$=$2; $$->postfix->push_front(new Parse_Postfix(constvol())); }
        | '*' type_qualifier_list old_function_declarator
          { $$=$3; $$->postfix->push_front(new Parse_Postfix($2)); }
        ;

postfix_old_function_declarator:
        paren_identifier_declarator '(' { enter_scope(); } identifier_list ')'
          {
              leave_scope();
              $$=$1;
          }
        | '(' old_function_declarator ')'
          { $$=$2;}
        | '(' old_function_declarator ')' postfixing_abstract_declarator
          { $2->postfix->splice($2->postfix->end(),*$4); $$=$2; }
        ;

abstract_declarator:  /* Plist<Parse_Postfix>*  */
        unary_abstract_declarator
        | postfix_abstract_declarator
        | postfixing_abstract_declarator
        ;

postfixing_abstract_declarator:  /* Plist<Parse_Postfix>*  */
        array_abstract_declarator
        | '(' ')' // Function type (unknown parameters).
          {
              Parse_Postfix* p = new Parse_Postfix(new Plist<Type>(),0);
              $$=new Plist<Parse_Postfix>();
              $$->push_back(p);
              // Make sure that there is an empty list of parameters.
              enter_scope();
              leave_scope();
          }
        | '(' {enter_scope();} parameter_type_list ')'
          {
            leave_scope(); // This scope might have to be re-entered.
            $$=$3;
          }
        ;

array_abstract_declarator:  /* list<Parse_Postfix>* */
        '[' ']'
          {
              $$=new Plist<Parse_Postfix>(new Parse_Postfix((Expr*)NULL));
          }
        | '[' constant_expression ']'
          { $$=new Plist<Parse_Postfix>(new Parse_Postfix($2)); }
        | array_abstract_declarator '[' constant_expression ']'
	  // Insert sizes in reverse order so the type chain will
	  // get the right direction, ie. "int x[2] -> int x[2][3]"
	  // will be "array 2 of int -> array 2 of array 3 of int".
          { $1->push_front(new Parse_Postfix($3));
            $$=$1; }
        ;

unary_abstract_declarator:  /* list<Parse_Postfix>* */
        '*'
	  { $$=new Plist<Parse_Postfix>(new Parse_Postfix(constvol())); }
        | '*' type_qualifier_list
	  { $$=new Plist<Parse_Postfix>(new Parse_Postfix($2)); }
        | '*' abstract_declarator
          { $$=$2; $$->push_front(new Parse_Postfix(constvol())); }
        | '*' type_qualifier_list abstract_declarator
          { $$=$3; $$->push_front(new Parse_Postfix($2)); }
        ;

postfix_abstract_declarator:  /* list<Parse_Postfix>* */
        '(' unary_abstract_declarator ')'
          { $$=$2; }
        | '(' postfix_abstract_declarator ')'
          { $$=$2; }
        | '(' postfixing_abstract_declarator ')'
          { $$=$2; }
        | '(' unary_abstract_declarator ')' postfixing_abstract_declarator
          { $2->splice($2->end(),*$4); $$=$2; }
        ;

%%
/* ----end of grammar----*/

#ifdef YYDEBUG
        static void settheflag() {
            if (trace_c_parser)
                yydebug=1 ;
        }

#endif

cccParser parser ;
