/*
 *	cook - file construction tool
 *	Copyright (C) 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to quote strings
 */

#include <ac/ctype.h>
#include <ac/stdio.h>
#include <ac/string.h>

#include <mem.h>
#include <quote.h>
#include <str.h>


static char	*buffer;
static size_t	buffer_position;
static size_t	buffer_maximum;


/*
 * NAME
 *	buffer_putc
 *
 * SYNOPSIS
 *	void buffer_putc(int);
 *
 * DESCRIPTION
 *	The buffer_putc function is used to append a single character
 *	in the quote buffer.
 */

static void buffer_putc _((int));

static void
buffer_putc(c)
	int		c;
{
	if (buffer_position >= buffer_maximum)
	{
		buffer_maximum = buffer_maximum * 2 + 8;
		buffer = mem_change_size(buffer, buffer_maximum);
	}
	buffer[buffer_position++] = c;
}


/*
 * NAME
 *	buffer_puts
 *
 * SYNOPSIS
 *	void buffer_puts(char *);
 *
 * DESCRIPTION
 *	The buffer_putc function is used to append a string
 *	to the quote buffer.
 */

static void buffer_puts _((char *));

static void
buffer_puts(s)
	char		*s;
{
	while (*s)
		buffer_putc(*s++);
}


/*
 * NAME
 *	quotes_required
 *
 * SYNOPSIS
 *	void quotes_required(void);
 *
 * DESCRIPTION
 *	The quotes_required function is used to test whether quotes are
 *	required for the given string.
 *
 * RETURNS
 *	int; 1 required, 0 not required
 *
 * CAVEAT
 *	It test for Bourne shell special characters.
 */

static int quotes_required _((string_ty *));

static int
quotes_required(s)
	string_ty	*s;
{
	char		*cp;
	int		c;

	/*
	 * The empty string is a special case.
	 */
	if (!s->str_length)
		return 1;

	/*
	 * See if there are any characters which need quoting: all
	 * control characters and spces, plus *all* shell magic
	 * characters.
	 */
	for (cp = s->str_text; *cp; ++cp)
	{
		c = (unsigned char)*cp;
		if (!isgraph(c) || strchr("\"#$&'()*:;<=>?[\\]^`{|}", c))
			return 1;
	}
	return 0;
}


/*
 * NAME
 *	quoted
 *
 * SYNOPSIS
 *	string_ty *quoted(string_ty *);
 *
 * DESCRIPTION
 *	The quoted function is used to UNconditionally quote the given
 *	string.
 *
 * RETURNS
 *	string_ty *; the quoted string
 *
 * CAVEAT
 *	Bourne shell quoting and escapes are used.
 */

string_ty *
quoted(s)
	string_ty	*s;
{
	char		*cp;
	int		c;

	buffer_position = 0;
	buffer_putc('"');
	for (cp = s->str_text; *cp; ++cp)
	{
		c = (unsigned char)*cp;
		if (!isprint(c))
		{
			char	tmp[10];

			if (cp[1] && isdigit((unsigned char)cp[1]))
				sprintf(tmp, "\\%03o", c);
			else
				sprintf(tmp, "\\%o", c);
			buffer_puts(tmp);
		}
		else
		{
			switch (c)
			{
			case '"':
			case '$':
			case '\\':
			case '`':
				buffer_putc('\\');
				break;
			}
			buffer_putc(c);
		}
	}
	buffer_putc('"');

	/*
	 * assemble the new string and return it
	 */
	return str_n_from_c(buffer, buffer_position);
}


/*
 * NAME
 *	quote
 *
 * SYNOPSIS
 *	string_ty *quote(string_ty *);
 *
 * DESCRIPTION
 *	The quote function is used to quote strings to protect them from
 *	the Unix shell.  The string is passed through unchanged if it
 *	does not require quotes.
 */

string_ty *
quote(s)
	string_ty	*s;
{
	return (quotes_required(s) ? quoted(s) : str_copy(s));
}
