(* camlp4o pa_extend.cmo *)

(******** Types ********)
(*A personal constr*)
type t=
   Idt of (int*int)*string;;

type name=
   Name of string
  |Anonymous;;

type oper = 
    Meta of int
  | XTRA of string * t list
  | Sort
  | Implicit
  | Cast | Prod | Lambda
  | AppL | Const of string | Abst of string
  | MutInd of string * int
  | MutConstruct of (string * int) * int
  | MutCase
  | Fix of int array * int
  | CoFix of int
;;

type 'oper term =
    DOP0 of 'oper
  | DOP1 of 'oper * 'oper term
  | DOP2 of 'oper * 'oper term * 'oper term
  | DOPN of 'oper * 'oper term array
  | DOPL of 'oper * 'oper term list
  | DLAM of name * 'oper term
  | DLAMV of name * 'oper term array
  | VAR of string
  | Rel of int
;;

(******** Patterns terms ********)
(*The abstract syntax*)
type lterm=
   Lamb of (string*lterm)*lterm
  |Prd of (string*lterm)*lterm
  |Appl of lterm list
  |Id of string
  |Mtind of string*string*int*string
  |Mtcst of string*string*int*string*int
  |Cnst of string*string*string;;

let gram_term=Grammar.create (Plexer.make ());;
let lterm_eoi = Grammar.Entry.create gram_term "glterm";;
let lterm = Grammar.Entry.create gram_term "glterm";;
EXTEND
  lterm_eoi: [[ x=lterm; EOI -> x ]];
  lterm: [ [ "["; x=LIDENT; ":"; typ=lterm; "]"; t=lterm -> Lamb((x,typ),t)
           | "[|"; x=LIDENT; ":"; typ=lterm; "|]"; t=lterm -> Prd((x,typ),t)
           | t1=lterm; "->"; t2=lterm -> Prd(("",t1),t2)
           | "@"; "("; l=LIST1 lterm SEP " " ; ")" -> Appl l
           | "("; t=lterm ; ")" -> t
           | x=LIDENT -> Id x
           | "["; x=LIDENT; ","; l=LIST1 LIDENT SEP ","; ":"; typ=lterm; "]";
             t=lterm ->
             List.fold_right (fun x t -> Lamb ((x,typ),t)) (x::l) t
           | "[|"; x=LIDENT; ","; l=LIST1 LIDENT SEP ","; ":"; typ=lterm; "|]";
             t=lterm ->
             List.fold_right (fun x t -> Prd ((x,typ),t)) (x::l) t
           | "{"; name=LIDENT; ","; modl=UIDENT; ","; rank=INT; ",";
             typ=LIDENT; "}" ->
             Mtind(name,modl,int_of_string rank,typ)
           | "{"; name=LIDENT; ","; modl=UIDENT; ","; rankt=INT; ",";
             typ=LIDENT; ","; rankc=INT; "}" ->
             Mtcst(name,modl,int_of_string rankt,typ,int_of_string rankc)
           | "{"; name=LIDENT; ","; modl=UIDENT; ","; typ=LIDENT; "}" ->
             Cnst(name,modl,typ) ] ];
END;;

(*Constructs a special constr from a lterm*)
let constr_of_lterm term=
  let rec incr_env=function
     [] -> []
    |(x,rel)::t -> (x,rel+1)::(incr_env t)
  and constr_rec env=function
     Lamb((x,typ),t) ->
       DOP2(Lambda,constr_rec env typ,DLAM(Name x,constr_rec ((x,1)::(incr_env
       env)) t))
    |Prd((x,typ),t) ->
      if (String.length x)=0 then
        DOP2(Prod,constr_rec env typ,DLAM(Anonymous,constr_rec (incr_env
        env) t))
      else
        DOP2(Prod,constr_rec env typ,DLAM(Name x,constr_rec ((x,1)::(incr_env
        env)) t))
    |Appl l ->
      DOPL(AppL,List.map (constr_rec env) l)
    |Id x ->
      (try(Rel (List.assoc x env)) with
          Not_found -> DOP0(XTRA(x,[])))
    |Mtind(name,modl,rank,typ) ->
      DOPL(XTRA("IT",[Idt((rank,0),"#"^modl^"#"^name^"."^typ)]),[])
    |Mtcst(name,modl,rankt,typ,rankc) ->
      DOPL(XTRA("IC",[Idt((rankt,rankc),"#"^modl^"#"^name^"."^typ)]),[])
    |Cnst(name,modl,typ) ->
      DOPL(XTRA("C",[Idt((0,0),"#"^modl^"#"^name^"."^typ)]),[])
  in
    constr_rec [] term;;

(*Transforms a special constr into a string*)
let string_of_constr constr=
  let rec stlist lst=
    let rec stlist_rec=function
       [] -> ""
      |a::b ->
        if b=[] then
          (stco a)
        else          
          (stco a)^";"^(stlist_rec b)
    in
      "["^(stlist_rec lst)^"]"
  and stco=function
     DOP0(XTRA(x,[])) -> x
    |DOP2(Lambda,a,DLAM(Name _,b)) ->
       "Generic.DOP2(Term.Lambda,"^(stco a)^",Generic.DLAM(Names.Name _,"^
       (stco b)^"))"
    |DOP2(Prod,a,DLAM(Name _,b)) ->
       "Generic.DOP2(Term.Prod,"^(stco a)^",Generic.DLAM(Names.Name _,"^
       (stco b)^"))"
    |DOP2(Prod,a,DLAM(Anonymous,b)) ->
       "Generic.DOP2(Term.Prod,"^(stco a)^",Generic.DLAM(Names.Anonymous,"^
       (stco b)^"))"
    |DOPL(AppL,l) ->
       if (List.length l)=1 then
         "Generic.DOPL(Term.AppL,"^(stco (List.hd l))^")"
       else
         "Generic.DOPL(Term.AppL,"^(stlist l)^")"
    |DOPL(XTRA("IT",[Idt((rankt,0),str)]),[]) ->
       "Generic.DOPL(Term.XTRA(\"IT\",[CoqAst.Id(("^(string_of_int rankt)^
       ",0),\""^str^"\")]),[])"
    |DOPL(XTRA("IC",[Idt((rankt,rankc),str)]),[]) ->
       "Generic.DOPL(XTerm.TRA(\"IT\",[CoqAst.Id(("^(string_of_int rankt)^","^
       (string_of_int rankc)^"),\""^str^"\")]),[])"
    |DOPL(XTRA("C",[Idt((0,0),str)]),[]) ->
       "Generic.DOPL(Term.XTRA(\"C\",[CoqAst.Id((0,0),\""^str^"\")]),[])"
    |Rel a -> "Generic.Rel "^(string_of_int a)
    |_ -> failwith "string_of_constr"
  in
    stco constr;;

let lterm_exp _ str=
  let cs=Stream.of_string str
  in
    let term=Grammar.Entry.parse lterm_eoi cs
    in
      string_of_constr(constr_of_lterm term);;

Quotation.add "qflterm" (Quotation.ExStr lterm_exp);;

(******** Transformation of the matched term ********)
let match_exp _ str=
  "matchable ("^str^")";;

Quotation.add "qmatch" (Quotation.ExStr match_exp);;

(******** Normal terms ********)
(*Transforms a special constr into a normal string*)
let nstring_of_constr constr=
  let rec stlist lst=
    let rec stlist_rec=function
       [] -> ""
      |a::b ->
        if b=[] then
          (stco a)
        else          
          (stco a)^";"^(stlist_rec b)
    in
      "["^(stlist_rec lst)^"]"
  and stco=function
     DOP0(XTRA(x,[])) -> x
    |DOP2(Lambda,a,DLAM(Name ne,b)) ->
       "Generic.DOP2(Term.Lambda,"^(stco a)^
       ",Generic.DLAM(Names.Name (id_of_string \""^ne^"\"),"^(stco b)^"))"
    |DOP2(Prod,a,DLAM(Name ne,b)) ->
       "Generic.DOP2(Term.Prod,"^(stco a)^
       ",Generic.DLAM(Names.Name (id_of_string \""^ne^"\"),"^(stco b)^"))"
    |DOP2(Prod,a,DLAM(Anonymous,b)) ->
       "Generic.DOP2(Term.Prod,"^(stco a)^",Generic.DLAM(Names.Anonymous,"^
       (stco b)^"))"
    |DOPL(AppL,l) ->
       if (List.length l)=1 then
         "Generic.DOPN(Term.AppL,Array.of_list "^(stco (List.hd l))^")"
       else
         "Generic.DOPN(Term.AppL,Array.of_list "^(stlist l)^")"
    |DOPL(XTRA("IT",[Idt((rankt,0),str)]),[]) ->
       "Generic.DOPN(Term.MutInd(path_of_string \""^str^"\","^
       (string_of_int rankt)^"),Array.of_list [])"
    |DOPL(XTRA("IC",[Idt((rankt,rankc),str)]),[]) ->
       "Generic.DOPN(Term.MutConstruct((path_of_string \""^str^"\","^
       (string_of_int rankt)^"),"^(string_of_int rankc)^"),Array.of_list [])"
    |DOPL(XTRA("C",[Idt((0,0),str)]),[]) ->
       "Generic.DOPN(Term.Const(path_of_string "^str^"),Array.of_list [])"
    |Rel a -> "Generic.Rel "^(string_of_int a)
    |_ -> failwith "string_of_constr"
  in
    stco constr;;

let nlterm_exp _ str=
  let cs=Stream.of_string str
  in
    let term=Grammar.Entry.parse lterm_eoi cs
    in
      "(unmatchable ("^(nstring_of_constr(constr_of_lterm term))^"))";;

Quotation.add "qnlterm" (Quotation.ExStr nlterm_exp);;

(* $Id: psyntax.ml4,v 1.1 1998/01/14 04:12:11 delahaye Exp $ *)
