(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               Jul 10th 1997                              *)
(*                                                                          *)
(****************************************************************************)
(*                           cmd_searchisos_line.ml                         *)
(****************************************************************************)

open Char;;
open Std;;
open Initial;;
open States;;
open System;;
open Pp;;
open Vernac;;
open Cmdobj;;
open Coqtoplevel;;

let print_search_header () =
  Printf.printf "Welcome to Coq_SearchIsos V%s (%s)\n" Coq_config.versionsi
    Coq_config.date;;
  flush stdout;;

(*
 * Parsing of the command line.
 *
 * We no longer use Arg.parse, in order to use Usage.print_usage to be coherent
 * with launch/coqtop and launch/coqc.
 *)
(* TODO: usage for coq_searchisos *)
let usage () =
  Usage.print_usage_coqtop ();
  flush stderr ;
  exit 1;;

(*Environment*)
let vodis=ref false;;

let parse_args () =
  let rec parse = function
      [] -> ()
    | ("-P"|"-print-vo") ::rem -> vodis:=true

    | ("-I"|"-include") :: d :: rem -> push_include d ; parse rem
    | ("-I"|"-include") :: []       -> usage ()

    | "-q" :: rem -> no_load_rc () ; parse rem

    | "-load-ml-object" :: f :: rem -> Mltop.dir_ml_load f ; parse rem
    | "-load-ml-object" :: []       -> usage ()

    | "-load-ml-source" :: f :: rem -> Mltop.dir_ml_use f ; parse rem
    | "-load-ml-source" :: []       -> usage ()

    | "-unsafe" :: f :: rem -> add_unsafe f ; parse rem
    | "-unsafe" :: []       -> usage ()

    | "-debug" :: rem -> set_debug () ; parse rem

    | "-init-file" :: f :: rem -> set_rcfile f ; parse rem
    | "-init-file" :: []       -> usage ()

    | "-user" :: u :: rem -> set_rcuser u ; parse rem
    | "-user" :: []       -> usage ()

    | "-just-parsing" :: rem -> Vernac.just_parsing := true ; parse rem

    | s :: _ -> prerr_endline ("Don't know what to do with " ^ s) ; usage ()

  in
    try
      parse (List.tl (Array.to_list Sys.argv))
    with 
      	UserError(_,s) as e -> begin
          try
	    Stream.empty s; exit 1
          with
	      Stream.Failure ->
		mSGNL (Errors.explain_sys_exn e); exit 1
	end
      | e -> begin mSGNL (Errors.explain_sys_exn e); exit 1 end
;;

let load_vo()=
  let names_lst=ref []
  in
    let fill()=
      names_lst := System.all_vo_in_path () 
    and load()=
      let rec mod_by_mod=function
         [] -> if not(!vodis) then
                 round_step()
        |a::b ->
          (if (!vodis) then
             (print_endline ("Loading "^a^".vo");
              flush stdout));
          Searchisos.require_module2 (Some false) a None false;
          (if not(!vodis) then 
             round_step());
          mod_by_mod b
      in
        mod_by_mod !names_lst
    in
      print_string "\nSearching for .vo ... ";
      flush stdout;
      round_reset();
      fill();
      round_end();
      print_endline "Done";
      flush stdout;
      print_string "Loading .vo ... ";
      flush stdout;
      (if !vodis then
         (print_newline();
          flush stdout)
       else
         round_reset());
      load();
      (if not(!vodis) then
         round_end());
      print_endline "Done";
      flush stdout;;

let rec load_library_entry depth name=function
   (sp,Lib.LEAF lobj) ->
     round_step();
     Searchisos.load_leaf_entry name (sp,lobj)
  |(_,Lib.ClosedDir(a,_,_,ctxt)) ->
    if depth=0 then
      if not(a="Cpo") then
        load_context 1 (name^a^"#") ctxt
  |_ -> ()

and load_context depth name=
 let rec prec = function
    [] -> ()
   |h::rest -> load_library_entry depth name h ; prec rest
 in prec;;


let load_rew ()=
  print_string "Normalizing types ... ";
  round_reset();
  load_context 0 "#" (List.rev(Lib.contents_after None));
  round_end();
  print_endline "Done";
  flush stdout;;


(* To prevent from doing the initialization twice *)
let initialized = ref false;;

(* Ctrl-C is fatal during the initialisation *)
let start () =
  if not !initialized then
    begin 
      initialized := true;
      Sys.catch_break false;
      Coqtoplevel.set_prompt (fun _ -> "Coq_SearchIsos < ");
      Searchisos.search_in_lib := true;
      try
        parse_args ();
        print_search_header ();
        init_load_path ();
        Cmdobj.intern_restore_state "barestate.coq";
        load_rcfile();
        load_vo();
        Searchisos.upd_tbl_ind_one();
        load_rew()
      with e ->
        flush_all();
        message "Error during initialization :";
        mSGNL (Coqtoplevel.print_toplevel_error e);
        exit 1
    end;
  Coqtoplevel.go();;

(* $Id: cmd_searchisos_line.ml,v 1.11 1999/06/29 07:47:03 loiseleu Exp $ *)
