(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                              hashtabl.mli                                *)
(****************************************************************************)

(* Hash tables and hash functions. The difference with the standard library
   module Hashtbl is that we can freeze/unfreeze the state of the table.
   Another point is that the resizing is based on the mean length of
   buckets, and not on the maximum length. *)

(* Hash tables are hashed association tables, with in-place modification. *)

type ('a, 'b) t;;
        (* The type of association tables from type ['a] to type ['b]. *)

type ('a,'b) frozen_t;;

val create : int -> ('a,'b) t
        (* [create n] creates a new, empty hash table, with initial size [n].
           The table grows as needed, so [n] is just an initial guess.
           Apparently produces better results when [n] is a prime
           number. *)

  val clear : ('a, 'b) t -> unit
        (* Empty a hash table. *)

  val add : ('a, 'b) t -> 'a -> 'b -> unit
        (* [add tbl x y] binds the value [y] to key [x] in table [tbl].
           Previous bindings for [x] are not removed, but simply
           hidden. That is, after performing [remove tbl x], the previous
           binding for [x], if any, is restored.
           (This is the semantics of association lists.) *)

  val find : ('a, 'b) t -> 'a -> 'b
        (* [find tbl x] returns the current binding of [x] in [tbl],
           or raises [Not_found] if no such binding exists. *)

  val find_all : ('a, 'b) t -> 'a -> 'b list
        (* [find_all tbl x] returns the list of all data associated with [x]
           in [tbl]. The current binding is returned first, then the previous
           bindings, in reverse order of introduction in the table. *)

  val remove : ('a, 'b) t -> 'a -> unit
        (* [remove tbl x] removes the current binding of [x] in [tbl],
           restoring the previous binding if it exists.
           It does nothing if [x] is not bound in [tbl]. *)

  val do_table : ('a -> 'b -> 'c) -> ('a, 'b) t -> unit
        (* [do_table f tbl] applies [f] to all bindings in table [tbl],
       discarding all the results.
           [f] receives the key as first argument, and the associated value
           as second argument. The order in which the bindings are passed to
           [f] is unpredictable, except that successive bindings for the same
           key are presented in reverse chronological order 
           (most recent first). *)

  val do_table_rev : ('a -> 'b -> 'c) -> ('a, 'b) t -> unit
        (* Same as [do_table], except that successive bindings for the same
           key are presented in chronological order (oldest first). *)

  val freeze : ('a,'b) t -> ('a,'b) frozen_t
  val unfreeze : ('a,'b) frozen_t -> ('a,'b) t -> unit
;;

(*** The polymorphic hash primitive *)

val hash : 'a -> int
        (* [hash x] associates a positive integer to any value of
           any type. It is guaranteed that
                if [x = y], then [hash x = hash y]. 
           Moreover, [hash] always terminates, even on cyclic
           structures. *)
;;
external hash_param : int -> int -> 'a -> int =  "hash_univ_param"
        (* [hash_param n m x] computes a hash value for [x], with the
           same properties as for [hash]. The two extra parameters [n] and
           [m] give more precise control over hashing. Hashing performs a
           depth-first, right-to-left traversal of the structure [x], stopping
           after [n] meaningful nodes were encountered, or [m] nodes,
           meaningful or not, were encountered. Meaningful nodes are: integers;
           floating-point numbers; strings; characters; booleans; and constant
           constructors. Larger values for [m] and [n] mean that more
           nodes are taken into account to compute the final hash
           value, and therefore collisions are less likely to happen.
           However, hashing takes longer. Parameters [m] and [n] allow to
           trade accuracy for speed. *)
;;
val set_meanlen : ('a,'b) t -> int -> unit;;

(* $Id: hashtabl.mli,v 1.7 1999/06/29 07:47:17 loiseleu Exp $ *)
