(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 ast.mli                                  *)
(****************************************************************************)

(* Abstract syntax trees *)

open Pp
open Pcoq

(* raising located exceptions *)
val anomaly_loc : CoqAst.loc * string * std_ppcmds -> 'a
val user_err_loc : CoqAst.loc * string * std_ppcmds -> 'a
val invalid_arg_loc : CoqAst.loc * string -> 'a


val dummy_loc : CoqAst.loc
val loc: CoqAst.t -> CoqAst.loc

(* ast constructors with dummy location *)
val ope : string * CoqAst.t list -> CoqAst.t
val slam : string option * CoqAst.t -> CoqAst.t
val nvar : string -> CoqAst.t
val ide : string -> CoqAst.t
val num : int -> CoqAst.t
val str : string -> CoqAst.t
val path : string list -> string -> CoqAst.t
val dynamic : Dyn.t -> CoqAst.t

val set_loc : CoqAst.loc -> CoqAst.t -> CoqAst.t

(* ast destructors *)
val num_of_ast: CoqAst.t -> int
val id_of_ast: CoqAst.t -> string
val nvar_of_ast: CoqAst.t -> string

(* ast processing datatypes *)

(* patterns of ast *)
type pat =
    Pquote of CoqAst.t
  | Pmeta of string * tok_kind
  | Pnode of string * patlist
  | Pslam of string option * pat
  | Pmeta_slam of string * pat

and patlist =
    Pcons of pat * patlist
  | Plmeta of string
  | Pnil

and tok_kind = Tnum | Tid | Tstr | Tpath | Tvar | Tany | Tlist

(* semantic actions of grammar rules *)
type act =
    Aast of pat
  | Aastlist of patlist
  | Acase of act * (pat * act) list
  | Acaselist of act * (patlist * act) list

(* values associated to variables *)
type v =
    Vast of CoqAst.t
  | Vastlist of CoqAst.t list


type env = (string * v) list

val coerce_to_var : string -> CoqAst.t -> string

exception No_match of string

val isMeta : string -> bool


val print_ast : CoqAst.t -> std_ppcmds
val print_astl : CoqAst.t list -> std_ppcmds
val print_astpat : pat -> std_ppcmds
val print_astlpat : patlist -> std_ppcmds

(* Meta-syntax operations: matching and substitution *)

type entry_env = (string * entry_type) list

val grammar_type_error : CoqAst.loc * string -> 'a
 
(* Converting and checking free meta-variables *)
val pat_sub : CoqAst.loc -> env -> pat -> CoqAst.t
val val_of_ast : entry_env -> CoqAst.t -> pat
val vall_of_astl : entry_env -> CoqAst.t list -> patlist

val alpha_eq : CoqAst.t * CoqAst.t -> bool
val alpha_eq_val : v * v -> bool
val bind_env : env -> string -> v -> env
val ast_match : env -> pat -> CoqAst.t -> env
val astl_match : env -> patlist -> CoqAst.t list -> env
val first_match : ('a -> pat) -> env -> CoqAst.t -> 'a list ->
  ('a * env) option
val first_matchl : ('a -> patlist) -> env -> CoqAst.t list -> 'a list ->
  ('a * env) option

val to_pat : entry_env -> CoqAst.t -> (pat * entry_env)

val eval_act : CoqAst.loc -> env -> act -> v
val to_act_check_vars : entry_env -> entry_type -> CoqAst.t -> act

(* Hash-consing *)
val hcons_ast: (string -> string) ->
  (CoqAst.t -> CoqAst.t) * (CoqAst.loc -> CoqAst.loc)


(* $Id: ast.mli,v 1.9 1999/06/29 07:47:29 loiseleu Exp $ *)

