(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 himsg.ml                                 *)
(****************************************************************************)

open Pp;;
open Std;;
open More_util;;
open Names;;
open Term;;
open CoqAst;;
open Printer;;

(* pENV Prints a context with a header when non empty *)
let pENV h e = pr_ne_env h CCI e;;
let pFTERM = fprterm;;
let pTERM = prterm;;
let pTERMINENV (env,t) = term0 env t;;
let pFTERMINENV (env,t) = fterm0 env t;;
let pFTYPEINENV (env,t) = fprtype_env env t;;
let pID id = [< 'sTR(string_of_id id) >];;

let prtrm k t = gentermpr k (gLOB nil_sign) t;;


(* Does not print a second "Error :" when explaining the exception *)
let rec wrap_error (loc,sh,hd,e,tl) =
  match e with 
  | Stdpp.Exc_located _ -> raise e
  | UserError(s,pps) -> Ast.user_err_loc(loc,sh,[< hd; 'fNL; pps; 'fNL; tl >])
  | e -> Ast.user_err_loc
        (loc,sh,[< hd; 'fNL; Errors.explain_sys_exn e; 'fNL; tl >]);;


(* TODO: il faudrait des fonctions non curryfiees pour eviter des
         applications avec un nombre incorrect d'arguments! *)

let error_var_not_found str id = 
  errorlabstrm str
    [< 'sTR "The variable"; 'sPC ; 'sTR (string_of_id id) ; 
       'sPC ; 'sTR "was not found"; 
       'sPC ; 'sTR "in the current"; 'sPC ; 'sTR "environment" >];;

let error_fixname_unbound str is_cofix name = 
  errorlabstrm "dbize (COFIX)"
    [< 'sTR "The name"; 'sPC ; 'sTR name ; 
       'sPC ; 'sTR "is not bound in the corresponding"; 'sPC ;
       'sTR ((if is_cofix then "co" else "")^"fixpoint definition") >];;

let error_sosub_execute k com =
  Ast.user_err_loc
    (Ast.loc com, "error_sosub_execute",
     [< 'sTR"Failed to contract second-order (parser-introduced)"; 'sPC ;
        'sTR"redexes; perhaps the term" ; 'bRK(1,1) ; gencompr k com;
        'sPC; 'sTR"is malformed?" >]);;

(* We process the terms OUTSIDE the [< ... >], to perform the
   translation constr -> ast when the error message is raised and not
   when it should be printed, because a rollback may occur, therefore
   some constants may disappear, and the term would not be printable
   anymore
 *)


(* Raise standard typing error messages *)
let error_cant_execute k env c =
  let tc = gentermpr k env c in
    errorlabstrm "Cannot execute"
      [< 'sTR"Cannot execute term:"; 'bRK(1,1); tc >]

let error_unbound_rel k env n =
  let pe = pr_ne_env [< 'sTR"in environment" >] k env in
    errorlabstrm "unbound rel"
      [< 'sTR"Unbound reference: "; pe; 'fNL;
         'sTR"The reference "; 'iNT n; 'sTR" is free" >];;


let error_not_type k env c =
  let pe = pr_ne_env [< 'sTR"In environment" >] k env in
  let pc = gentermpr k env c in
    errorlabstrm "Not a type"
      [< pe; 'cUT; 'sTR "the term"; 'bRK(1,1); pc; 'sPC;
         'sTR"should be typed by Set, Prop or Type." >];;

let error_assumption k env c =
  let pc = gentermpr k env c in
    errorlabstrm "Bad assumption"
      [< 'sTR "Cannot declare a variable or hypothesis over the term";
         'bRK(1,1); pc; 'sPC; 'sTR "because this term is not a type." >];;

               
let error_generalization k env (name,var) j =
  let pe = pr_ne_env [< 'sTR"in environment" >] k env in
  let pv = gentermpr k env var.body in
  let pj = gentermpr k (add_rel (name,var) env) j._VAL in
    errorlabstrm "gen_rel"
      [< 'sTR"Illegal generalization: "; pe; 'fNL;
         'sTR"Cannot generalize"; 'bRK(1,1); pv; 'sPC;
         'sTR"over"; 'bRK(1,1); pj; 'sPC;
         'sTR"which should be typed by Set, Prop or Type."
       >]
;;

let error_cant_apply s k env rator randl =
  let pe = pr_ne_env [< 'sTR"in environment" >] k env in
  let pr = gentermpr k env rator._VAL in
  let prt = gentermpr k env rator._TYPE in
  let term_string = if List.length randl > 1 then "terms" else "term" in
  let appl = prlist_with_sep pr_fnl
               (fun c ->
                  let pc = gentermpr k env c._VAL in
                  let pct = gentermpr k env c._TYPE in
                    hOV 2 [< pc; 'sPC; 'sTR": " ; pct >]) randl in
    errorlabstrm "Illegal application"
      [< 'sTR"Illegal application ("; 'sTR s; 'sTR"): "; pe; 'fNL;
         'sTR"The term"; 'bRK(1,1); pr; 'sPC;
         'sTR"of type"; 'bRK(1,1); prt; 'sPC ;
         'sTR("cannot be applied to the "^term_string); 'fNL; 'sTR" "; v 0 appl
      >]
;;

let error_actual_type k env cj tj =
  let pe = pr_ne_env [< 'sTR"In environment" >] k env in
  let pc = gentermpr k env cj._VAL in
  let pct = gentermpr k env cj._TYPE in
  let pt = gentermpr k env tj._VAL in
    errorlabstrm "Bad cast"
      [< pe; 'fNL;
         'sTR"The term"; 'bRK(1,1); pc ; 'sPC ;
         'sTR"does not have type"; 'bRK(1,1); pt; 'fNL;
         'sTR"Actually, it has type" ; 'bRK(1,1); pct >]
;;


let error_ise_resolve k c pps =
  let pc = prtrm k c in
    errorlabstrm "ise_resolve"
      [< 'sTR"The term"; 'bRK(1,1); pc; 'sPC; 'sTR"failed to typecheck:"; 'sPC;
         pps >];;

(* inductive types *)
let error_not_inductive k c =
 let pc = prtrm k c in
   errorlabstrm "Non inductive type"
     [< 'sTR"The term"; 'bRK(1,1); pc; 'sPC;
        'sTR "is not an inductive definition" >];;

let error_case_not_inductive k env c ct =
  let pc = gentermpr k env c in
  let pct = gentermpr k env ct in
    errorlabstrm "Cases on non inductive type"
      [< 'sTR "In Cases expression"; 'bRK(1,1); pc; 'sPC; 
         'sTR "has type"; 'bRK(1,1); pct; 'sPC; 
         'sTR "which is not an inductive definition" >];;

let msg_bad_elimination k = function
    Some(ki,kp,explanation) ->
      let pki = prtrm k ki in
      let pkp = prtrm k kp in
        (hOV 0 
           [< 'fNL; 'sTR "Elimination of an inductive object of sort : ";
              pki; 'bRK(1,0);
              'sTR "is not allowed on a predicate in sort : "; pkp ;'fNL;
              'sTR "because"; 'sPC; 'sTR explanation >])
  | None -> [<>];;

let error_elim_arity k env ind aritylst c p pt okinds = 
  let pi = gentermpr k env ind in
  let ppar = prlist_with_sep pr_coma (gentermpr k env) aritylst in
  let pc = gentermpr k env c in
  let pp = gentermpr k env p in
  let ppt = gentermpr k env pt in
    errorlabstrm "incorrect elimimnation arity"
      [< 'sTR "Incorrect elimination of"; 'bRK(1,1); pc; 'sPC;
         'sTR "in the inductive type"; 'bRK(1,1); pi; 'fNL;
         'sTR "The elimination predicate"; 'bRK(1,1); pp; 'sPC;
         'sTR "has type"; 'bRK(1,1); ppt; 'fNL;
         'sTR "It should be one of :"; 'bRK(1,1) ; hOV 0 ppar; 'fNL;
         msg_bad_elimination k okinds >];;


let error_number_branches k env c ct expn =
  let pc = gentermpr k env c in
  let pct = gentermpr k env ct in
    errorlabstrm "Cases with wrong number of cases"
      [< 'sTR "Cases on term"; 'bRK(1,1); pc; 'sPC ;
         'sTR "of type"; 'bRK(1,1); pct; 'sPC;
         'sTR "expects ";  'iNT expn; 'sTR " branches" >];;

let error_ill_formed_branch k env c i actty expty =
  let pc = gentermpr k env c in
  let pa = gentermpr k env actty in
  let pe = gentermpr k env expty in
    errorlabstrm "Ill-formed branches" 
      [< 'sTR "In Cases expression on term"; 'bRK(1,1); pc;
         'sPC; 'sTR "the branch " ; 'iNT (i+1);
         'sTR " has type"; 'bRK(1,1); pa ; 'sPC; 
         'sTR "which should be:"; 'bRK(1,1); pe >];;

let error_ill_formed_inductive k lna c v =
  let env = assumptions_for_print lna in
  let pc = gentermpr k env c in
  let pv = gentermpr k env v in
    errorlabstrm "Ill-formed Inductive Specification"
      [< 'sTR "Not enough arguments applied to the "; pv;
	 'sTR " in"; 'bRK(1,1); pc >]

let str_of_nth n =
  (string_of_int n)^
  (match n mod 10 with
    1 -> "st"
  | 2 -> "nd"
  | 3 -> "rd"
  | _ -> "th");;

let error_bad_ind_parameters k lna c n v1 v2  =
  let env = assumptions_for_print lna in
  let pc = gentermpr k env c in
  let pv1 = gentermpr k env v1 in
  let pv2 = gentermpr k env v2 in
    errorlabstrm "Ill-formed Inductive Specification"
      [< 'sTR ("The "^(str_of_nth n)^" argument of "); pv2; 'bRK(1,1);
	 'sTR "must be "; pv1; 'sTR " in"; 'bRK(1,1); pc >]

let error_non_strictly_positive k lna c v  =
  let env = assumptions_for_print lna in
  let pc = gentermpr k env c in
  let pv = gentermpr k env v in
    errorlabstrm "Ill-formed Inductive Specification"
      [< 'sTR "Non strictly positive occurrence of "; pv; 'sTR " in";
	 'bRK(1,1); pc >]

let error_ill_formed_constructor k lna c v =
  let env = assumptions_for_print lna in
  let pc = gentermpr k env c in
  let pv = gentermpr k env v in
    errorlabstrm "Ill-formed Inductive Specification"
      [< 'sTR "The conclusion of"; 'bRK(1,1); pc; 'bRK(1,1); 
	 'sTR "is not valid;"; 'bRK(1,1); 'sTR "it must be built from "; pv >]

let error_ml_case mes env c ct br brt =
   let pc = pTERMINENV (env,c) in
   let pct = pTERMINENV (env,ct) in
   let expln =
     match mes with
         "Inductive" -> [< pct;'sTR "is not an inductive definition">]
       | "Predicate" -> [< 'sTR "ML case not allowed on a predicate">]
       | "Absurd" -> [< 'sTR "Ill-formed case expression on an empty type" >]
       | "Decomp" ->
           let plf = pTERMINENV(env,br) in
           let pft = pTERMINENV(env,brt) in
             [< 'sTR "The branch "; plf; 'wS 1; 'cUT; 'sTR "has type "; pft;
                'wS 1; 'cUT;
                'sTR "does not correspond to the inductive definition" >]
       | "Dependent" ->
           [< 'sTR "ML case not allowed for a dependent case elimination">]
       | _ -> [<>]
   in
     errorlabstrm "Ill-formed case expression"
       (hOV 0 [< 'sTR "In ML case expression on "; pc; 'wS 1; 'cUT ;
                 'sTR "of type";  'wS 1; pct; 'wS 1; 'cUT; 
                 'sTR "which is an inductive predicate."; 'fNL; expln >]);;

(* (co)fixpoints *)
let error_ill_formed_rec_body str k lna i vdefs =
  let env = assumptions_for_print lna in
  let s =
    match List.nth lna i with Name id -> string_of_id id | Anonymous -> "_" in
  let pvd = gentermpr k env vdefs.(i) in
    errorlabstrm "Ill-formed rec body"
      [< str; 'fNL; 'sTR"The ";
         if Array.length vdefs = 1 then [<>] else [<'iNT (i+1); 'sTR "-th ">];
         'sTR"recursive definition"; 'sPC; 'sTR s;
	 'sPC ; 'sTR":="; 'sPC; pvd; 'sPC;
         'sTR "is not well-formed">];;

let error_ill_typed_rec_body k i lna vdefj vargs =
  let ass = assumptions_for_print lna in
  let pvd = gentermpr k ass (vdefj.(i))._VAL in
  let pvdt = gentermpr k ass (vdefj.(i))._TYPE in
  let pv = prtrm k (body_of_type vargs.(i)) in
    errorlabstrm "Ill-typed rec body"
      [< 'sTR"The " ;
         if Array.length vdefj = 1 then [<>] else [<'iNT (i+1); 'sTR "-th">];
         'sTR"recursive definition" ; 'sPC; pvd; 'sPC;
         'sTR "has type"; 'sPC; pvdt;'sPC; 'sTR "it should be"; 'sPC; pv>];;


(* Unification errors *)

let error_cannot_unify k (m,n) =
  let pm = prtrm k m in 
  let pn = prtrm k n in
    errorlabstrm "Cannot unify"
      [< 'sTR"Impossible to unify"; 'bRK(1,1) ; pm; 'sPC ;
         'sTR"with"; 'bRK(1,1) ; pn >];;


(* $Id: himsg.ml,v 1.18 1999/11/10 16:17:57 herbelin Exp $ *)
