(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                fw_env.ml                                 *)
(****************************************************************************)

open Pp
open Std
open More_util
open Names
open Lib
open Generic
open Term
open Environ
open Termenv
open Constants
open Constrtypes
open Ml_import

(**************************************************************************)
(*                   recuperer un environnement FW minimal                *)
(**************************************************************************)

(* collect : constr -> section_path list
 * [collect t] collects the section paths appearing in the term t. *)

let collect = 
 let rec collect_rec acc = function
    DOP1(_,c)  			   -> collect_rec acc c
  | DOP2(Cast,c,_)                 -> collect_rec acc c
  | DOP2(_,c1,c2)                  -> 
      let l1 = collect_rec acc c1 in collect_rec l1 c2
  | DOPN(Const sp,_) 		   -> if List.mem sp acc then acc else sp::acc
  | DOPN(MutInd(sp,_),_) 	   -> if List.mem sp acc then acc else sp::acc
  | DOPN(MutConstruct((sp,_),_),_) -> if List.mem sp acc then acc else sp::acc
  | DOPN(MutCase (Some (sp,_)),cv) -> 
      let acc' = if List.mem sp acc then acc else sp::acc in
      it_vect collect_rec acc' cv
  | DOPN(_,cv) 			   -> it_vect collect_rec acc cv
  | DOPL(_,cl) 			   -> List.fold_left collect_rec acc cl
  | DLAM(_,c) 			   -> collect_rec acc c
  | DLAMV(_,cv) 		   -> it_vect collect_rec acc cv
  | _ 				   -> acc
 in collect_rec []

(* List of the needed section paths, with corresponding operations. *)

let nSP = (Hashtabl.create 17 : (section_path,unit) Hashtabl.t)
let reset_needed () = Hashtabl.clear nSP
let add_needed_sp sp = Hashtabl.add nSP sp ()
let is_needed sp = try Hashtabl.find nSP sp; true with Not_found -> false

(* The FW optimized environment (in reverse order). *)

let fWenv = ref ([] : section_path list)
let reset_fwenv () = fWenv := []
let add_fwenv sp = fWenv := sp :: !fWenv

(* traverse_object : object -> section_path list
 * If the object is a constant or an inductive then look for its sp. *)

let traverse_object sp obj =
  match (Libobject.object_tag obj) with
    "CONSTANT" ->
      	    let (cdecl,_,_) = outConstant obj in
      	    let cbody = try Listmap.map cdecl FW
		        with Not_found -> anomaly "traverse_object: FW" in
      	    if cookable_constant sp then begin
      	      cook_constant sp;		  (* cook the constant if needed *)
	      (match cbody.cONSTBODY with
	         Some {contents=COOKED c} -> collect c
	       | _ -> anomaly "traverse_object: should be cooked !")
	    end else
	      collect (body_of_type cbody.cONSTTYPE)

  | "MUTUALINDUCTIVE" ->
      	    let (midecl,_) = outMutualInductive obj in
	    let mibody = try Listmap.map midecl FW
		         with Not_found -> anomaly "traverse_object: FW" in
	    List.flatten (List.map (fun mipacket -> collect mipacket.mINDLC)
      	       	                   (Array.to_list mibody.mINDPACKETS))
      	    
  | _ -> anomaly "leaf needed which is not a constant or an inductive"

(* traverse_stack : (section_path -> bool)
                 -> bool
                 -> library_segment -> unit
 * travels (recursively) in a stack, collect section paths with 
 * traverse_object and add them to the needed sp list. *)

let traverse_stack critere recurs s = 
  let rec traverse = function
      (sp,ClosedDir(_,{module_p=true},_,ctxt))::stk ->
	traverse (List.rev ctxt); traverse stk
	  
    | (sp,ClosedDir(_,_,_,ctxt))::stk ->
	traverse stk
	  
    | (sp,LEAF obj)::stk ->
	let sp' = fwsp_of sp in
	  if (critere sp') & (not (is_import_or_extract sp')) then begin
            add_fwenv sp;
	    if recurs then
	      let spl = traverse_object sp' obj in List.iter add_needed_sp spl
	  end;
	  if (sp_is_ml_import sp') then begin
            let spl = traverse_object sp' obj in
              if not (List.for_all sp_is_ml_import_or_prod spl) then
		errorlabstrm "Fw_env.traverse_stack"
		  [< 'sTR"The type of the imported object" ; 'sPC ; 
		     'sTR(string_of_id (basename sp')) ; 'sPC ; 
		     'sTR"is not an ML type." ; 'fNL >]
	  end;
	  traverse stk
	    
    | _::stk -> traverse stk
	  
    | [] -> ()
  in
    traverse s

(* optimize : unit -> section_path list
 * [optimize ()] builds an optimized environment for ML programs *)

let optimize critere recurs =
  let initial_segment = contents_after None in
  reset_fwenv();
  traverse_stack critere recurs initial_segment;
  !fWenv

(* recursive_env : section_path list -> section_path list
 * [recursive_env spl] returns a minimal environment for the FW section
 * paths list spl. *)

let recursive_env recurs spl =
  reset_needed();
  List.iter (fun sp -> add_needed_sp sp) spl;
  optimize is_needed recurs

(* module_env : identifier -> section_path list
 * [module_env m] returns the environment for the FW module m *)

let rec is_ml_type = function
    DOP0 _ as op -> not (is_Type op)
  | DOP1(_,c) -> is_ml_type c
  | DOP2(_,c1,c2) -> is_ml_type c1 & is_ml_type c2
  | DOPN(_,a) -> Vectops.for_all_vect is_ml_type a
  | DOPL(_,l) -> List.for_all is_ml_type l
  | DLAM(_,c) -> is_ml_type c
  | DLAMV(_,a) -> Vectops.for_all_vect is_ml_type a
  | _ -> true

let is_ml_object sp =
  try let (_,cb) = const_of_path sp in is_ml_type (body_of_type cb.cONSTTYPE)
  with _ ->
    try let (_,ib) = mind_of_path sp in 
        Vectops.for_all_vect (fun p -> is_Set p.mINDARITY.body) ib.mINDPACKETS
    with _ -> false

let module_env m =
  let ms = string_of_id m in
  let critere sp = dirpath sp = [ms] & is_ml_object sp in
  optimize critere false

(* $Id: fw_env.ml,v 1.15 1999/08/06 20:49:20 herbelin Exp $ *)
