(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                fwtoml.ml                                 *)
(****************************************************************************)

open Pp
open Std
open Vectops
open More_util
open System
open Generic
open Names
open Term
open Vartab
open Constrtypes
open Machops
open Himsg
open Mach
open Reduction

open Environ
open Ml_import
open Mlterm
open Vernacinterp

(* Added for the contribution by J.-F. Monin to extract non 
   ML typable functions *) 

let lOOSE_EXTRACTION = ref false

let loose_extraction () = !lOOSE_EXTRACTION

let set_loose b = lOOSE_EXTRACTION:= b

let _ = overwriting_vinterp_add("LOOSEF", 
  function
      [] -> (fun () -> set_loose false)
    | _ -> assert false)

let _ = overwriting_vinterp_add("LOOSET",
  function
      [] -> (fun () -> set_loose true)
    | _ -> assert false)

let cofix = ref false

let execute rc env t =
  try execute_rec rc env t
  with UserError("cannot execute Implicit",_) -> errorlabstrm
           "fwtoml__execute" [< 'sTR"Cannot extract terms with Implicit" >]

let execute_type rc env t =
  try execute_rec_type rc env t
  with UserError("cannot execute Implicit",_) -> errorlabstrm
           "fwtoml__execute" [< 'sTR"Cannot extract terms with Implicit" >]

let notanMLtype = "not an ML type";;

(**************************************************************************)
(*                    informations on inductive types:                    *)
(*      we keep them in a table in order to get quickly constructor       *)
(*                           names and arities                            *)
(**************************************************************************)

type packet_info = 
  { pACKET_name    : identifier ;
    pACKET_ncons   : int ;
    pACKET_cnames  : identifier array ;
    pACKET_arities : int array }

let mind_table =
  (Hashtabl.create 17 : (section_path * int , packet_info) Hashtabl.t)

let reset_mind_table () =
  Hashtabl.clear mind_table

let store_mind_info key info =
  Hashtabl.add mind_table key info

let get_mind_info key =
  Hashtabl.find mind_table key

(**************************************************************************)
(*          translation of type expressions and inductive types           *)
(**************************************************************************)

(* In order to be general with respect to renaming functions, we define
 * here a functor taking a module of renaming functions as argument
 * and returning the module of translation functions *)

type renaming_function = identifier list -> name -> identifier

module type RENAMING =
  sig
    val rename_type_parameter : renaming_function
    val rename_type           : renaming_function
    val rename_term           : renaming_function
    val rename_global_type    : renaming_function
    val rename_global_constructor : renaming_function
    val rename_global_term    : renaming_function
  end

(* and we keep a table of renamings for the globals.
 * Then every renaming function will be called with its own "avoid list"
 * and the list of already known globals. *)

let globals_renaming = ref ([] : (identifier * identifier) list)

let globals = ref ([] : identifier list)

let reset_globals_renaming () =
  globals_renaming := [] ;
  globals          := []

let add_global_renaming ((_,id') as r) = 
  globals_renaming :=   r::!globals_renaming ;
  globals          := id'::!globals

let get_global_name id =
  try  List.assoc id !globals_renaming
  with Not_found -> anomalylabstrm "Fwtoml.get_global_name"
    (hOV 0 [< 'sTR "Found a global " ; print_id id ;
              'sTR " without renaming" >])

let name_of_oper oper =
  if is_ml_import oper then
    find_ml_import oper
  else if is_ml_extract oper then
    find_ml_extract oper
  else match oper with
      Const sp -> get_global_name (basename sp)
    | MutInd (sp,i) ->
	let info = get_mind_info (sp,i) in info.pACKET_name
    | MutConstruct ((sp,i),j) ->
	let info = get_mind_info (sp,i) in info.pACKET_cnames.(j-1)
    | _ -> invalid_arg "name_of_oper"

(* The functor : *)

module Mlenv_of_fwenv = functor (R : RENAMING) -> struct

(* the renaming functions must take globals into account *)

let ren_type_parameter av n = R.rename_type_parameter (!globals@av) n
let ren_type           av n = R.rename_type           (!globals@av) n
let ren_term           av n = R.rename_term           (!globals@av) n

let ren_global_type id        = R.rename_global_type        !globals (Name id)
let ren_global_constructor id = R.rename_global_constructor !globals (Name id)
let ren_global_term id        = R.rename_global_term        !globals (Name id)

(* mltype_of_constr : typeid list -> constr -> MLtype
 * [mltype_of_constr dbenv c] translates the type expression c into
 * an ML type expression, in a de Bruijn environment dbenv. *)

let rec mltype_of_constr dbenv c = match whd_beta c with
  | DOP2 (Cast,c,_) ->
      mltype_of_constr dbenv c

  | DOP2(Prod,c1,DLAM(name,c2)) ->
      	let avoid = List.map (function TYPEparam id -> id
      	       	       	       	     | TYPEname id -> id) dbenv in
      	let id = ren_type avoid name in
	TYarr(mltype_of_constr dbenv c1,
              mltype_of_constr ((TYPEname id)::dbenv) c2)

  | DOPN(AppL,cv) ->
        TYapp(List.map (fun c -> mltype_of_constr dbenv c) (Array.to_list cv))

  | Rel n ->
      	TYvar(List.nth dbenv (n-1))

  | DOPN(MutInd _ as oper,_) ->
      TYglob (name_of_oper oper)

  | DOPN(Const _ as oper,_) ->
      TYglob (name_of_oper oper)

  (* added for a contribution of Lannion *)
  | DOP0(Sort (Prop Pos)) -> 
      if not (loose_extraction ()) then failwith notanMLtype;
      warning "extracting a too polymorphic function for the ML type-checker"; 
      TYglob (basename (path_of_string "#Datatypes#unit.fw"))

  | _ -> failwith notanMLtype

let rec isSet = function
    DOP0(Sort(Prop(Pos))) -> true
  | DOP2(Cast,c,_)        -> isSet c
  | _			  -> false

let abbrev_of_constr c =
  let rec get_params acc c = match strip_outer_cast c with
    DOP2(Lambda, typ, DLAM(name,t)) ->
      if isSet typ then 
      	 let id = ren_type_parameter acc name in
      	 get_params (id::acc) t
      else
      	 failwith notanMLtype
  | t -> acc,t  in

  let idl,t = get_params [] c in
  let env = List.map (fun id -> TYPEparam id) idl in
  List.rev idl, mltype_of_constr env t

(* decl_of_inductive : section_path -> mutual_inductive_body -> MLdecl
 * [decl_of_inductive sp cb] translates an inductive definition into
 * an ML type declaration. *)

let rec type_of_c idl env acc = function
      DOP2(Cast,t,_) ->
      	    type_of_c idl env acc t

    | DOP2(Prod,t1,DLAM(name,t2)) -> 
      	    let c = execute_type (Evd.mt_evd()) env t1 in
	    let env' = add_rel (name,c) env in
	    let avoid = List.map (function TYPEparam id -> id
      	         	       	         | TYPEname id -> id) idl in
   	    let idl' = (TYPEname (ren_type avoid name))::idl in

      	    if isType (level_of_type c) then
	      type_of_c idl' env' acc t2
	    else
	      type_of_c idl' env' ((mltype_of_constr idl t1)::acc) t2

    | _ ->  List.rev acc

let decl_of_one_inductive (sp,i) env np packet =
  let name = packet.mINDTYPENAME in

  let rec params_names_and_env t acc env' = function
     0 -> acc,env'
   | n -> 
       (match (strip_outer_cast t) with
            DOP2(Prod,t_0,DLAM(nm,t')) -> 
	      let cj = execute_type (Evd.mt_evd()) env' t_0 in
      	      let id = ren_type_parameter acc nm in
      	      params_names_and_env t' (id::acc) (add_rel (nm,cj) env') (n-1)
	  | _ -> anomaly "decl_of_one_inductive: should be a product!")
  in
  let rec remove_params t = function
     0 -> t
   | n -> (match (strip_outer_cast t) with
             DOP2(Prod,_,DLAM(_,t')) -> remove_params t' (n-1)
	   | _ -> anomaly "decl_of_one_inductive: should be a product!")
  in
  let namel,consv = decomp_all_DLAMV_name packet.mINDLC in
  let nbcons = Array.length consv in
  let arl = Array.map (fun c -> remove_params c np) consv in
  let pl,env' = params_names_and_env consv.(0) [] env np in
  let idl = (List.map (fun id -> (TYPEparam id)) pl)
      	   @(List.map (function (Name id) -> (TYPEname (get_global_name id))
			 | _ -> assert false)
                      namel)
  in
  (* renaming of constructors *)
  let cnames = Array.copy packet.mINDCONSNAMES in
  for i = 0 to nbcons-1 do
    let id = cnames.(i) in
    let id' = ren_global_constructor id in
    add_global_renaming (id,id') ;
    cnames.(i) <- id'
  done ;

  let info = { pACKET_name    = get_global_name name ;
      	       pACKET_ncons   = nbcons ;
      	       pACKET_cnames  = cnames ;
	       pACKET_arities = Array.create nbcons 0 }
  in
  let fmt_constructor j =
    let l = type_of_c idl env' [] arl.(j) in
    info.pACKET_arities.(j) <- List.length l ;
    (cnames.(j), l)
  in

  store_mind_info (sp,i) info;
  (List.rev pl, get_global_name name, 
                List.map (fun i -> fmt_constructor i) (list0n (nbcons-1)))

let decl_of_inductive sp mib =
  let nparams = mib.mINDNPARAMS in
  let env = it_vect (fun e p -> add_rel (Name p.mINDTYPENAME, p.mINDARITY) e)
      	       	    (gLOB(initial_sign()))
      	       	    mib.mINDPACKETS in
  let ipl = List.combine (list0n (mib.mINDNTYPES - 1))
      	       	     (Array.to_list mib.mINDPACKETS) in

  (* renaming of the types names *)
  List.iter (fun id -> let id' = ren_global_type id in
                       add_global_renaming (id,id')) 
            (List.map (fun (_,p) -> p.mINDTYPENAME) ipl) ;

  DECLtype(List.map 
     (fun (i,packet) -> decl_of_one_inductive (sp,i) env nparams packet) ipl)


(**************************************************************************)
(*    translation of constants : type abbreviations and declarations      *)
(**************************************************************************)

let collect_n_lambda = 
 let rec collect acc = fun p_0 p_1 ->
   match p_0,p_1 with
       (0, t)              -> acc,0,t
     | (n, (MLlam(id,t'))) -> collect (id::acc) (n-1) t'
     | (n, t)              -> acc,n,t
 in collect []
  


(* mlast_of_constr : identifier list -> constr assumptions -> constr -> MLast
 * [mlast_of_constr idl env c] give an MLast for the term c in the
 * environment env. idl is the list of the identifiers of env. *)

let rec mlast_of_constr idl env args =
  let apply t = match args with
     [] -> t
   | _  -> MLapp(t,args) in

  function c -> match strip_outer_cast c with
    DOPN(Fix _,_) as t ->
      	 apply (mlast_of_fix false idl env true t)

  | DOPN(CoFix _,_) as t ->
      	 if !cofix then
      	   apply (mlast_of_fix false idl env true t)
	 else
	   failwith "cofix"

  | DOPN(MutCase _,_) as t ->
         let (_,_,c,tv) = destCase t in
	 let j = execute (Evd.mt_evd()) env c in
	 let typ = whd_betadeltaiota (Evd.mt_evd()) j._TYPE in
	 let mi = match List.hd (Tactics.head_constr typ) with
	                DOPN(MutInd (x_0,x_1),_) -> (x_0,x_1)
		      | _ -> anomaly "mlast_of_constr: bad type in case" in
	 let h = mlast_of_constr idl env [] c in
	 let args = Array.map (fun c -> mlast_of_constr idl env [] c) tv in
	 let pl = mlpat_of_constr idl mi args in
	 apply (MLcase(h, pl))

  | DOPN(MutConstruct((sp,i),j) as oper,_) ->
      	 let info = get_mind_info (sp,i) in
	 let cname = name_of_oper oper in
	 let n = info.pACKET_arities.(j-1) in
	 let d = n - (List.length args) in
	 if d>0 then
	   let vl = new_vars idl d in
	   let rels = List.map 
                  (fun i -> MLrel i) (List.rev (List.tl (list0n d))) in
	   let args' = List.map (fun t -> ml_lift d t) args in
	   let argl = args'@rels in
      	   List.fold_left (fun t id -> MLlam(id,t)) (MLcons(j, cname, argl)) vl
         else
	   let (l,m) = chop_list n args in
	   if m=[] then MLcons(j, cname, l)
	           else MLapp(MLcons(j, cname, l), m)

  | DOPN(Const _ as oper,_) ->
      	 let id = name_of_oper oper in
      	 apply (MLglob id)

  | DOPN(AppL,cv) ->
      	 if Array.length cv = 0 then
	   anomaly "mlast_of_constr: application without argument!"
	 else
	   let args' = it_vect 
      	      (fun l c -> let j = execute (Evd.mt_evd()) env c in
	                  if is_Type j._KIND then
      	       	       	    l
      	       	       	  else
      	       	       	    (mlast_of_constr idl env [] c)::l)
              [] (tl_vect cv) in
	   mlast_of_constr idl env ((List.rev args')@args) cv.(0) 
  
  | DOP2(Lambda,t,DLAM(name,c')) ->
      	 let id = ren_term idl name in
	 let j = execute_type (Evd.mt_evd()) env t in
	 if isType (level_of_type j) then
	   let t_0 = mlast_of_constr (id::idl) (add_rel (name,j) env) [] c' in
	   apply (ml_pop t_0)
	 else
	   apply (MLlam (id,
      	       	    mlast_of_constr (id::idl) (add_rel (name,j) env) [] c'))

  | Rel n ->
      	 apply (MLrel n)

  | _ -> invalid_arg "mlast_of_constr"

and mlast_of_fix glob_p idl env in_p c = match strip_outer_cast c with
   (DOPN(Fix _,_) | DOPN(CoFix _,_)) as t ->
      	  let (j,cv) = match t with
      	    DOPN(Fix (_,j),cv) -> j,cv
	  | DOPN(CoFix j,cv)   -> j,cv 
	  | _ -> assert false in
      	  let k = (Array.length cv)-1 in
	  let fiv,tiv = decomp_all_DLAMV_name (cv.(k)) in
	  let aiv = Array.sub cv 0 k in
      	  let cAl = List.map
		      (fun a -> execute_type (Evd.mt_evd()) env a)
		      (Array.to_list aiv) in
	  let env',_ = List.fold_left2 
              (fun (acc,i) name cA 
		 -> (add_rel (name,type_app (lift i) cA) acc),(succ i))
              (env,0) (List.rev fiv) cAl in
              
          (* here we rename the recursive functions *)
	  let fid = List.fold_left 
             (fun l name -> let id = ren_term (l@idl) name in id::l)
	     [] fiv in

	  let idl' = (List.rev fid)@idl in
      	  let tv = List.map (fun c -> mlast_of_constr idl' env' [] c)
      	       	       (Array.to_list tiv) in

      	  MLfix (j, in_p, fid, tv)

 | _ -> invalid_arg "mlast_of_fix"

and mlpat_of_constr idl mi cl =
  let info = get_mind_info mi in
  let one_pat j =
    let c = cl.(j) in
    let name = name_of_oper (MutConstruct (mi,succ j)) in
    let n = info.pACKET_arities.(j) in
    let ids,r,t = collect_n_lambda n c in
    let ids' = rename_list idl ids in
    if r=0 then
      (name, ids', t)
    else
      let ids'' = new_vars (ids'@idl) r in
      let rels = List.map (fun i -> MLrel i) (List.rev (List.tl (list0n r))) in
      (name, ids''@ids', MLapp(ml_lift r t, rels))

  in Array.map (fun i -> one_pat i)
      	      (Array.of_list (list0n (info.pACKET_ncons - 1)))


let rec is_rec = function
    DOPN(Fix _,_)  -> true
  | DOP2(Cast,c,_) -> is_rec c
  | _		   -> false


(* decl_of_constant : section_path -> constant_body -> MLdecl
 * [decl_of_constant sp cb] returns the ML declaration for constant cb
 * of section path sp (OBJ). *)

let axiom_type_msg id =
  errorlabstrm "decl_of_constant" (hOV 0
  [< 'sTR"You must link axiom " ; print_id id ; 'sTR" first !" >] )

let not_an_ML_type_msg id =
  errorlabstrm "decl_of_constant" (hOV 0
  [< 'sTR"Constant " ; print_id id ;
     'sTR" does not correspond to an ML type" >] )

let decl_of_constant axioms sp cb =
  let id = basename sp in
  let typ = cb.cONSTTYPE in
  if isType (level_of_type typ) then
    let c = match Constants.cooked_body cb with
       	      Some c -> c
	    | _      -> axiom_type_msg id in
    let idl,t = try abbrev_of_constr c
      	       	with (Failure "not an ML type") 
                -> not_an_ML_type_msg id 
    in
    let id' = ren_global_type id in
    add_global_renaming (id,id') ;
    DECLabbrev(id', idl, t)
  else
    (let t = match Constants.cooked_body cb with
         Some c -> if is_rec c 
	     then mlast_of_fix true [] (gLOB(initial_sign())) false c
	     else mlast_of_constr [] (gLOB(initial_sign())) [] c
       | None -> (mSGNL [< 'sTR"The axiom " ; print_id id ;
                     'sTR" is translated into an exception." >] ;
	      MLexn id)
     in let id' = ren_global_term id in
           add_global_renaming (id,id') ; 
	   DECLglob(id', t))
;;


(**************************************************************************)
(*           translation of an environment into ML declarations           *)
(*            traverse the environment and distinguish between            *)
(*                  constants and inductive definitions                   *)
(**************************************************************************)

(* decl_of_sp : bool -> section_path -> MLdecl
 * [decl_of_env ax spl] returns the declaration corresponding to
 * sp (OBJ). ax=true if axioms are allowed. *)

let decl_of_sp axioms sp =
  match Lib.map sp with
      Lib.LEAF obj ->
  	(match Libobject.object_tag obj with
	     "CONSTANT" ->
      	       let (cdecl,_,_) = outConstant obj in
      		 (try decl_of_constant axioms sp (Listmap.map cdecl FW)
		  with Failure "cofix" -> errorlabstrm "decl_of_sp"
		      [< 'sTR"Co-recursive definition in " ;
      			 'sTR(string_of_id (basename sp)) ; 
			 'sTR": extract into Haskell !" >]
                )
		 
	   | "MUTUALINDUCTIVE" ->
      	       let (midecl,_) = outMutualInductive obj in
      	       let fsp = fwsp_of sp in
      		 decl_of_inductive fsp (Listmap.map midecl FW)
		   
	   | _  -> anomaly "caml_of_sp: not a constant or an inductive")
    | _ -> assert false

let mlenv_of_fwenv (axioms,b) env =
  reset_caml_names () ;
  reset_mind_table () ;
  reset_globals_renaming () ;
  cofix := b;
  (* on commence par traiter tous les inductifs, de manire  ce que la
   * table d'informations sur les inductifs (arits, nb. constr., ...)
   * soit complte. *)
  List.iter (fun sp -> let _ = decl_of_sp axioms (objsp_of sp) in ())
            (List.rev !mL_INDUCTIVES) ;
  (* On peut alors appliquer la traduction Fw -> ML  l'environnement Fw *)
  List.map (decl_of_sp axioms) env


end (* of functor Fwenv_of_mlenv *)


(* The type of this functor is functor(R : RENAMING) -> TRANSLATION
   where *)

module type TRANSLATION = 
    sig
      val ren_type_parameter :
        Names.identifier list -> Names.name -> Names.identifier
      val ren_type : Names.identifier list -> Names.name -> Names.identifier
      val ren_term : Names.identifier list -> Names.name -> Names.identifier
      val ren_global_type : Names.identifier -> Names.identifier
      val ren_global_constructor : Names.identifier -> Names.identifier
      val ren_global_term : Names.identifier -> Names.identifier
      val mltype_of_constr :
        Mlterm.typeid list -> Term.constr -> Mlterm.mLtype
      val isSet : Term.sorts Term.oper Generic.term -> bool
      val abbrev_of_constr :
        Term.constr -> Names.identifier list * Mlterm.mLtype
      val type_of_c :
        Mlterm.typeid list ->
        Term.environment ->
        Mlterm.mLtype list ->
        Term.sorts Term.oper Generic.term -> Mlterm.mLtype list
      val decl_of_one_inductive :
        Names.section_path * int ->
        Term.environment ->
        int ->
        Constrtypes.mutual_inductive_packet ->
        Names.identifier list * Names.identifier *
        (Names.identifier * Mlterm.mLtype list) list
      val decl_of_inductive :
        Names.section_path ->
        Constrtypes.mutual_inductive_body -> Mlterm.mLdecl
      val collect_n_lambda :
        int -> Mlterm.mLast -> Names.identifier list * int * Mlterm.mLast
      val mlast_of_constr :
        Names.identifier list ->
        Term.environment ->
        Mlterm.mLast list -> Term.constr -> Mlterm.mLast
      val mlast_of_fix :
        bool ->
        Names.identifier list ->
        Term.environment -> bool -> Term.constr -> Mlterm.mLast
      val mlpat_of_constr :
        Names.identifier list ->
        Names.section_path * int ->
        Mlterm.mLast array ->
        (Names.identifier * Names.identifier list * Mlterm.mLast) array
      val is_rec : 'a Term.oper Generic.term -> bool
      val axiom_type_msg : Names.identifier -> 'a
      val not_an_ML_type_msg : Names.identifier -> 'a
      val decl_of_constant :
        'a ->
        Names.section_path -> Constrtypes.constant_body -> Mlterm.mLdecl
      val decl_of_sp : 'a -> Names.section_path -> Mlterm.mLdecl
      val mlenv_of_fwenv :
        'a * bool -> Names.section_path list -> Mlterm.mLdecl list
    end
(***************************************************************************)

(* $Id: fwtoml.ml,v 1.32 1999/11/03 08:58:11 mohring Exp $ *)
