(require 'compile)
(require 'lib-complete)

(defconst coq-load-path 
  (let ((coqtop (getenv "COQTOP")))
    (mapcar '(lambda (x) (concat coqtop x))
	    (list 
	     "/"
	     "/doc/"
	     "/doc/library/"
	     "/doc/omega/"
	     "/distrib/"
	     "/src/"
	     "/src/config/"
	     "/src/constr/"
	     "/src/env/"
	     "/src/launch/"
	     "/src/lib/util/"
	     "/src/man/"
	     "/src/meta/"
	     "/src/parsing/"
	     "/src/proofs/"
	     "/src/syntax/"
	     "/src/tactics/"
	     "/src/typing/"
	     "/tactics/"
	     "/tactics/contrib/"
	     "/tactics/contrib/polynom/"
	     "/tactics/contrib/extraction/"
	     "/tactics/contrib/linear/"
	     "/tactics/contrib/natural/"
	     "/tactics/contrib/omega/"
	     "/tactics/contrib/reflexion/"
	     "/tactics/tcc/"
	     "/tactics/programs/"
	     "/tactics/programs/EXAMPLES/"
	     "/theories/"
	     "/theories/ARITH/"
	     "/theories/BOOL/"
	     "/theories/DEMOS/"
	     "/theories/DEMOS/OMEGA/"
	     "/theories/DEMOS/PROGRAMS/"
	     "/theories/INIT/"
	     "/theories/TREES/"
	     "/theories/LISTS/"
	     "/theories/LOGIC/"
	     "/theories/RELATIONS/"
	     "/theories/RELATIONS/WELLFOUNDED/"
	     "/theories/SETS/"
	     "/theories/REALS/"
	     "/theories/SORTING/"
	     "/theories/TESTS/"
	     "/theories/TREES/"
	     "/theories/ZARITH/"
	     "/tools/"
	     "/tools/camlp4/"
	     "/tools/coq-tex/"
	     "/tools/coqdep/"
	     "/tools/do_Makefile/"
	     "/tools/emacs/"
	     "/tools/filters/"
	     "/tools/gallina/"
	     "/tools/dev/"
	     "/tools/coqinstall/"))))

(defun coq-find-file-internal (filename read-only)
  "Don't call this. Use coq-find-file instead."
  (catch 'answer
    (mapcar
     '(lambda (dir)
	(mapcar
	 '(lambda (suf)
	    (let ((try (concat dir (concat filename suf))))
              (and (file-readable-p try)
                   (null (file-directory-p try))
                   (progn
                     (if read-only 
			 (find-file-read-only try)
		       (find-file try))
                     (throw 'answer try)))))
         '(".ml" ".ml4" ".mli" ".v" ".tex" "")))
     coq-load-path)
    (message "No file %s in coqtop subdirectories" filename)
    nil))

(defun coq-find-file-filter (filename) 
  "Returns filename unmodified, or nil if filename ends with .o, .cmo, 
or similar extentions"
  (if (or (string-match "\\.cm[oixa]$" filename)
	  (string-match "\\.v[oi]$" filename)
	  (string-match "\\.cmxa$" filename)
	  (string-match "\\.o$" filename)
	  (string-match "\\.aux$" filename)
	  (string-match "\\.log$" filename)
	  (string-match "\\.dvi$" filename)
	  (string-match "\\.ps$" filename)
	  (string-match "\\.v\\.tex$" filename)
	  (string-match "/$" filename))
      nil
    filename))

(defun coq-find-file ()
  "Find and open a file in the $COQTOP subdirectories"
  (interactive)
  (let ((file
	 (read-library "Open Coq file: " coq-load-path "" nil nil
		       'coq-find-file-filter)))
    (coq-find-file-internal file nil)))

(defun coq-find-file-read-only ()
  "Find and open in read-only mode a file in the $COQTOP subdirectories"
  (interactive)
  (let ((file
	 (read-library "Open Coq file: " coq-load-path "" nil nil
		       'coq-find-file-filter)))
    (coq-find-file-internal file t)))

(defun grepcoq (file-pattern pattern mode-name)
  "Does grep pattern in all files whose name matches file-pattern in 
     $COQTOP subdirectories"
  (compile-internal (concat "grep -n \""
			    pattern
			    "\" `find " 
			    (getenv "COQTOP") 
			    " -type f -name "
			    file-pattern
			    "` || true")
		    "No more matches"
		    (concat "Grep in Coq " mode-name)
		    nil
		    grep-regexp-alist
		    ))

(defun gml (pattern)
  "Does a grep in all ml files of the $COQTOP subdirectories"
  (interactive "sGrep in Coq ml files the regexp: ")
  (grepcoq "\\*.ml" pattern "ML files"))

(defun gmli (pattern)
  "Does a grep in all mli files of the $COQTOP subdirectories"
  (interactive "sGrep in Coq mli files the regexp: ")
  (grepcoq "\\*.mli" pattern "MLI files"))

(defun gml4 (pattern)
  "Does a grep in all ml4 files of the $COQTOP subdirectories"
  (interactive "sGrep in Coq ml4 files the regexp: ")
  (grepcoq "\\*.ml4" pattern "ML4 files"))

(defun gv (pattern)
  "Does a grep in all ml files of the $COQTOP subdirectories"
  (interactive "sGrep in Coq vernac files the regexp: ")
  (grepcoq "\\*.v" pattern "Vernac files"))

;; charge les TAGS si ils existent

(let ((tagsfile (concat (getenv "COQTOP") "/TAGS")))
  (if (file-readable-p tagsfile)
      (visit-tags-table tagsfile))
  )

;; commentaire avec centrage

(defun coq-comment ()
  "Make a Coq comment with the current line"
  (interactive)
  (end-of-line)
  (let ((eolpos 0) (spl 0) (i 0))
  (setq eolpos (current-column))
  (setq spl (- 70 eolpos))
  (beginning-of-line)
  (insert "(**************************************************************************)")
  (newline)
  (beginning-of-line)
  (insert "(*")
  (setq i (/ spl 2))
  (while (>= i 0)
    (insert " ")
    (setq i (1- i)))
  (end-of-line)
  (setq i (- spl (/ spl 2)))
  (while (>= i 0)
    (insert " ")
    (setq i (1- i)))
  (insert "*)")
  (newline)
  (insert "(**************************************************************************)")
  (newline))
)
