------------------------------------------------------------------------------
--                                                                          --
--                            GCH COMPONENTS                                --
--                                                                          --
--                          G C H .  O U T P U T                            --
--                                                                          --
--                              B o d y                                     --
--                                                                          --
--                                                                          --
--              Copyright (c) 1999, Vitali Sh.Kaufman.                      --
--                                                                          --
--  Gch is distributed as free software; that is with full sources          --
--  and in the hope that it will be useful, but WITHOUT ANY WARRANTY;       --
--  without even the implied warranty of MERCHANTABILITY or FITNESS         --
--  FOR A PARTICULAR PURPOSE. You can freely copy, modify and redistribute  --
--  this software, provided that full sources are available for the version --
--  being distribute (original and modified), and for a modified version,   --
--  any changes that you have made are clearly indicated.                   --
--                                                                          --
--  Gch was developed by Vitali Sh. Kaufman using a prototype               --
--  and consultations by Sergey I. Rybin.                                   --
------------------------------------------------------------------------------

--  This package defines subprograms providing various forms of output needed
--  by Gch

with Ada.Unchecked_Deallocation;
-- to clean memory in Output_Diagnostics

with Asis;                    use Asis;
with Asis.Text;               use Asis.Text;
with Asis.Implementation;     use Asis.Implementation;
with Asis.Elements;           use Asis.Elements;

with Gch.Options;         use Gch.Options;
with Gch.Globals;         use Gch.Globals;
with Gch.Rules;           use Gch.Rules;
with Gch.Init;            use Gch.Init;

package body Gch.Output is

--  ###VK The package should be clean up to improve its structure

   package Int_IO is new Integer_IO (Integer);  --  ###VK to remove?
   use Int_IO;
   package Nat_IO is new Integer_IO (Natural);  --  ###VK to remove?
   use Nat_IO;

   ----------------
   -- Brief_Help --
   ----------------

   procedure Brief_Help is
   begin

      Put_Line ("Usage: Gch options filename");
      New_Line;

      Put_Line ("  filename  source file (wild cards are allowed)");
      New_Line;

      Put_Line ("  options are standard set of options used for gcc "
                & "to call GNAT (starting from '-c')");

      Put_Line ("  - if 'options' contais '-gnatg', the GNAT "
                & "style rule are checked first");

      Put_Line ("  - if 'options' contais '-gnatc', the source file is not "
                & "compile to create an object");

   end Brief_Help;

   -------------------------
   -- Put_Gch_Version --
   -------------------------

   procedure Put_Gch_Version is
   begin
      Put ("version ");
      Put (To_Wide_String (Gch_Version));
      Put (" built in ");
      Put (Asis.Implementation.ASIS_Implementor_Version);
      New_Line;
   end Put_Gch_Version;

   ------------------------------
   -- Output_Global_Statistics --
   ------------------------------

   procedure Output_Global_Statistics is
   begin
      if Total_Units > 0 then
         New_Line;
         Put_Line (">>>>  GLOBAL STATISTICS <<<<");
         Put      ("  unit processed:  ");
         Put_Line (Natural'Wide_Image (Total_Units));
         Put      ("  unit checked:    ");
         Put_Line (Natural'Wide_Image (Checked_Units));
         Put      ("  unit passed:     ");
         Put_Line (Natural'Wide_Image (Passed_Units));
         Put      ("  unit not_passed: ");
         Put_Line (Natural'Wide_Image (Not_Passed_Units));
         Put      ("  unit rejected:   ");
         Put_Line (Natural'Wide_Image (Rejected_Units));

         if Warning_Mode = Warning then
            Put   ("  all warnings:    ");
            Put_Line (Natural'Wide_Image (Total_Warnings));
         end if;

      end if;
   end Output_Global_Statistics;

   -----------------------
   -- Output_Statistics --
   -----------------------

   procedure Output_Statistics is
   begin
      if Errors_Per_Unit > 0 or else Warnings_Per_Unit > 0 then
         New_line;
         Put ("Detected:");

         if Errors_Per_Unit > 0 then
            Put (" errors   -");
            Put (Natural'Wide_Image (Errors_Per_Unit));
            Errors_Per_Unit := 0;
         end if;

         if Warnings_Per_Unit > 0 then
            Put (" warnings -");
            Put (Natural'Wide_Image (Warnings_Per_Unit));
            Warnings_Per_Unit := 0;
         end if;

         if Failures_Per_Unit > 0 then
            Put (" failures -");
            Put (Natural'Wide_Image (Failures_Per_Unit));
            Failures_Per_Unit := 0;
         end if;

      end if;
   end Output_Statistics;

   ----------------------
   --  Report_Element  --  ###VK not used
   ----------------------  ??? is ASIS-for-GNAT-specific


   procedure Report_Element  (El : Asis.Element) is
      El_Span : Span;
   begin
      if not Is_Text_Available (El) then
         Put_Line ("!!! cannot report about implicit elements  !!!");
         Put_Line ("!!! send a message to asis-report@gnat.com !!!");
      end if;

      El_Span := Element_Span (El);

      Put ("positions ");
      Put (Line_Number'Wide_Image (El_Span.First_Line));
      Put (" :");
      Put (Character_Position'Wide_Image (El_Span.First_Column));
      Put (" -");
      Put (Line_Number'Wide_Image (El_Span.Last_Line));
      Put (" :");
      Put (Character_Position'Wide_Image (El_Span.Last_Column));

      if Output_Elements then
         New_Line;
         Put (Element_Image (El));
      end if;

   --  ??? exception handler for possible errors in Asis.Text  queries
   end Report_Element;

   -----------------------------
   -- Report_Non_ASIS_Failure --
   -----------------------------

   procedure Report_Non_ASIS_Failure (El : Element; Ex_Name : String) is
   begin
      Put ("Non-ASIS Failure (");
      Put (To_Wide_String (Ex_Name));
      Put_Line (") when processing the Element:");
      Put_Line (Debug_Image (El));

      Failures_Per_Unit := Failures_Per_Unit + 1;

   end Report_Non_ASIS_Failure;

   -----------------------------
   -- Output_Diagnostics --
   -----------------------------

   procedure Output_Diagnostics (Checking_File_Name : String) is

      Current_Pointer : Rule_Violation_Node_Access := Diagnostics;
      To_Free_Pointer : Rule_Violation_Node_Access := Current_Pointer;

      E_Span : Span;
      E_First_Line : Line_Number;

      Unit_Span : Span;
      Rule : Rule_Index;

   procedure Free is new Ada.Unchecked_Deallocation
         (Rule_Violation_Node, Rule_Violation_Node_Access);

   begin

      while Current_Pointer /= null loop
         E_Span := Element_Span (Current_Pointer.Bad_Element);

      --  to calculate line numbers from the beginning of
      --  the checked compilation unit
         E_First_Line := First_Line_Number (Current_Pointer.Bad_Element);

         Rule := Current_Pointer.Violated_Rule;


         if Gnat_Mode then

            declare
            --  we use the following two variable because of a redundant space
            --  of Wide_Image attribute
               Line_Number_Wide_Image : Wide_String
                                   := Line_Number'Wide_Image (E_First_Line);
               Character_Position_Wide_Image : Wide_String
                                   := Character_Position'Wide_Image
                                         (E_Span.First_Column);

               E_Lines : Line_List :=
                  Lines (Element    => Current_Pointer.Bad_Element,
                         First_Line => E_Span.First_Line,
                         Last_Line  => E_Span.First_Line);

            begin

               if Verbose_Mode then
                  New_Line;
                  New_Line;
                  Put (Line_Number_Wide_Image
                         (2 .. Line_Number_Wide_Image'Last));
                  Put (". ");
                  Put (Line_Image (E_Lines (E_Lines'First)));
                  New_Line;

                  for I in 1 .. E_Span.First_Column + 3 loop
                     Put (' ');
                  end loop;

                  Put ('|');
                  New_Line;
                  Put (">>> ");

               else
                  New_Line;
                  Put (To_Wide_String (Checking_File_Name));
                  Put (":");

                  Put (Line_Number_Wide_Image
                         (2 .. Line_Number_Wide_Image'Last));

                  Put (":");

                  Put (Character_Position_Wide_Image
                         (2 .. Character_Position_Wide_Image'Last));

                  Put (":");

               end if;

               Put (To_Wide_String (Gch.Rules.Rules (Rule).Diagnosis.all));
               Put (" [");
               Put (To_Wide_String (Gch.Rules.Rules (Rule).Rule_Name.all));
               Put ("]");

            end;

         else

            New_Line;
            Put ("line");
            Put (Line_Number'Wide_Image
                  (E_First_Line));
            Put (" col");
            Put (Character_Position'Wide_Image (E_Span.First_Column));
            Put (":");
            Put (" ");

            Put (To_Wide_String (Gch.Rules.Rules (Rule).Diagnosis.all));
            Put (" [");
            Put (To_Wide_String (Gch.Rules.Rules (Rule).Rule_Name.all));
            Put ("]");

            if Verbose_Mode then
               New_Line;
               Put (Element_Image (Current_Pointer.Bad_Element));
            end if;
         end if;

         Current_Pointer := Current_Pointer.Next_Node;
         Free (To_Free_Pointer);
         To_Free_Pointer := Current_Pointer;
      end loop;

      Diagnostics := null; --  to clean Diagnostics list for next usage
      if not Verbose_Mode then
         New_Line;
      end if;
   end Output_Diagnostics;

end Gch.Output;