// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "fileio.h"
#include "thread.h"
#include "socket.h"

static	void endsocket(SOCKET so)
{
#ifdef	SO_LINGER
	struct linger linger;

	linger.l_onoff = linger.l_linger = 0;
	setsockopt(so, SOL_SOCKET, SO_LINGER, (const char *)&linger, sizeof(linger));
#endif	
	shutdown(so, 2);
	closesocket(so);
}

Socket::Socket(int domain, int type, int protocol)
{
	state = SOCKET_INITIAL;
	so = socket(domain, type, protocol);
	if(so < 0)
		throw(this);

	state = SOCKET_AVAILABLE;
}

Socket::Socket(SOCKET fd)
{
	state = SOCKET_INITIAL;
	if(fd < 0)
	{
		throw(this);
		return;
	}
	so = fd;
	state = SOCKET_AVAILABLE;
}

Socket::Socket(const Socket &orig)
{
	state = SOCKET_INITIAL;
	so = orig.so;
	if(so < 0)
		throw(this);
	state = orig.state;
}

Socket::~Socket()
{
	if(state == SOCKET_INITIAL)
		return;

	endsocket(so);
	state = SOCKET_INITIAL;
	so = -1;
}		

Socket &Socket::operator=(const Socket &from)
{
	if(so == from.so)
		return *this;

	if(state != SOCKET_INITIAL)
		endsocket(so);

	so = from.so;
	if(so < 0)
		state = SOCKET_INITIAL;
	else
		state = from.state;

	return *this;
}

InetHostAddress Socket::Peek(short *port)
{
	struct sockaddr_in from;
	InetHostAddress host;
	char buf;
	int len = sizeof(from);

	int rc = recvfrom(so, &buf, 1, MSG_PEEK, (struct sockaddr *)&from, &len);
	if(rc == 1)
	{
		host = from.sin_addr;
		if(port)
			*port = ntohs(from.sin_port);
	}
	return host;
}

UDPSocket::UDPSocket(void) : Socket(PF_INET, SOCK_DGRAM, 0)
{
}

UDPSocket::UDPSocket(InetHostAddress &host, short port) :
Socket(PF_INET, SOCK_DGRAM, 0)
{
	struct sockaddr_in addr;
	
	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(host);
	addr.sin_port = htons(port);
	if(connect(so, (sockaddr *)&addr, sizeof(addr)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}
}
	
UDPSocket::UDPSocket(InetAddress &ia, short port) : 
Socket(PF_INET, SOCK_DGRAM, 0)
{
	struct	sockaddr_in addr;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);
	if(bind(so, (sockaddr *)&addr, sizeof(addr)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}
	state = SOCKET_BOUND;
}

UDPSocket::UDPSocket(BroadcastAddress &ia, short port) : 
Socket(PF_INET, SOCK_DGRAM, 0)
{
	struct sockaddr_in addr;
	int	bcast = 1;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);

	if(bind(so, (sockaddr *)&addr, sizeof(addr)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}

	if(setsockopt(so, SOL_SOCKET, SO_BROADCAST, (const char *)&bcast, sizeof(bcast)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}
	state = SOCKET_BOUND;
}

int UDPSocket::Write(InetHostAddress &host, short port, void *addr, size_t len)
{
	struct sockaddr_in to;

	to.sin_family = AF_INET;
	to.sin_addr = getaddress(host);
	to.sin_port = htons(port);	

	return sendto(so, (char *)addr, len, 0, (struct sockaddr *)&to, sizeof(to));
}


TCPSocket::TCPSocket(InetHostAddress &host, short port) : 
Socket(PF_INET, SOCK_STREAM, 0)
{
	struct sockaddr_in addr;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(host);
	addr.sin_port = htons(port);

	if(connect(so, (struct sockaddr *)&addr, sizeof(addr)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}
	state = SOCKET_CONNECTED;
}

TCPSocket::TCPSocket(InetAddress &ia, short port, int backlog) : 
Socket(PF_INET, SOCK_STREAM, 0)
{
	struct sockaddr_in addr;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);

	if(bind(so, (struct sockaddr *)&addr, sizeof(addr)))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}

	if(listen(so, backlog))
	{
		endsocket(so);
		state = SOCKET_INITIAL;
		throw((Socket *)this);
		return;
	}
	state = SOCKET_BOUND;
}

TCPSocket::TCPSocket(TCPSocket &server) : 
Socket(accept(server.so, NULL, NULL)) 
{
	InetHostAddress temp;
	short port;

	temp = getPeer(&port);
	if(!server.OnAccept(temp, port))
	{
		state = SOCKET_INITIAL;
		endsocket(so);
		throw((Socket *)this);
		return;
	}
		
	state = SOCKET_CONNECTED;
}

TCPSocket::TCPSocket(TCPSocket *server) :
Socket(accept(server->so, NULL, NULL))
{
	InetHostAddress temp;
	short port;

	temp = getPeer(&port);
	if(!server->OnAccept(temp, port))
	{
		state = SOCKET_INITIAL;
		endsocket(so);
		throw((Socket *)this);
		return;
	}
	state = SOCKET_CONNECTED;
}

int TCPSocket::Readline(char *buf, size_t max)
{
	int pos = 0;
	int len;
	int eol;
	bool nl = false;

	while(!nl)
	{
		len = Peek(buf + pos, max - pos);
		if(len <= 0)
		{
			if(!pos)
				return len;

			break;
		}
		eol = 0;
		while(eol <= len && !nl)
		{
			if(buf[pos + eol] == '\n')
				nl = true;

			++eol;
		}
		len = Read(buf + pos, eol);
		if(len < 0)
			break;
							 		
		if(len < eol)
			nl = false;

		pos += len;
	}
	buf[pos] = 0;
	return pos;
}

InetHostAddress TCPSocket::getHost(short *port)
{
	struct sockaddr_in addr;
	int	len = sizeof(addr);

	if(getsockname(so, (struct sockaddr *)&addr, &len))
		return InetHostAddress();

	if(port)
		*port = ntohs(addr.sin_port);
	return InetHostAddress((struct in_addr)addr.sin_addr);
}

InetHostAddress TCPSocket::getPeer(short *port)
{
	struct sockaddr_in addr;
	int len = sizeof(addr);

	if(getpeername(so, (struct sockaddr *)&addr, &len))
		return InetHostAddress();

	if(port)
		*port = ntohs(addr.sin_port);
	return InetHostAddress((struct in_addr)addr.sin_addr);
}

TCPSession::TCPSession(Semaphore *start, TCPSocket &server, int pri, int stack) :
TCPSocket(server), Thread(start, pri, stack) {};

TCPSession::TCPSession(TCPSocket *server, Semaphore *start, int pri, int stack) :
TCPSocket(server), Thread(start, pri, stack) {};

