#ifndef SIGCXX_THREAD_H
#define SIGCXX_THREAD_H
#include <sigc++config.h>

#ifdef SIGC_PTHREADS
#include <pthread.h>

#ifdef connect
#undef connect
#endif

#ifdef SIGC_CXX_NAMESPACES
namespace SigC
{
namespace Threads
{
#else
#define Threads 
#endif

#ifdef SIGC_PTHREAD_DCE
typedef pthread_condattr_t CondAttr;
typedef pthread_mutexattr_t MutexAttr;
typedef pthread_attr_t ThreadAttr;
#else
typedef pthread_condattr_t* CondAttr;
typedef pthread_mutexattr_t* MutexAttr;
typedef pthread_attr_t* ThreadAttr;
#endif

// Mutual Exclusion
class Mutex
  {
   typedef pthread_mutex_t Impl;
   private:
     Impl mutex_;
   public:
     static MutexAttr Default;
     operator Impl* ()  {return &mutex_;}

     Mutex(const MutexAttr attr=Default) 
       {
        pthread_mutex_init(&mutex_,attr);
       }
     // (needs work) 
     ~Mutex()
       {destroy();}

     int lock()    {return pthread_mutex_lock(&mutex_);}
     int trylock() {return pthread_mutex_trylock(&mutex_);}
     int unlock()  {return pthread_mutex_unlock(&mutex_);}
     int destroy() {return pthread_mutex_destroy(&mutex_);}
  };

// A lazy way to unlock at end of scope
struct MLock
  {
   Mutex &mutex_;
   MLock(Mutex& mutex):mutex_(mutex) {mutex_.lock();}
   ~MLock()                          {mutex_.unlock();}
  };

// Condition Variable
struct Condition
  {
   typedef pthread_cond_t Impl;
   private:
     Impl cond_;
   public:
     static CondAttr Default;
     operator Impl* ()  {return &cond_;}

     Condition(const CondAttr &attr=Default) 
       {
        pthread_cond_init(&cond_,attr);
       }
     ~Condition()
       {destroy();}

     // restarts exactly one thread hung on condition
     int signal()       {return pthread_cond_signal(&cond_);}

     // restarts all threads waiting on condition
     int broadcast()    {return pthread_cond_broadcast(&cond_);}

     // unlocks a mutex while waiting on a condition, then reaquires lock.
     int wait(Mutex &m) {return pthread_cond_wait(&cond_,m);}

     // unlocks a mutex while waiting on a condition, then reaquires lock
     // with a fixed maximum duration.
     int wait(Mutex &m,timespec* spec) 
       {return pthread_cond_timedwait(&cond_,m,spec);}

     int destroy()      {return pthread_cond_destroy(&cond_);}
  };

// Integer Semaphore
struct Semaphore
  {
   int value_;
   Condition sig_;
   Mutex access_;

   void up()
     { 
      access_.lock();
      value_++;
      access_.unlock();
      sig_.signal();
     }
   void down()
     {
      access_.lock();
      while (value_<1)
        {sig_.wait(access_);}
      value_--;
      access_.unlock();
     }
   
   Semaphore(int value=1):value_(value) 
     {}
  };

// Private is a thread split static.  
template <class T>
struct Private
  {
   pthread_key_t key_;
   Mutex         access_;

   T& operator =(const T& t)
     {return (((T&)*this)=t);}

   operator T& ()
     {
      T *value;
#ifdef SIGC_PTHREAD_DCE
      pthread_getspecific(key_,(pthread_addr_t*)(&value));
#else
      value=(T*)(pthread_getspecific(key_));
#endif
      if (!value)
        pthread_setspecific(key_,(void*)(value=new T()));  
      return *(value);
     }

   static void dtor(void* v)
     {T* obj=(T*)v;
      delete obj;
     }

   Private()
     {
#ifdef SIGC_PTHREAD_DCE
      pthread_keycreate(&key_,dtor);
#else
      pthread_key_create(&key_,dtor);
#endif
     }
   ~Private()
     {
#ifndef SIGC_PTHREAD_DCE
      pthread_key_delete(key_);
#endif
     }
  };

// int needs to initialized
template <>
struct Private<int>
  {
   pthread_key_t key_;
   Mutex         access_;

   int& operator =(const int& t)
     {return (((int&)*this)=t);}

   operator int& ()
     {
      int *value;
#ifdef SIGC_PTHREAD_DCE
      pthread_getspecific(key_,(pthread_addr_t*)(&value));
#else
      value=(int*)(pthread_getspecific(key_));
#endif
      if (!value)
        pthread_setspecific(key_,(void*)(value=new int(0)));  
      return *(value); 
     }

   static void dtor(void* v)
     {int* obj=(int*)v;
      delete obj;
     }

   Private()
     {
#ifdef SIGC_PTHREAD_DCE
      pthread_keycreate(&key_,dtor);
#else
      pthread_key_create(&key_,dtor);
#endif
     }
   ~Private()
     {
#ifndef SIGC_PTHREAD_DCE
      pthread_key_delete(key_);
#endif
     }
  };

struct Thread
  {
   protected:
     pthread_t thread_;
     void*     arg_;
     ThreadAttr attr_;

     static void* call_main_(void* obj)
       {
        Thread *thread=(Thread*)obj;
        return thread->main(thread->arg_);
       }
   public:
     operator pthread_t* () {return &thread_;}

     virtual void* main(void*)=0;
#ifdef SIGC_PTHREAD_DCE
     int detach() {return pthread_detach(&thread_);}
#else
     int detach() {return pthread_detach(thread_);}
#endif
     static ThreadAttr Default;

     // arg is for passing extra data to main, but never pass a
     // local variable or address of local variable.  Arg must
     // be available throughout life of program.
     int start(void* arg=NULL)
       {
        arg_=arg;
        Thread *t=this;
        return pthread_create(&thread_,attr_,call_main_,t);
       }
     
     Thread(const ThreadAttr &attr=Default):attr_(attr)
       {}
     virtual ~Thread();
  };


#ifdef SIGC_CXX_NAMESPACES
};
};
#endif 

#endif /* SIGC_PTHREADS */
#endif /* SIGCXX_THREAD_H */
