/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */

#include "BinaryExpr.h"
#include "parser.h"
#include "Bytecode.h"
#include "RCX_Cmd.h"


static RCX_VarCode GetBinaryCode(int op);

static RCX_VarCode GetBinaryCode(int op)
{
	switch(op)
	{
		case '+':
			return kRCX_AddVar;
		case '-':
			return kRCX_SubVar;
		case '*':
			return kRCX_MulVar;
		case '/':
			return kRCX_DivVar;
		case '&':
			return kRCX_AndVar;
		case '|':
			return kRCX_OrVar;
		default:
			return kRCX_IllegalVar;
	}
}


BinaryExpr::BinaryExpr(Expr *lhs, int op, Expr *rhs)
	: fLeft(lhs), fRight(rhs), fOp(op)
{
}


BinaryExpr::~BinaryExpr()
{
	delete fLeft;
	delete fRight;
}


BinaryExpr* BinaryExpr::Clone(Mapping *b) const
{
	return new BinaryExpr(fLeft->Clone(b), fOp, fRight->Clone(b));
}


bool BinaryExpr::PromiseConstant() const
{
	return fLeft->PromiseConstant() && fRight->PromiseConstant();
}


bool BinaryExpr::Evaluate(int &value) const
{
	int v1, v2;
	
	if (!fLeft->Evaluate(v1)) return false;
	if (!fRight->Evaluate(v2)) return false;
	
	switch(fOp)
	{
		case '+':
			value = v1 + v2;
			break;
		case '-':
			value = v1 - v2;
			break;
		case '*':
			value = v1 * v2;
			break;
		case '/':
			value = v1 / v2;
			break;
		case '&':
			value = v1 & v2;
			break;
		case '|':
			value = v1 | v2;
			break;
		case '%':
			value = v1 % v2;
			break;
		case LEFT:
			value =	v1 << v2;
			break;
		case RIGHT:
			value = v1 >> v2;
			break;
		case '^':
			value = v1 ^ v2;
			break;
		default:
			return false;

	}
	return true;
}


bool BinaryExpr::Contains(int var) const
{
	return fLeft->Contains(var) || fRight->Contains(var);
}


RCX_Value BinaryExpr::EmitAny_(Bytecode &b, const LexLocation *loc) const
{
	RCX_Cmd cmd;	
	RCX_Value ea;
	RCX_Value dstEA;
	int dst;
	RCX_VarCode code;
	
	ea = fLeft->EmitAny(b, loc);
	if (ea == kIllegalEA) return ea;

	if (b.IsTempEA(ea))
		dst = RCX_VALUE_DATA(ea);
	else
	{
		dst = GetTempVar(b, loc);
		if (dst < 0)
			return kIllegalEA;
		
		b.AddMove(dst, ea);
	}
	
	code = GetBinaryCode(fOp);
	dstEA = RCX_VALUE(kRCX_VariableType, dst);
	ea = fRight->EmitConstrained(b, CODE_MASK(code), loc);
	if (ea == kIllegalEA)
	{
		b.ReleaseTempEA(dstEA);
		return ea;
	}

	cmd.MakeVar(code, dst, ea);
	b.Add(cmd);
	b.ReleaseTempEA(ea);
	
	return dstEA;
}


bool BinaryExpr::EmitTo_(Bytecode &b, int dst, const LexLocation *loc) const
{
	if (fRight->Contains(dst))
		return Expr::EmitTo_(b, dst, loc);
	else
	{
		fLeft->EmitTo(b, dst, loc);
		
		RCX_Cmd cmd;
		RCX_VarCode	code = GetBinaryCode(fOp);
		RCX_Value ea = fRight->EmitConstrained(b, CODE_MASK(code), loc);
		if (ea == kIllegalEA) return false;
		
		cmd.MakeVar(code, dst, ea);
		b.Add(cmd);
		b.ReleaseTempEA(ea);
		return true;					
	}
}

bool BinaryExpr::EmitSide_(Bytecode &b, const struct LexLocation *loc) const
{
	return fLeft->EmitSide(b,loc) && fRight->EmitSide(b, loc);
}


bool BinaryExpr::NeedsConstant(int op)
{
	return GetBinaryCode(op) == kRCX_IllegalVar; 
}
