/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "RepeatStmt.h"
#include "Bytecode.h"
#include "CheckState.h"
#include "RCX_Cmd.h"
#include "Clause.h"
#include "Error.h"

#define REPEAT_MASK	(TYPEMASK(kRCX_VariableType) + \
					TYPEMASK(kRCX_ConstantType) + \
					TYPEMASK(kRCX_RandomType))

RepeatStmt::RepeatStmt(Clause *c, Stmt *s)
{
	fCount = c;
	fBody = s;
}


RepeatStmt::~RepeatStmt()
{
	delete fCount;
	delete fBody;
}


bool RepeatStmt::Check(CheckState &state)
{
	bool ok = true;
	
	state.fLoopDepth++;
	
	if (!fBody->Check(state)) ok = false;
	
	if (fBody->IsNullable())
		fNullable = true;
	else
	{
		const Expr *e = fCount->GetExpr();
		int n;
		if (e->Evaluate(n))
		{
			if (n==0)
				fNullable = true;
			else if (n< 0 || n >255)
			{
				Error(kErr_BadRepeatCount).Raise(fCount->GetLoc());
				ok = false;
			}
		}
	}

	state.fLoopDepth--;
	
	return ok;
}


void RepeatStmt::Emit(Bytecode &b)
{
	/*
			set loop count
		cPos:
			check loop
			body
			jump -> cPos
	*/

	RCX_Cmd cmd;
	int cPos;
	RCX_Value ea;
	
	b.PushLoopContext();
	
	ea = fCount->EmitConstrained(b, REPEAT_MASK);
	cmd.MakeSetLoop(ea);
	b.Add(cmd);
	b.ReleaseTempEA(ea);
	
	cPos = b.GetLength();
	
	cmd.MakeCheckLoop(0);
	b.Add(cmd);
	b.AddFixup(kPatch_Normal, b.GetLength() - 2, b.GetBreakLabel());

	fBody->Emit(b);
	b.AddJump(b.GetContinueLabel());
	
	b.PopLoopContext(cPos, b.GetLength());
}


RepeatStmt* RepeatStmt::Clone(Mapping *b) const
{
	return new RepeatStmt(fCount->Clone(b), fBody->Clone(b));
}
