/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include "RCX_Disasm.h"
#include "RCX_Constants.h"

static void SPrintOutputArgs(char *argText, const UByte *code);
static void SPrintValue(char *value, int type, short data);
static void SPrintTest(char *text, const UByte *code, UShort pc);

static int ComputeOffset(UByte b1, UByte b2=0);

#define LOOKUP(i,a)	(((unsigned)(i)<sizeof(a)/sizeof(char*)) ? a[i] : "?")
#define WORD(ptr)	((short)((((ptr)[1]) << 8) + ((ptr)[0])))

static const char *inputTypeNames[] = { "None", "Switch", "Temp", "Light", "Angle" };
static const char *inputModeNames[] = { "Raw", "Boolean", "Edge", "Pulse", "Percent", "Celcius", "Fahrenheit", "Angle" };
static const char *outputDirName[] = { "Rev", "Flip", "Fwd" };
static const char *outputModeName[] = { "Float", "Off", "On" };
static const char *relNames[] = { "<=", ">=", "!=", "==" };
static const char *typeNames[] = {
	"Var", "Timer", "Const", "%3", "Random", "TachoCount", "TachoSpeed", "MotorCurrent",
	"%8", "Input", "%10", "%11", "%12", "%13", "Watch", "Message",
	"AGC" };

RCX_Disasm gRCX_Disasm;

class Instruction
{
public:
	const char*	fName;
	UByte		fOpcode;
	UByte		fLength;
};


static Instruction sInitData[] = {
{ "OutMode", kRCX_OutputModeOp, 2 },
{ "OutPwr", kRCX_OutputPowerOp, 4 },
{ "OutDir", kRCX_OutputDirOp, 2 },
{ "InMode", kRCX_InputModeOp, 3 },
{ "InType", kRCX_InputTypeOp, 3 },
{ "Delay", kRCX_DelayOp, 4 },
{ "Test", kRCX_TestOp, 8 },
{ "Jump", kRCX_JumpOp, 3},
{ "Jump", kRCX_SJumpOp, 2},
{ "PlayTone", kRCX_PlayToneOp, 4},
{ "PlaySound", kRCX_PlaySoundOp, 2},
{ "Display", kRCX_DisplayOp, 4},
{ "Set", kRCX_VarOp(kRCX_SetVar), 5},
{ "Add", kRCX_VarOp(kRCX_AddVar), 5},
{ "Sub", kRCX_VarOp(kRCX_SubVar), 5},
{ "Div", kRCX_VarOp(kRCX_DivVar), 5},
{ "Mul", kRCX_VarOp(kRCX_MulVar), 5},
{ "And", kRCX_VarOp(kRCX_AndVar), 5},
{ "Or", kRCX_VarOp(kRCX_OrVar), 5},
{ "Abs", kRCX_VarOp(kRCX_AbsVar), 5},
{ "Sgn", kRCX_VarOp(kRCX_SgnVar), 5},
{ "StartTask", kRCX_StartTaskOp, 2},
{ "StopTask", kRCX_StopTaskOp, 2},
{ "StopAll", kRCX_StopAllOp, 1},
{ "ClearTimer", kRCX_ClearTimerOp, 2},
{ "GoSub", kRCX_GoSubOp, 2},
{ "SetLoop", kRCX_SetLoopOp, 3 },
{ "CheckLoop", kRCX_CheckLoopOp, 3 },
{ "ClearMsg", kRCX_ClearMsgOp, 1 },
{ "SendMsg", kRCX_SendMsgOp, 3 },
{ "ClearSensor", kRCX_ClearSensorOp, 2},
{ "SetLog", kRCX_SetDatalogOp, 3},
{ "Datalog", kRCX_DatalogOp, 3},
{ "UploadLog", kRCX_UploadDatalogOp, 5},
{ "Drive", kRCX_DriveOp, 2 },
{ "OnWait", kRCX_OnWaitOp, 3 },
{ "OnWaitDifferent", kRCX_OnWaitDifferentOp, 4 },
{ "ClearTacho", kRCX_ClearTachoOp, 2 },
{ "SetWatch", kRCX_SetWatchOp, 3},
{ "SetTxPower", kRCX_IRModeOp, 2}
};

static Instruction *sDispatch[256] = { 0};
static int sInited = 0;

void InitDispatch();

RCX_Disasm::RCX_Disasm()
{
	InitDispatch();
}


void RCX_Disasm::Print(RCX_Printer *dst, const UByte *code, int length)
{
	char text[256];
	char line[256];
	UShort pc = 0;
	RCX_Result result;
	
	while(length > 0)
	{
		result = SPrint1(text, code, length, pc);
		
		if (result < 1)
		{
			result = 1;
			sprintf(text, "?");
		}

		sprintf(line, "%03d %-32s ", pc, text);
		dst->Print(line);
		
		for(int i=0; i<result; i++)
		{
			sprintf(line, "%02x ", code[i]);
			dst->Print(line);
		}
		sprintf(line, "\n");
		dst->Print(line);
		
		pc += result;
		code += result;
		length -= result;
	}
}	


RCX_Result RCX_Disasm::SPrint1(char *text, const UByte *code, int length, UShort pc)
{
	int iLength;
	Instruction *inst;
	char argText[256];
	UByte op;
	
	InitDispatch();
	
	if (length < 1) return -1;
	
	op = code[0];
	inst = sDispatch[op];
	if (!inst) return -1;
	
	iLength = inst->fLength;
	if (length < iLength) return -1;

	// process args
	switch(op & 0xf7)
	{
		case kRCX_InputTypeOp:
			sprintf(argText,"%d, %s", code[1],  LOOKUP(code[2], inputTypeNames));
			break;
		case kRCX_InputModeOp:
			sprintf(argText,"%d, %s", code[1],  LOOKUP((code[2] >> 5) & 7, inputModeNames));
			break;
		case kRCX_OutputDirOp:
		case kRCX_OutputModeOp:
		case kRCX_OutputPowerOp:
			SPrintOutputArgs(argText, code);
			break;
		case kRCX_DelayOp:
		case kRCX_DisplayOp:
			// 16 bit value
			SPrintValue(argText, code[1], WORD(code+2));
			break;
		case kRCX_TestOp:
			SPrintTest(argText, code, pc);
			break;
		case kRCX_JumpOp:
			sprintf(argText,"%d", pc + 1 + ComputeOffset(code[1], code[2])); 
			break;
		case kRCX_CheckLoopOp:
			sprintf(argText,"%d", pc + 1 + (short)WORD(code+1));
			break;
		case kRCX_SJumpOp:
			sprintf(argText,"%d", pc + 1 + ComputeOffset(code[1])); 
			break;
		case kRCX_PlayToneOp:
			sprintf(argText,"%d, %d", code[1] + (code[2] << 8), code[3]);
			break;
		case kRCX_PlaySoundOp:
		case kRCX_StartTaskOp:
		case kRCX_StopTaskOp:
		case kRCX_ClearTimerOp:
		case kRCX_GoSubOp:
		case kRCX_ClearSensorOp:
		case kRCX_IRModeOp:
			// 8 bit raw
			sprintf(argText, "%d", code[1]);
			break;
		case kRCX_VarOp(kRCX_SetVar):
		case kRCX_VarOp(kRCX_AddVar):
		case kRCX_VarOp(kRCX_SubVar):
		case kRCX_VarOp(kRCX_MulVar):
		case kRCX_VarOp(kRCX_DivVar):
		case kRCX_VarOp(kRCX_AbsVar):
		case kRCX_VarOp(kRCX_SgnVar):
		case kRCX_VarOp(kRCX_AndVar):
		case kRCX_VarOp(kRCX_OrVar):
			sprintf(argText, "var[%d], ", code[1]);
			SPrintValue(argText + strlen(argText), code[2], WORD(code+3));
			break;
		case kRCX_SendMsgOp:
		case kRCX_SetLoopOp:
		case kRCX_DatalogOp:
			// 8 bit value
			SPrintValue(argText, code[1], code[2]);
			break;
		case kRCX_SetDatalogOp:
			// 16 bit raw
			sprintf(argText, "%d", WORD(code+1));
			break;
		case kRCX_UploadDatalogOp:
			sprintf(argText, "%d, %d", WORD(code+1), WORD(code+3));
			break;
		case kRCX_SetWatchOp:
			sprintf(argText, "%02d:%02d", code[1], code[2]);
			break;
		default:
			argText[0] = 0;
			break;
	}
	
	sprintf(text,"%-10s %s", inst->fName, argText);

	return iLength;
}


void InitDispatch()
{
	unsigned i;
	
	if (sInited) return;
	
	for(i=0; i < sizeof(sInitData) / sizeof(Instruction); i++)
	{
		sDispatch[sInitData[i].fOpcode ^ 0x8] = &sInitData[i];
		sDispatch[sInitData[i].fOpcode] = &sInitData[i];
	}
	
	sInited = 1;
}



void SPrintOutputArgs(char *argText, const UByte *code)
{
	char *ptr = argText;
	
	// list outputs
	for(int i=0; i<3; i++)
		if (code[1] & (1 << i))
			*ptr++ = (char)('A' + i);
	*ptr++ = ',';
	*ptr++ = ' ';
	
	switch(code[0])
	{
		case kRCX_OutputDirOp:
			strcpy(ptr, LOOKUP((code[1] >> 6) & 3, outputDirName));
			break;
		case kRCX_OutputModeOp:
			strcpy(ptr, LOOKUP((code[1] >> 6) & 3, outputModeName));
			break;
		case kRCX_OutputPowerOp:
			SPrintValue(ptr, code[2], code[3]);
			break;
		default:
			*ptr = 0;
			break;
	}
}



void SPrintValue(char *text, int type, short data)
{
	switch(type)
	{
		case 0:
			sprintf(text, "var[%d]", data);
			break;
		case 2:
			sprintf(text,"%d", data);
			break;
		default:
			sprintf(text, "%s(%d)", LOOKUP(type, typeNames), data);
			break;
	}
}


void SPrintTest(char *text, const UByte *code, UShort pc)
{
	char v1Text[16];
	char v2Text[16];
	short offset;
	
	offset = (short)((code[7] << 8) + code[6]);
		
	SPrintValue(v1Text, (code[1] & 0xf), (short)((code[4]<<8) + code[3]));
	SPrintValue(v2Text, (code[2] & 0xf), code[5]); 
	sprintf(text, "%s %s %s, %d", v1Text, LOOKUP((code[1]>>6) & 3, relNames),
		v2Text, pc + 6 + offset);
}


int ComputeOffset(UByte b1, UByte b2)
{
	int x = (b1 & 0x7f) + (b2 << 7);
	
	if (b1 & 0x80) x = -x;
	
	return x;
}
