/*
 * MLMATH.C - random math routines which are missing in some
 *          - libraries or are just too useful not to have
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pml.h"

#ifdef MAC
#define BESSEL_JN
#define BESSEL_YN
#endif

#ifdef TURBOC
#define BESSEL_JN
#define BESSEL_YN
#endif

#define BESS_ACC 40.0

#ifdef PCK_COMPLEX

double
 cx_ttr = 0.0,
 cx_tti = 0.0,
 cx_ttm = 0.0;

complex
 cx_reg;

#endif

/* you cannot trust vendors to handle these declarations in <math.h>
 * some regard them as special extensions which must be specifically
 * asked for - phooey
 */

double
 SC_DECLARE(j0, (double x)),
 SC_DECLARE(j1, (double x)),
 SC_DECLARE(jn, (int n, double x)),
 SC_DECLARE(y0, (double x)),
 SC_DECLARE(y1, (double x)),
 SC_DECLARE(yn, (int n, double x));

int
 SC_DECLARE(*PM_fft_index, (int n)),
 SC_DECLARE(_PM_fold, 
            (int sgn, int na, REAL *xa, REAL *ya, int nb,
             REAL *xb, REAL *yb, REAL **pxf, REAL **pyf));

double
 SC_DECLARE(_PM_jn, (int n, double x)),
 SC_DECLARE(_PM_yn, (int n, double x));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SET_OPERS - set the opers member of the set to correspond to
 *              - the element type
 *              - return TRUE is there is a known type of PM_field
 *              - to associate with the set
 *              - return FALSE otherwise
 *              - NOTE: so far this will work only for some standard
 *              -       types
 */

int PM_set_opers(set)
   PM_set *set;
   {int ret;
    char type[MAXLINE], *token, *s = NULL;

    if (set == NULL)
       return(FALSE);

    ret = TRUE;
    strcpy(type, set->element_type);
    token = SC_strtok(type, " (*", s);
    if (strcmp(token, "double") == 0)
       set->opers = PM_REAL_Opers;
    else if (strncmp(token, "int", 3) == 0)
       set->opers = PM_Int_Opers;
    else if (strcmp(token, "long") == 0)
       set->opers = PM_Long_Opers;
    else
       ret = FALSE;

    return(ret);}

/*--------------------------------------------------------------------------*/

/*                              ARRAY FUNCTIONS                             */

/*--------------------------------------------------------------------------*/

/* PM_SCALE_ARRAY - multiply all of the elements of an array
 *                - by a conversion factor
 */

void PM_scale_array(p, n, f)
   double *p;
   int n;
   double f;
   {int i;

    if (p != NULL)
       for (i = 0; i < n; i++)
           *(p++) *= f;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SHIFT_ARRAY - add a delta to all of the elements of an array */

void PM_shift_array(p, n, f)
   double *p;
   int n;
   double f;
   {int i;

    if (p != NULL)
       for (i = 0; i < n; i++)
           *(p++) += f;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SET_VALUE - broadcast a value into an array */

void PM_set_value(p, n, f)
   REAL *p;
   int n;
   double f;
   {int i;

    if (p != NULL)
       for (i = 0; i < n; i++)
           *(p++) = f;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_COPY_ARRAY - copy one array to another
 *               - that is copy N double values from array T to array S
 */

void PM_copy_array(s, t, n)
   REAL *s, *t;
   int n;
   {int i;

    if ((s != NULL) && (t != NULL))
       for (i = 0; i < n; i++)
           *(s++) = *(t++);

    return;}

/*--------------------------------------------------------------------------*/

/*                               MATH FUNCTIONS                             */

/*--------------------------------------------------------------------------*/
 
/* PM_SQR - fast version of sqr? */
 
double PM_sqr(x)
   double x;
   {return(x*x);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SQRT - a safe square root */

double PM_sqrt(x)
   double x;
   {return(sqrt(ABS(x)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_ATAN - return the arc tangent of Y/X in radians
 *         - using the signs of X and Y to determine the quadrant
 *         - the range is -PI < ang <= PI
 */

double PM_atan(x, y)
   double x, y;
   {double a, r;

    if ((x == 0.0) && (y == 0.0))
       a = 0.0;

    else if (PM_LESS_LESS(x, y))
       a = (y > 0.0) ? PI/2.0 : -PI/2.0;

    else if (PM_LESS_LESS(y, x))
       a = (x > 0.0) ? 0.0 : PI;

    else
       {r = y/x;
	a = atan(r);

	if (x < 0.0)
           a += (r > 0.0) ? -PI : PI;};

    return(a);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_RECIP - a callable, safe inverse function */

double PM_recip(x)
   double x;
   {if (ABS(x) > SMALL)
       return(1/x);
    else
       return(HUGE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_LN - safe natural logarithm */

double PM_ln(x)
   double x;
   {if (x > 0)
       return (log(x));
    else
       return(-HUGE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_LOG - safe logarithm base 10 */

double PM_log(x)
   double x;
   {if (x > 0)
       return(log10(x));
    else
       return(-HUGE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PM_POW - smarter verson of pow */
 
double PM_pow(x, y)
   double x;
   double y;
   {if (y == 1.0)
       return(x);
    else
       return(pow(ABS(x), y));}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PM_POWER - fast version of pow? */
 
double PM_power(x, p)
   double x;
   int p;
   {double y;
 
    if (p == 0)
       return(1.0);
    if (p & 1)                                             /* check for odd */
       {if (p > 0)
           y = x*PM_power(x, p-1);
        else
           y = PM_power(x, p+1)/x;}
    else
       {y = PM_sqr(PM_power(x, (p >> 1)));};
 
    return(y);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PM_SIGN - return the sign */
 
double PM_sign(x)
   double x;
   {if (x != 0)
       return(ABS(x)/x);
    else
       return(1);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PM_ROUND - round off the given number */
 
double PM_round(x)
   double x;
   {return(PM_sign(x)*floor(ABS(x)+0.5));}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PM_FRAC - return the fractional part of the given number */
 
double PM_frac(x)
   double x;
   {double frac = 1.0;
 
    return(fmod(x, frac));}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_FIX - truncates the fractional part of a double
 *        - the difference between this function and the C library's FLOOR is
 *        -
 *        - floor(-2.3)  = -3.0
 *        - PM_fix(-2.3) = -2.0
 */

double PM_fix(value)
   double value;
   {long ivalue;
    double rvalue;
    static double maxlv = (double) LONG_MAX;

    if (value < 0.0)
       {value  = max(value, -maxlv);
        ivalue = (long) ABS(value);
        rvalue = -((double) ivalue);}
    else
       {value  = min(value, maxlv);
        ivalue = (long) value;
        rvalue = (double) ivalue;};

    return(rvalue);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SGN - transfers the sign of sign to value */

double PM_sgn(value, sign)
   double sign, value;
   {value = ABS(value);
    if (sign < 0.0)
       value = -value;

    return((double) value);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_HYPOT - return the root mean square value of the arguments */

double PM_hypot(x, y)
   double x, y;
   {return(sqrt(x*x + y*y));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_CURVE_LEN_2D - return the length of the given curve in 2 space */

double PM_curve_len_2d(x, y, n)
   REAL *x, *y;
   int n;
   {int i;
    double s;
    REAL *px1, *px2, *py1, *py2;

    px1 = x;
    px2 = x + 1;
    py1 = y;
    py2 = y + 1;
    for (i = 1, s = 0.0; i < n; i++)
        s += PM_hypot(*px2++ - *px1++, *py2++ - *py1++);

    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_CURVE_LEN_3D - return the length of the given curve in 3 space */

double PM_curve_len_3d(x, y, z, n)
   REAL *x, *y, *z;
   int n;
   {int i;
    double dx, dy, dz, s;
    REAL *px1, *px2, *py1, *py2, *pz1, *pz2;

    px1 = x;
    px2 = x + 1;
    py1 = y;
    py2 = y + 1;
    pz1 = z;
    pz2 = z + 1;
    for (i = 1, s = 0.0; i < n; i++)
        {dx = *px2++ - *px1++;
         dy = *py2++ - *py1++;
         dz = *pz2++ - *pz1++;
	 s += sqrt(dx*dx + dy*dx + dz*dz);};

    return(s);}

/*--------------------------------------------------------------------------*/

/*                          ADVANCED MATH FUNCTIONS                         */

/*--------------------------------------------------------------------------*/

/* PM_ROMBERG - Romberg integration routine
 *            - integrates func(x) from x0 to x1 to accuracy of tol
 *            - returned as y0.  Returns number of iterations
 *            - in n (2^(n-1) subdivisions) or -1 if integral has not
 *            - converged within 16 iterations (32768 subdivisions)
 */

double PM_romberg(func, x0, x1, tol)
   PFDouble func;
   double x0, x1, tol;
   {double y0, toln, h, x;
    double a[15], b[15];
    int i, n, in;

    h = x1 - x0;
    x = x0;
    a[0] = h*((*func)(x0) + (*func)(x1))/2.0;
    toln = 2.0*tol;
    for (n = 2; (n < 16) || (toln > tol); n++)
        {h    = h/2.0;
         b[0] = a[0]/2.0;
         x    = x0 + h;
         in   = (int) POW(2.0, (n-2));
         for (i = 0; i < in; i++)
             {b[0] += h*func(x);
              x = x + 2.0*h;};

         for (i = 1; i < n; i++)
             b[i] = (POW(4.0, i)*b[i-1]-a[i-1])/(POW(4.0, i) - 1.0);

         y0   = b[n-1];

         toln = ABS((y0 - b[n-2])/y0);
         if (toln > tol)
            for (i = 0; i < n; i++)
                a[i] = b[i];};

      return(y0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_INTEGRATE_TZR - do simple trapezoid rule integeration of a data set */

double PM_integrate_tzr(xmn, xmx, pn, fncx, fncy, intx, inty)
   double xmn, xmx;
   int *pn;
   REAL *fncx, *fncy, *intx, *inty;
   {int i, k, n;
    REAL sum, lasty;
    REAL xta, xtb, xtc, yta, ytb;

    n   = *pn;
    i   = 0;
    sum = 0;

    xta = fncx[n-1];
    xmx = min(xmx, xta);

/* first point */
    if (xmn <= *fncx)
       {lasty   = *fncy;
        intx[0] = *fncx;

        i++;}

    else
       {for (; (fncx[i] < xmn) && (i < n); ++i);
        PM_interp(lasty, xmn, fncx[i-1], fncy[i-1], fncx[i], fncy[i]);

        intx[0] = xmn;};

    inty[0] = sum;

/* interior */
    for (k = 1; (i < (n - 1)) && (fncx[i] < xmx); i++, k++)
       {sum += 0.5*(fncy[i] + lasty)*(fncx[i] - intx[k-1]);

        inty[k] = sum;
        intx[k] = fncx[i];

        lasty   = fncy[i];};

/* last point */
    if (i < n)
       {PM_interp(inty[k], xmx, fncx[i-1], fncy[i-1], fncx[i], fncy[i]);
        sum += 0.5*(inty[k] + lasty)*(xmx - intx[k-1]);

        inty[k] = sum;
        intx[k] = xmx;

        k++;};

    *pn = k;

    return((double) sum);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_DERIVATIVE - take df/dx from fncx and fnc and put the values in
 *               - derx and dery
 */

void PM_derivative(n, fncx, fncy, derx, dery)
   int n;
   REAL *fncx, *fncy;
   REAL *derx, *dery;
   {int i;
    REAL x1, x2, y1, y2, yt;

    x1 = *fncx++;
    y1 = *fncy++;
    for (i = 1; i < n; i++)
        {x2 = *fncx++;
         y2 = *fncy++;
         *derx++ = 0.5*(x1 + x2);
         if (x2 != x1)
            *dery++ = (y2 - y1)/(x2 - x1);
         else
	    {yt = dery[-1];
	     *dery++ = yt;};
         x1 = x2;
         y1 = y2;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_THIN_1D_DER - take a 1d data set and return a new set of points which
 *                - captures the essential features of the original set
 *                - but with fewer points
 *                - this uses a differential measure criterion
 */

int PM_thin_1d_der(n, fncx, fncy, thnx, thny, toler)
   int n;
   REAL *fncx, *fncy;
   REAL *thnx, *thny;
   double toler;
   {int i, i0, j;
    REAL x1, x2, y1, y2, xt, dydx, odydx;

    if (toler == HUGE)
       {toler = 10.0/((double) n);
        toler = min(toler, 0.1);};

    x1 = *fncx++;
    y1 = *fncy++;
    x2 = x1;
    y2 = y1;

    j  = 0;
    i0 = -1;

    odydx = -HUGE;
    for (i = 1; i < n; i++)
        {x2 = *fncx++;
         y2 = *fncy++;

         if (x2 != x1)
            dydx = (y2 - y1)/(x2 - x1);
         else
	    dydx = y2 > y1 ? HUGE : -HUGE;

         xt = 0.5*ABS(dydx - odydx)/(ABS(dydx) + ABS(odydx) + SMALL);
         if (xt > toler)
            {if (i0 < i-1)
                {thnx[j] = x1;
                 thny[j] = y1;
                 j++;};

	     odydx = dydx;
             i0    = i;

             thnx[j] = x2;
             thny[j] = y2;
             j++;};

         x1 = x2;
         y1 = y2;};

    if (x2 != thnx[j-1])
       {thnx[j] = x2;
        thny[j] = y2;
        j++;};

    return(j);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_THIN_1D_INT - take a 1d data set and return a new set of points which
 *                - captures the essential features of the original set
 *                - but with fewer points
 *                - this uses an integral measure criterion
 */

int PM_thin_1d_int(n, fncx, fncy, thnx, thny, toler)
   int n;
   REAL *fncx, *fncy;
   REAL *thnx, *thny;
   double toler;
   {int i, nth, nc, iamn, iamna, iamnb;
    REAL xc, yc, sm, b, ht, amn;
    REAL *area;

    memcpy(thnx, fncx, n*sizeof(REAL));
    memcpy(thny, fncy, n*sizeof(REAL));

    nth = -1;
    xc  = 0.0;

/* compute the areas */
    area = FMAKE_N(REAL, n, "PM_THIN_1D_INT:area");
    for (i = 1; i < n-1; i++)
        {xc = thnx[i+1] - thnx[i-1];
         if (xc == 0.0)
            break;

         yc = thny[i+1] - thny[i-1];
         sm = yc/xc;
         b  = thny[i+1] - sm*thnx[i+1];
         ht = ABS(thny[i] - sm*thnx[i] - b)/sqrt(sm*sm + 1.0);

         area[i] = 0.5*ht*HYPOT(xc, yc);};

/* if there were no bad points proceed */
    if (xc != 0.0)
       {nth = (int) toler;
        for (nc = n; nth < nc; nc--)

/* find the smallest area */
            {iamn = 0;
             amn  = HUGE;
             for (i = 1; i < nc-1; i++)
                 if (area[i] < amn)
                    {amn  = area[i];
                     iamn = i;};
 
/* compress out point there */
             for (i = iamn; i < nc-1; i++)
                 {thnx[i] = thnx[i+1];
                  thny[i] = thny[i+1];
                  area[i] = area[i+1];};

/* recompute the areas around that point */
	     iamna = iamn - 1;
	     iamna = max(iamna, 1);
	     iamnb = iamn + 1;
	     iamnb = min(iamnb, nc-2);
             for (i = iamna; i < iamnb; i++)
                 {xc = thnx[i+1] - thnx[i-1];
                  yc = thny[i+1] - thny[i-1];
                  sm = yc/xc;
                  b  = thny[i+1] - sm*thnx[i+1];
                  ht = ABS(thny[i] - sm*thnx[i] - b)/sqrt(sm*sm + 1.0);
                  area[i] = 0.5*ht*HYPOT(xc, yc);};};};

    SFREE(area);

    return(nth);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_FILTER_COEFF - replace the y values with new ones obtained by
 *                 - combining the old ones according to a set of supplied
 *                 - constant coefficients
 *                 -
 *                 -   NC is the number of coefficients in the general case
 *                 -      (it must be odd)
 *                 -   COEF is an array of nct coefficients where:
 *                 -
 *                 -   nh = nc/2
 *                 -   nt = total number of coefficients
 *                 -      = nc*[1 + 2*nh*(nh+1)/3]
 *                 -
 *                 -   COEF is layed out as follows:
 *                 -   ni = i*nh + 0.5*i*(i+1)
 *                 -   mi = nt - i*nh - 0.5*i*(i+1)
 *                 -   np = ni for i = nh
 *                 -
 *                 -   ynew[0] = c[n0]*y[0] + ... + c[n1-1]*y[nh]
 *                 -   ynew[1] = c[n1]*y[0] + ... + c[n2-1]*y[nh+1]
 *                 -   ynew[2] = c[n3]*y[0] + ... + c[n3-1]*y[nh+2]
 *                 -                        .
 *                 -                        .
 *                 -
 *                 -   ynew[i] = c[np]*y[i-nh] + ... + c[np+nh-1]*y[i+nh]
 *                 -                        .
 *                 -                        .
 *                 -
 *                 -   ynew[n-3] = c[m3]*y[n-nh-3] + ... + c[m2-1]*y[n-1]
 *                 -   ynew[n-2] = c[m2]*y[n-nh-2] + ... + c[m1-1]*y[n-1]
 *                 -   ynew[n-1] = c[m1]*y[n-nh-1] + ... + c[m0-1]*y[n-1]
 *                 -
 *                 - return TRUE iff successful
 */

int PM_filter_coeff(y, n, coef, nc)
   REAL *y;
   int n;
   REAL *coef;
   int nc;
   {int i, j, nh, nm, nu, jn, jx;
    REAL *ny, *py, *pc, yn;

    nh = nc >> 1;
    nm = nh*(3*nh + 1) >> 1;
    nu = n - nh;

    ny = FMAKE_N(REAL, n, "PM_FILTER_COEFF:ny");

    jn = 0;
    jx = 0;
    for (i = 0; i < n; i++)
        {if (i < nh)
	    {jx = nh + i + 1;
             pc = coef + jn;
             py = y;
             jn += jx;}

         else if (i < nu)
	    {jx = nc;
             pc = coef + jn;
             py = y + i - nh;
	     jn = nm;}

	 else
            {jn += jx;
	     jx = nh + n - i;
             pc = coef + jn;
             py = y + n - jx;};

         yn = 0.0;
         for (j = 0; j < jx; j++)
             yn += (*py++)*(*pc++);

         ny[i] = yn;};

    memcpy(y, ny, n*sizeof(REAL));
    SFREE(ny);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SMOOTH_INT_AVE - smooth an array Y of N values using PTS points
 *                   - the algorithm does a replacement by integral averages
 *                   - return TRUE iff successful
 */

int PM_smooth_int_ave(x, y, n, pts)
   REAL *x, *y;
   int n, pts;
   {int i, j, jn, jx, nh, nt;
    REAL dx, Dx, ya, xo, xn, yo, yn;
    REAL *nx, *ny, *px, *py;

    pts = max(pts, 3);

    nh = pts >> 1;
    nt = n - 1;

    nx = FMAKE_N(REAL, n, "PM_SMOOTH_INT_AVE:nx");
    ny = FMAKE_N(REAL, n, "PM_SMOOTH_INT_AVE:ny");

    for (i = 0; i < n; i++)
        {jn = i - nh;
         jn = max(0, jn);

         jx = i + nh;
         jx = min(nt, jx);

         px = x + jn;
         xo = *px++;
         py = y + jn;
         yo = *py++;
         ya = 0.0;
         Dx = SMALL;
         for (j = jn; j < jx; j++)
             {xn  = *px++;
              dx  = ABS(xn - xo);
              Dx += dx;
              xo  = xn;

              yn  = *py++;
              ya += 0.5*(yn + yo)*dx;
              yo  = yn;};

         ny[i] = ya/Dx;
         nx[i] = ((jx - i)*x[jn] + (i - jn)*x[jx])/(jx - jn);};

    n *= sizeof(REAL);
    memcpy(x, nx, n);
    memcpy(y, ny, n);

    SFREE(nx);
    SFREE(ny);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SMOOTH_FFT - smooth an array Y of N values using PTS points
 *               - the algorithm uses FFT, filter, IFFT
 *               - return TRUE iff successful
 */

int PM_smooth_fft(x, y, n, pts, fnc)
   REAL *x, *y;
   int n, pts;
   DECLFPTR(byte, fnc, (complex *cx, int nt, double tol));
   {int i, j, nt, np, nh;
    REAL xa, xb, yb, ya, dt, idn;
    REAL xta, xtb, xtc, yta, ytb;
    REAL *xo;
    complex *cx;

/* remove the linear trend */
    idn = 1.0/(n - 1);
    xb  = 0.0;
    ya  = idn*y[0];
    yb  = idn*y[n-1];
    for (i = 0; i < n; i++)
        y[i] -= (ya*(n - 1 - i) + yb*i);

/* do the Fourier transform of the data */
    nt = PM_fft_sc_real_data(&cx, &xo, x, y, n, x[0], x[n-1], TRUE);

    nh = nt >> 1;

/* filter the transform */
    if (fnc != NULL)
       (*fnc)(cx, nt, (double) pts);

/* rearrange for inverse FFT */
    for (i = 0; i < nh; i++)
        {PM_COMPLEX_SWAP(cx[i], cx[nh+i]);};

/* do the inverse transform */
    PM_fft_sc_complex(cx, nt-1, -1);

/* interpolate the smoothed curve back onto the old mesh */
    np = nt - 1;
    dt = (x[n-1] - x[0])/((double) (np - 1));
    for (i = 0; i < n; i++)
        {for (j = 0, xa = x[0]; j < np; j++, xa = xb)
             {xb = xa + dt;
              if ((xa <= x[i]) && (x[i] < xb))
                 break;};
         PM_interp(y[i], x[i],
		   xa, PM_REAL_C(cx[j]),
		   xb, PM_REAL_C(cx[j+1]));};

/* restore the linear trend */
    for (i = 0; i < n; i++)
        y[i] += (ya*(n - 1 - i) + yb*i);

    SFREE(xo);
    SFREE(cx);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SMOOTH_FILTER - a generic smoothing filter for PM_smooth */

void PM_smooth_filter(z, n, pts)
   complex *z;
   int n;
   double pts;
   {int i, nh;
    REAL c0, m0, mlt;

    nh = n >> 1;
    c0 = pts/nh;
    c0 = c0*c0;
    m0 = 1.0/(1.0 - c0);

/* do the low pass filter */
    for (mlt = 1.0, i = 0; i <= nh; i++)
        {if (mlt != 0.0)
            {mlt = m0*(1.0 - c0*i*i);
             mlt = min(mlt, 1.0);
             if (mlt < 0.0)
                mlt = 0.0;
             z[nh+i] = PM_TIMES_RC(mlt, z[nh+i]);
             z[nh-i] = PM_TIMES_RC(mlt, z[nh-i]);}
         else
            {z[nh+i] = PM_COMPLEX(0.0, 0.0);
             z[nh-i] = PM_COMPLEX(0.0, 0.0);};};

    mlt = (1.0 - 0.25*pts*pts);
    if (mlt < 0.0)
       mlt = 0.0;
    PM_IMAGINARY_C(z[nh]) *= mlt;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_FOLD - fundamental kernel for convolution or correlation integrals
 *          - each curve is assumed to have the same fixed dx
 */

int _PM_fold(sgn, na, xa, ya, nb, xb, yb, pxf, pyf)
   int sgn, na;
   REAL *xa, *ya;
   int nb;
   REAL *xb, *yb, **pxf, **pyf;
   {int i, j, ja, jb, nf, nm, nmn, nt;
    REAL *xf, *yf, *yn;
    double Dxa, Dxb, dxa, dxb, xt, yt, xmn, xmx;

    Dxa = xa[na-1] - xa[0];
    Dxb = xb[nb-1] - xb[0];

/* quick feather count */
    dxa = Dxa/((double) (na - 1));
    dxb = Dxb/((double) (nb - 1));
    if (dxa != dxb)
       return(-1);

    if (sgn == 1)
       {xmn = xb[0] - xa[na - 1];
        xmx = xb[nb-1] - xa[0];}
    else
       {xmn = xa[0] - xb[nb-1];
        xmx = xa[na - 1] - xb[0];};

    nf = (xmx - xmn)/dxa + 1;
    xf = *pxf = FMAKE_N(REAL, nf, "_PM_FOLD:xf");
    yf = *pyf = FMAKE_N(REAL, nf, "_PM_FOLD:yf");

/* decide how many points at most are in the product function */
    nm  = 0;
    nmn = min(na, nb);
    for (xt = xmn, i = 0; i < nf; i++, xt += dxa)
        {if ((i < na) && (i < nb))
            nt = i;
         else if ((na <= i) && (nb <= i))
            nt = nf - 1 - i;
         else
            nt = nmn;

         nm = max(nm, nt);};

    yn = FMAKE_N(REAL, nm, "_PM_FOLD:yn");

    dxb *= 0.5;
    for (xt = xmn, i = 0; i < nf; i++, xt += dxa)
        {xf[i] = xt;

/* decide how many points are in the product function */
         if ((i < na) && (i < nb))
            nt = i;
         else if ((na <= i) && (nb <= i))
            nt = nf - 1 - i;
         else
            nt = nmn;

/* make the product function */
         if (sgn == 1)
            {ja = (i < (na - 1)) ? (na - 1 - i) : 0;
             jb = (i < na) ? 0 : i - na;}
         else
            {ja = (i < nb) ? 0 : i - nb;
             jb = (i < (nb - 1)) ? (nb - 1 - i) : 0;}

         for (j = 0; j < nt; j++)
             yn[j] = ya[j + ja]*yb[j + jb];

/* integrate the product function */
         yt = 0.0;
         for (j = 1; j < nt; j++)
             yt += dxb*(yn[j] + yn[j-1]);

/* stash the value */
         yf[i] = yt;};

    SFREE(yn);

    return(nf);}

/*--------------------------------------------------------------------------*/

/*                              SPECIAL FUNCTIONS                           */

/*--------------------------------------------------------------------------*/

#ifndef BESSEL_JN

/*--------------------------------------------------------------------------*/

/* PM_J0 - j0 */

double PM_j0(x)
   double x;
   {return(j0(x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_J1 - j1 */

double PM_j1(x)
   double x;
   {return(j1(x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_JN - Bessel function of 1rst kind */

double PM_jn(x, n)
   double x, n;
   {return(jn((int) n, x));}

/*--------------------------------------------------------------------------*/

#else

/*--------------------------------------------------------------------------*/

/* PM_J0 - j0 */

double PM_j0(x)
   double x;
   {return(_PM_jn(0, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_J1 - j1 */

double PM_j1(x)
   double x;
   {return(_PM_jn(1, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_JN - jn */

double PM_jn(x, n)
   double x, n;
   {return(_PM_jn(n, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_JN - Bessel function of the first kind primitive
 *        - using algorithm from Numerical Recipies in C
 */

double _PM_jn(n, x)
   int n;
   double x;
   {int j, jtot, m;
    double ax, z, bj, bjm, bjp, tot, tox, ret;
    double xx, y, num, den;

    if (n == 0)
       {if ((ax = ABS(x)) < 8.0)
           {y   = x*x;
            num = 57568490574.0 + y*(-13362590354.0
                                + y*(651619640.7
                                + y*(-11214424.18
                                + y*(77392.33017
                                + y*(-184.9052456)))));
            den = 57568490411.0 + y*(1029532985.0
                                + y*(9494680.718
                                + y*(59272.64853
                                + y*(267.8532712
                                + y*1.0))));
            ret = num/den;}
        else
           {z   = 8.0/ax;
            y   = z*z;
            xx  = ax - 0.785398164;
            num = 1.0 + y*(-0.1098628627e-2
                      + y*(0.2734510407e-4
                      + y*(-0.2073370639e-5
                      + y*0.2093887211e-6)));
            den = -0.1562499995e-1 + y*(0.1430488765e-3
                                   + y*(-0.6911147651e-5
                                   + y*(0.7621095161e-6
                                   - y*0.934935152e-7)));
            ret = sqrt(0.636619772/ax)*(cos(xx)*num - z*sin(xx)*den);};}
       
    else if (n == 1)
       {if ((ax = ABS(x)) < 8.0)
           {y   = x*x;
            num = x*(72362614232.0 + y*(-7895059235.0
                                   + y*(242396853.1
                                   + y*(-2972611.439
                                   + y*(15704.48260
                                   + y*(-30.16036606))))));
            den = 144725228442.0 + y*(2300535178.0
                                 + y*(18583304.74
                                 + y*(99447.43394
                                 + y*(376.9991397
                                 + y*1.0))));
            ret = num/den;}
        else
           {z   = 8.0/ax;
            y   = z*z;
            xx  = ax - 2.356194491;
            num = 1.0 + y*(0.183105e-2
                      + y*(-0.3516396496e-4
                      + y*(0.2457520174e-5
                      + y*(-0.240337019e-6))));
            den = 0.04687499995 + y*(-0.2002690873e-3
                                + y*(0.8449199096e-5
                                + y*(-0.88228987e-6
                                + y*0.105787412e-6)));
            ret = sqrt(0.636619772/ax)*(cos(xx)*num - z*sin(xx)*den);
            if (x < 0.0)
               ret = -ret;};}

    else
       {ax = ABS(x);
        if (ax == 0.0)
           return(0.0);

        else if (ax > (double) n)
           {tox = 2.0/ax;
            bjm = _PM_jn(0, ax);
            bj  = _PM_jn(1, ax);
            for (j = 1; j < n; j++)
                {bjp = j*tox*bj - bjm;
                 bjm = bj;
                 bj  = bjp;};
            ret = bj;}

        else
           {tox  = 2.0/ax;
            m    = 2*((n + ((int) sqrt(BESS_ACC*n))) >> 1);
            jtot = 0;
            bjp  = 0.0;
            ret  = 0.0;
            tot  = 0.0;
            bj   = 1.0;
            for (j = m; j > 0; j--)
                {bjm = j*tox*bj - bjp;
                 bjp = bj;
                 bj  = bjm;
                 if (ABS(bj) > HUGE)
                    {bj  *= SMALL;
                     bjp *= SMALL;
                     ret *= SMALL;
                     tot *= SMALL;};

                 if (jtot)
                    tot +=  bj;
                 jtot = !jtot;
                 if (j == n)
                    ret = bjp;};

            tot  = 2.0*tot - bj;
            ret /= tot;};

        ret = ((x < 0.0) && ((n % 2) == 1)) ? -ret : ret;};

    return(ret);}

/*--------------------------------------------------------------------------*/

#endif

#ifndef BESSEL_YN

/*--------------------------------------------------------------------------*/

/* PM_Y0 - y0 */

double PM_y0(x)
   double x;
   {return(y0(x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_Y1 - y1 */

double PM_y1(x)
   double x;
   {return(y1(x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_YN - Bessel function of 2nd kind */

double PM_yn(x, n)
   double x, n;
   {return(yn((int) n, x));}

/*--------------------------------------------------------------------------*/

#else

/*--------------------------------------------------------------------------*/

/* PM_Y0 - y0 */

double PM_y0(x)
   double x;
   {return(_PM_yn(0, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_Y1 - y1 */

double PM_y1(x)
   double x;
   {return(_PM_yn(1, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_YN - Bessel function of 2nd kind */

double PM_yn(x, n)
   double x, n;
   {return(_PM_yn((int) n, x));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_YN - Bessel function of the second kind
 *        - algorithm from Numerical Recipes in C
 */

double _PM_yn(n, x)
   int n;
   double x;
   {int j;
    double by, bym, byp, tox;
    double z;
    double xx, y, ret, num, den;

    if (n == 0)
       {if (x < 8.0)
           {y   = x*x;
            num = -2957821389.0 + y*(7062834065.0
                                + y*(-512359803.6
                                + y*(10879881.29
                                + y*(-86327.92757
                                + y*228.4622733))));
            den = 40076544269.0 + y*(745249964.8
                                + y*(7189466.438
                                + y*(47447.26470
                                + y*(226.1030244 + y))));
            ret = (num/den) + 0.636619772*_PM_jn(0, x)*log(x);}
        else
           {z   = 8.0/x;
            y   = z*z;
            xx  = x - 0.785398164;
            num = 1.0 + y*(-0.1098628627e-2
                      + y*(0.2734510407e-4
                      + y*(-0.2073370639e-5
                      + y*0.2093887211e-6)));
            den = -0.1562499995e-1 + y*(0.1430488765e-3
                                   + y*(-0.6911147651e-5
                                   + y*(0.7621095161e-6
                                   + y*(-0.934945152e-7))));
            ret = sqrt(0.636619772/x)*(sin(xx)*num + z*cos(xx)*den);};}

    else if (n == 1)
       {if (x < 8.0)
           {y   = x*x;
            num = x*(-0.4900604943e13 + y*(0.1275274390e13
                                      + y*(-0.5153438139e11
                                      + y*(0.7349264551e9
                                      + y*(-0.4237922726e7
                                      + y*0.8511937935e4)))));
            den = 0.2499580570e14 + y*(0.4244419664e12
                                  + y*(0.3733650367e10
                                  + y*(0.2245904002e8
                                  + y*(0.1020426050e6
                                  + y*(0.3549632885e3+y)))));
            ret = (num/den) + 0.636619772*(_PM_jn(1, x)*log(x) - 1.0/x);}
        else
           {z   = 8.0/x;
            y   = z*z;
            xx  = x-2.356194491;
            num = 1.0 + y*(0.183105e-2
                      + y*(-0.3516396496e-4
                      + y*(0.2457520174e-5
                      + y*(-0.240337019e-6))));
            den = 0.04687499995 + y*(-0.2002690873e-3
                                + y*(0.8449199096e-5
                                + y*(-0.88228987e-6
                                + y*0.105787412e-6)));
            ret = sqrt(0.636619772/x)*(sin(xx)*num + z*cos(xx)*den);};}

    else
       {tox = 2.0/x;
        by  = _PM_yn(1, x);
        bym = _PM_yn(0, x);
        for (j = 1; j < n; j++)
            {byp = j*tox*by - bym;
             bym = by;
             by  = byp;};

        ret = by;};

    return(ret);}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* PM_TCHN - Tchebyshev polynomial of order N */

double PM_tchn(x, n)
   double x, n;
   {int i, m;
    double ta, tb, tc;

    if ((x < -1.0) || (1.0 < x))
       tc = -HUGE;

    else
       {ta = 1.0;
	tb = x;
	m  = (int) n;

	switch (m)
	   {case 0 :
	         tc = ta;
		 break;

	    case 1 :
	         tc = tb;
		 break;

	    default :
	         tc = -HUGE;
	         for (i = m-1; i > 0; i--)
		     {tc = 2*x*tb - ta;
		      ta = tb;
		      tb = tc;};
		 break;};};

    return(tc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
