/*
 * SHMM.C - memory management routines for Scheme
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "scheme.h"

#define OMSB

#ifdef OMSB

static object
 *_SS_free_object_list = NULL;

static object
 **_SS_object_list = NULL;

static long
 _SS_n_objects = 0L,
 _SS_n_objects_max = 0L;

#endif

#ifdef LARGE

static void
 SC_DECLARE(_SS_rl_char, (object *obj)),
 SC_DECLARE(_SS_rl_vector, (object *obj)),
 SC_DECLARE(_SS_wr_vector, (object *obj, object *strm));

#endif

static void
 SC_DECLARE(_SS_rl_integer, (object *obj)),
 SC_DECLARE(_SS_rl_float, (object *obj)),
 SC_DECLARE(_SS_rl_string, (object *obj)),
 SC_DECLARE(_SS_rl_variable, (object *obj)),
 SC_DECLARE(_SS_rl_cons, (object *obj)),
 SC_DECLARE(_SS_rl_procedure, (object *obj)),
 SC_DECLARE(_SS_rl_inport, (object *obj)),
 SC_DECLARE(_SS_rl_outport, (object *obj)),
 SC_DECLARE(_SS_rl_boolean, (object *obj)),
 SC_DECLARE(_SS_wr_inport, (object *obj, object *strm)),
 SC_DECLARE(_SS_wr_outport, (object *obj, object *strm));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_INSTALL - install procedure objects in the symbol table
 *            - a procedure object is a struct
 *            -      struct s_SS_proc
 *            -         {char type;
 *            -          char *doc;
 *            -          char *name;
 *            -          short int trace;
 *            -          object *proc;};
 *            -
 *            -      typedef struct s_SS_proc procedure;
 *            -
 *            - and the proc member points off to a C_procedure for
 *            - the purposes of SS_install
 *            -
 *            -      struct s_SS_C_proc
 *            -         {object *(*handler)();
 *            -          object *(*proc)();}
 *            -
 *            -      typedef struct s_SS_C_proc C_procedure;
 */

void SS_install(pname, pdoc, phand, pproc, ptype)
   char *pname, *pdoc;
   PFPObject phand;
   PFPObject pproc;
   int ptype;
   {object *op, *vp;
    procedure *pp;
    C_procedure *Cp;

    pp = FMAKE(procedure, "SS_INSTALL:pp");
    if (pp == NULL)
       {PRINT(ERRDEV, "\nError installing procedure %s\n", pname);
        longjmp(SC_top_lev, ABORT);};

    Cp = FMAKE(C_procedure, "SS_INSTALL:Cp");
    if (Cp == NULL)
       {PRINT(ERRDEV, "\nError installing procedure %s\n", pname);
        longjmp(SC_top_lev, ABORT);};

    pp->doc = SC_strsavef(pdoc, "char*:SS_INSTALL:doc");
    if (pp->doc == NULL)
       {PRINT(ERRDEV, "\nError installing procedure documentation - %s\n",
                      pname);
        longjmp(SC_top_lev, ABORT);};

    pp->name = SC_strsavef(pname, "char*:SS_INSTALL:name");
    if (pp->name == NULL)
       {PRINT(ERRDEV, "\nError installing procedure name - %s\n", pname);
        longjmp(SC_top_lev, ABORT);};
        
    pp->trace = FALSE;
    pp->type  = (char) ptype;
    pp->proc  = (object *) Cp;

    Cp->handler = phand;
    Cp->proc    = pproc;

    op     = SS_mk_proc_object(pp);
    SS_UNCOLLECT(op);

    vp     = SS_mk_variable(pname, op);
    SS_UNCOLLECT(vp);

    SC_install(pname, vp, SS_POBJECT_S, SS_symtab);

    return;}

/*--------------------------------------------------------------------------*/

/*                            CONSTRUCTORS                                  */

/*--------------------------------------------------------------------------*/

/* SS_MK_PROC_OBJECT - encapsulate a procedure as an object */

object *SS_mk_proc_object(pp)
   procedure *pp;
   {object *op;

    op = SS_mk_object(pp, PROC_OBJ, SELF_EV, pp->name);
    op->print   = SS_wr_proc;
    op->release = _SS_rl_procedure;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_PROCEDURE - make-procedure returns a procedure
 *                 - the proc part of the procedure struct is a pointer
 *                 - to an S_procedure (compound or Scheme procedure)
 */

object *SS_mk_procedure(lam_exp, penv)
   object *lam_exp, *penv;
   {Register S_procedure *Sp;
    Register procedure *pp;
    object *op;

    Sp = FMAKE(S_procedure, "SS_MK_PROCEDURE:Sp");

/* it is a circular gc problem to have a procedure point
 * to the environment in which it is defined
 * such an environment can never be reclaimed
 */
    Sp->proc = SS_mk_cons(penv, lam_exp);
    SFREE(penv);

    SS_MARK(Sp->proc);

    pp = FMAKE(procedure, "SS_MK_PROCEDURE:pp");

    pp->type  = SS_PROC;
    pp->doc   = NULL;
    pp->name  = SC_strsavef("lambda", "char*:SS_MK_PROCEDURE:name");
    pp->trace = FALSE;
    pp->proc  = (object *) Sp;

    op = SS_mk_proc_object(pp);

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_ESC_PROC - make an escape procedure object */

object *SS_mk_esc_proc(cont, stck, err, type)
   int cont, stck, err, type;
   {Register procedure *pp;
    Register Esc_procedure *ep;
    object *op;

    ep = FMAKE(Esc_procedure, "SS_MK_ESC_PROC:ep");
    ep->cont = cont;
    ep->stck = stck;
    ep->err  = err;
    ep->type = type;

    pp = FMAKE(procedure, "SS_MK_ESC_PROC:pp");
    pp->type = SS_ESC_PROC;
    pp->proc = (object *) ep;
    pp->name = SC_strsavef("escape", "char*:SS_MK_ESC_PROC:name");

    op = SS_mk_proc_object(pp);

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_VARIABLE - encapsulate a VARIABLE in an object */

object *SS_mk_variable(n, v)
   char *n;
   object *v;
   {Register variable *vp;
    object *op;

    vp = FMAKE(variable, "SS_MK_VARIABLE:vp");
    vp->name  = SC_strsavef(n, "char*:SS_MK_VARIABLE:name");
    vp->value = v;

    op = SS_mk_object(vp, VARIABLE, VAR_EV, vp->name);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_variable;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_STRING - make a string and imbed it in an object */

object *SS_mk_string(s)
   char *s;
   {Register string *sp;
    object *op;

    sp = FMAKE(string, "SS_MK_STRING:sp");
    sp->length = strlen(s);
    sp->string = SC_strsavef(s, "char*:SS_MK_STRING:string");

    op = SS_mk_object(sp, SC_STRING_I, SELF_EV, sp->string);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_string;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_INPORT - make an object of type IN_PORT which encapsulates
 *              - a port struct with FILE pointer str
 */

object *SS_mk_inport(str)
   FILE *str;
   {Register input_port *pp;
    object *op;

    pp = FMAKE(input_port, "SS_MK_INPORT:pp");
    pp->str  = str;
    pp->ptr  = pp->buffer;
    *pp->ptr = '\0';

    op = SS_mk_object(pp, IN_PORT, SELF_EV, NULL);

    op->print   = _SS_wr_inport;
    op->release = _SS_rl_inport;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_OUTPORT - make an object of type OUT_PORT which encapsulates
 *               - a port struct with FILE pointer str
 */

object *SS_mk_outport(str)
   FILE *str;
   {Register output_port *pp;
    object *op;

    if (str != NULL)
       SC_setbuf(str, NULL);

    pp = FMAKE(output_port, "SS_MK_OUTPORT:pp");
    pp->str = str;

    op = SS_mk_object(pp, OUT_PORT, SELF_EV, NULL);
    op->print   = _SS_wr_outport;
    op->release = _SS_rl_outport;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_INTEGER - make an integer object */

object *SS_mk_integer(i)
   BIGINT i;
   {Register BIGINT *lp;
    object *op;

    lp = FMAKE(BIGINT, "SS_MK_INTEGER:lp");
    *lp = i;

    op = SS_mk_object(lp, SC_INTEGER_I, SELF_EV, NULL);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_integer;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_FLOAT - make an float object */

object *SS_mk_float(d)
   double d;
   {Register double *dp;
    object *op;

    dp = FMAKE(double, "SS_MK_FLOAT:dp");
    *dp = d;

    op = SS_mk_object(dp, SC_FLOAT_I, SELF_EV, NULL);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_float;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_BOOLEAN - encapsulate a BOOLEAN in an object */

object *SS_mk_boolean(s, v)
   char *s;
   int v;
   {Register boolean *bp;
    object *op;

    bp = FMAKE(boolean, "SS_MK_BOOLEAN:bp");
    bp->name  = SC_strsavef(s, "char*:SS_MK_BOOLEAN:name");
    bp->value = v;

    op = SS_mk_object(bp, BOOLEAN, SELF_EV, bp->name);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_boolean;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_CONS - put the two args together into a new cons,
 *            - puts that into a new object, and
 *            - returns a pointer to it
 *            - marks both car and cdr to do the garbage collection
 *            - bookkeeping this is the C version of cons
 */

object *SS_mk_cons(ca, cd)
   object *ca, *cd;
   {Register cons *cp;
    object *op;

    cp = FMAKE(cons, "SS_MK_CONS:cp");
    SS_MARK(ca);
    SS_MARK(cd);
    cp->car = ca;
    cp->cdr = cd;

    op = SS_mk_object(cp, CONS, PROC_EV, NULL);
    op->print   = SS_wr_lst;
    op->release = _SS_rl_cons;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_OBJECT - make a new object and initialize its garbage collection */

object *SS_mk_object(np, type, ev_type, pname)
   byte *np;
   int type, ev_type;
   char *pname;
   {Register object *op;

#ifdef OMSA

    op = FMAKE(object, "SS_MK_OBJECT:op");

#endif

#ifdef OMSB

/* allocate objects one at a time so that SC_arrlen and PDBLib will be
 * able to know about objects in the prescribed manner
 */
    if (_SS_free_object_list == NULL)
       {op = FMAKE(object, "SS_MK_OBJECT:op");
	op->val = NULL;
        SC_REMEMBER(object *, op, _SS_object_list,
		    _SS_n_objects, _SS_n_objects_max, 100);
	_SS_free_object_list = op;}
    else
       SC_mem_stats_acc((long) sizeof(object), 0L);

    op = _SS_free_object_list;
    _SS_free_object_list = (object *) op->val;

#endif

    if ((pname != NULL) && (SC_arrlen(pname) < 1))
       pname = SC_strsavef(pname, "char*:SS_MK_OBJECT:pname");

    SC_arrtype(op, type);

    op->eval_type  = ev_type;
    op->print_name = pname;
    op->val        = np;

    op->release    = SS_rl_object;

    return(op);}

/*--------------------------------------------------------------------------*/

#ifdef LARGE

/*--------------------------------------------------------------------------*/

/* SS_MK_CHAR - make character object */

object *SS_mk_char(i)
   int i;
   {Register int *ip;
    object *op;

    ip = FMAKE(int, "SS_MK_CHAR:ip");
    *ip = i;

    op = SS_mk_object(ip, CHAR_OBJ, SELF_EV, NULL);
    op->print   = SS_wr_atm;
    op->release = _SS_rl_char;

    return(op);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_VECTOR - make a vector object
 *              - this is a dummy for now
 */

object *SS_mk_vector(l)
   int l;
   {Register vector *vp;
    Register int i;
    Register object **va;
    object *op;

    vp = FMAKE(vector, "SS_MK_VECTOR:vp");
    va = FMAKE_N(object *, l, "SS_MK_VECTOR:va");
    for (i = 0; i < l; i++)
        va[i] = SS_null;

    vp->length = l;
    vp->vect   = va;

    op = SS_mk_object(vp, VECTOR, SELF_EV, NULL);
    op->print   = _SS_wr_vector;
    op->release = _SS_rl_vector;

    return(op);}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/*                            DESTRUCTORS                                   */

/*--------------------------------------------------------------------------*/

#ifdef LARGE

/*--------------------------------------------------------------------------*/

/* _SS_RL_CHAR - release an char object */

static void _SS_rl_char(obj)
   object *obj;
   {SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_VECTOR - release a vector object */

static void _SS_rl_vector(obj)
   object *obj;
   {Register int i, k;
    Register object **va;

    k  = SS_VECTOR_LENGTH(obj);
    va = SS_VECTOR_ARRAY(obj);

    for (i = 0; i < k; i++)
        {SS_Assign(va[i], SS_null);};

    SFREE(va);
    SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* _SS_RL_INTEGER - release an integer object */

static void _SS_rl_integer(obj)
   object *obj;
   {SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_FLOAT - release a float object */

static void _SS_rl_float(obj)
   object *obj;
   {SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_STRING - release a string object */

static void _SS_rl_string(obj)
   object *obj;
   {SFREE(SS_STRING_TEXT(obj));
    SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_VARIABLE - release a variable object */

static void _SS_rl_variable(obj)
   object *obj;
   {SFREE(SS_VARIABLE_NAME(obj));
    SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_CONS - release a CONS object
 *             - since this also SS_GC's lists be careful of
 *             - conses with multiple pointers
 */

static void _SS_rl_cons(obj)
   object *obj;
   {Register object *lst, *cdr, *car;

    lst = obj;
    while (TRUE)
       {if (SS_OBJECT_GC(lst) > 1)
           {SFREE(lst);
	    return;};

        cdr = SS_cdr(lst);
        car = SS_car(lst);
        SS_GC(car);

        SFREE(SS_OBJECT(lst));
        SS_rl_object(lst);
/*        SFREE(lst); */
        if (SS_nullobjp(cdr) || !SS_consp(cdr))
           {SS_GC(cdr);
            return;}
        else
           lst = cdr;};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_PROCEDURE - release a procedure object */

static void _SS_rl_procedure(obj)
   object *obj;
   {Register object *op, *cdr;

/* release the name of the procedure */
    SFREE(SS_PROCEDURE_NAME(obj));

    switch (SS_PROCEDURE_TYPE(obj))
       {case SS_PR_PROC  :
        case SS_UR_MACRO :
        case SS_EE_MACRO :
        case SS_UE_MACRO : SFREE(SS_PROCEDURE_PROC(obj));
                           break;

/* since the environment part of the procedure was not SS_MARKed at the
 * time the procedure was made because of the circularity of the situation 
 * GC the cdr of the S_procedure and release the cons only
 */
        case SS_MACRO    :
        case SS_PROC     : op  = SS_COMPOUND_PROCEDURE_FUNCTION(obj);
                           cdr = SS_cdr(op);
                           SS_GC(cdr);

                           SFREE(SS_OBJECT(op));
                           SFREE(op);
                           SFREE(SS_PROCEDURE_PROC(obj));
                           break;

        case SS_ESC_PROC : SFREE(SS_PROCEDURE_PROC(obj));
        default          : break;};

    SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_INPORT - release an object of type IN_PORT which encapsulates
 *               - a port struct with FILE pointer str
 */

static void _SS_rl_inport(obj)
   object *obj;
   {SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_OUTPORT - release an object of type OUT_PORT which encapsulates
 *                - a port struct with FILE pointer str
 */

static void _SS_rl_outport(obj)
   object *obj;
   {SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_BOOLEAN - release a boolean object */

static void _SS_rl_boolean(obj)
   object *obj;
   {SFREE(SS_BOOLEAN_NAME(obj));
    SFREE(SS_OBJECT(obj));
    SS_rl_object(obj);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_EOFOBJ - release an eof object */

#if 0
static void _SS_rl_eofobj(obj)
   object *obj;
   {_SS_rl_boolean(obj);

    return;}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_NULLOBJ - release a null object */

#if 0
static void _SS_rl_nullobj(obj)
   object *obj;
   {_SS_rl_boolean(obj);

    return;}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_RL_ERROBJ - release an error object */

#if 0
static void _SS_rl_errobj(obj)
   object *obj;
   {return;}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RL_OBJECT - release a SCHEME object */

void SS_rl_object(obj)
   object *obj;
   {

#ifdef OMSA

    SFREE(obj);

#endif

#ifdef OMSB

    SC_set_count(obj, 0);
    memset(obj, 0, sizeof(object));
    obj->val = (byte *) _SS_free_object_list;

    SC_mem_stats_acc(0L, (long) sizeof(object));

    _SS_free_object_list = obj;

#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_GC - garbage collection, incremental
 *       - each object has number of pointers to itself (except #t, #f, etc)
 *       - when the number of pointers is 1 then garbage collection means
 *       - freeing the space associated with the object
 *       - when the number is greater than 1 garbage collection means
 *       - decrementing the number of pointers to itself
 */

void SS_gc(obj)
   object *obj;
   {

    if (obj == NULL)
       return;

    if ((SS_OBJECT_GC(obj) != 1) || (obj->val == NULL))
       {SFREE(obj);}
    else
       SS_OBJECT_FREE(obj);

    return;}

/*--------------------------------------------------------------------------*/

/*                                MONITORS                                  */

/*--------------------------------------------------------------------------*/

/* SS_PR_OBJ_MAP - dump the list of objects and associated info */

object *SS_pr_obj_map()

#ifdef OMSA

   {return(SS_f);}

#endif

#ifdef OMSB

   {long i, ngc, nfr, nref, ityp;
    int otyp, frp, *types;
    object *obj;

    types = FMAKE_N(int, 101, "SS_PR_OBJ_MAP:types");

    ngc = 0L;
    nfr = 0L;
    for (i = 0L; i < _SS_n_objects; i++)
        {obj = _SS_object_list[i];
         nref = SC_ref_count(obj);
         ityp = SC_arrtype(obj, 0);
         if (ityp < 100)
            types[ityp]++;
         else
            types[100]++;
         frp  = ((obj->print_name == NULL) &&
		 (obj->eval_type  == '\0') &&
		 (obj->print      == NULL) &&
		 (obj->release    == NULL));
         ngc += (nref != UNCOLLECT) && !frp;
         nfr += frp;};

    PRINT(stdout, "Objects by type:\n");
    for (i = 0L; i < 101; i++)
        {if (types[i] > 0)
            PRINT(stdout, "   %d type %d objects\n", types[i], i);};

    PRINT(stdout, "\n");
    PRINT(stdout, "Objects: %ld total, %ld permanent, %ld collectable, and %ld free\n\n",
	  _SS_n_objects, _SS_n_objects - ngc - nfr, ngc, nfr);
    PRINT(stdout, "  Address   # Refs Type    Print Name\n");
    for (i = 0L; i < _SS_n_objects; i++)
        {obj = _SS_object_list[i];
         nref = SC_ref_count(obj);
	 frp  = ((obj->print_name == NULL) &&
		 (obj->eval_type  == '\0') &&
		 (obj->print      == NULL) &&
		 (obj->release    == NULL));
         if ((nref != UNCOLLECT) && !frp)
            {otyp = SC_arrtype(obj, -1);
             PRINT(stdout, "0x%8lx %6ld  %3d  :  ", obj, nref, otyp);
	     if (obj->print_name != NULL)
	        PRINT(stdout, "%s\n", obj->print_name);

	     else if (SS_consp(obj))
	        {PRINT(stdout, "(");
		 _SS_print(SS_car(obj), "", " ", SS_outdev);
	         PRINT(stdout, "... )\n");}

             else
	        _SS_print(obj, "", "\n", SS_outdev);};};

    PRINT(stdout, "\n");

    SFREE(types);

    return(SS_f);}

#endif

/*--------------------------------------------------------------------------*/

/*                                PRINTERS                                  */

/*--------------------------------------------------------------------------*/

/* _SS_WR_INPORT - print an input port */

static void _SS_wr_inport(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<INPUT_PORT|0x%lx|%ld>",
	  SS_INSTREAM(obj),
	  (long) (SS_PTR(obj) - SS_BUFFER(obj)));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_WR_OUTPORT - print an output port */

static void _SS_wr_outport(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "<OUTPUT_PORT|0x%lx>", SS_OUTSTREAM(obj));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_WR_ERROBJ - print an error object */

#if 0
static void _SS_wr_errobj(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "ERROR: %s", SS_STRING_TEXT(obj));

    return;}
#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_WR_NULLOBJ - print an nil object */

#if 0
static void _SS_wr_nullobj(obj, strm)
   object *obj, *strm;
   {PRINT(SS_OUTSTREAM(strm), "()");

    return;}
#endif

/*--------------------------------------------------------------------------*/

#ifdef LARGE

/*--------------------------------------------------------------------------*/

/* _SS_WR_VECTOR - print a vector object */

static void _SS_wr_vector(obj, strm)
   object *obj, *strm;
   {object *lst;

    lst = SS_vctlst(obj);
    SS_MARK(lst);
    PRINT(SS_OUTSTREAM(strm), "#");
    SS_wr_lst(lst, strm);
    SS_GC(lst);

    return;}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

