
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;		afm font name -> file name translation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define font-xlate-pat (reg-expr->proc
			'(entire
			  (seq
			   (* space)
			   (save (+ (or alpha #\- #\_ digit)))
			   (+ space)
			   (save (+ (or alpha #\- #\_ digit)))
			   (* space)))))

;;;  These patterns match what I see in Ghostscript's Fontmap file

;;; "/Bar   (blech.pfb) ;"

(define font-xlate-pat2 (reg-expr->proc
			'(entire
			  (seq
			   (* space)
			   #\/
			   (save (+ (or alpha #\- #\_ digit)))
			   (+ space)
			   #\(
			   (save (+ (or alpha #\- #\_ digit)))
			   (? (seq #\. (+ (or alpha digit))))
			   #\)
			   (* space)
			   #\;
			   (* space)))))

;;; "/Foo   /Bar ;"

(define font-xlate-pat3 (reg-expr->proc
			'(entire
			  (seq
			   (* space)
			   #\/
			   (save (+ (or alpha #\- #\_ digit)))
			   (+ space)
			   #\/
			   (save (+ (or alpha #\- #\_ digit)))
			   (* space)
			   #\;
			   (* space)))))

(define (read-font-translation-line line)
  (bind ((s e from to (font-xlate-pat line)))
    (if s
	(values from to)
	(bind ((s e from to (font-xlate-pat2 line)))
	  (if s
	      (values from to)
	      (bind ((s e from to (font-xlate-pat3 line)))
		(if s
		    (values from (list to))
		    (values))))))))

(define (read-font-translation-file file-name)
  (if (file-exists? file-name)
      (call-with-input-file
	  file-name
	(lambda (port)
	  (let ((tbl (make-table string=? string->hash)))
	    (let loop ()
	      (let ((ln (read-line port)))
		(if (eof-object? ln)
		    tbl
		    (begin
		      (bind ((from to (read-font-translation-line ln)))
			(if from
			    (table-insert! tbl from to))
			(loop)))))))))
      #f))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;		      afm directories and paths
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-class <afm-directory> (<object>)
  (afm-directory type: <directory-name>)
  (translation init-value: #f)) ; #f=>none, string=>file, table=>map

(define (check-afm-file (self <afm-directory>) (file-name <string>))
  (let ((f (append-path (afm-directory self) 
			(string->file (string-append file-name ".afm")))))
    (if (file-exists? f)
	f
	#f)))

(define-method get-afm-file ((self <afm-directory>) (font-name <string>))
  (if (string? (translation self))
      (set-translation! self (read-font-translation-file
			      (pathname->os-path
			       (append-path
				(afm-directory self)
				(string->file (translation self)))))))
  (or (and (translation self)
	   (let ((try (table-lookup (translation self) font-name)))
	     (if (pair? try)
		 (get-afm-file self (car try))
		 (and try (check-afm-file self try)))))
      (check-afm-file self font-name)))

(define *afm-path*
  (list (make <afm-directory>
	      afm-directory: (string->dir "~/lib/afm"))
	(make <afm-directory>
	      afm-directory: (string->dir "/usr/lib/ghostscript/fonts")
	      translation: "Fontmap")
	(make <afm-directory>
	      afm-directory: (string->dir "/usr/lib/enscript"))))

(define (push-afm-directory (dir <string>) 
			    #optional (translation-file default: "font.map"))
  (set! *afm-path* (cons (make <afm-directory>
			       afm-directory: (string->dir dir)
			       translation: translation-file)
			 *afm-path*)))

(define (get-afm-from-path font-name)
  (let loop ((p *afm-path*))
    (if (null? p)
	#f
	(let ((f (get-afm-file (car p) font-name)))
	  (if f
	      (register-afm! font-name (load-afm (pathname->os-path f)))
	      (loop (cdr p)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;			 in-memory afm cache
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define *afm-cache* (make-string-table))

(define (register-afm! font-name (afm <afm>))
  (table-insert! *afm-cache* font-name afm)
  afm)

(define (get-afm-from-table font-name)
  (table-lookup *afm-cache* font-name))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;				 API
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (get-afm font-name)
  (or (get-afm-from-table font-name)
      (get-afm-from-path font-name)
      (error "~a: no AFM loaded" font-name)))
