
(define $portable-sockopt-options
 '((socket/debug 0 <boolean>)
   (socket/reuse-addr 1 <boolean>)
   (socket/keep-alive 2 <boolean>)
   (socket/dont-route 3 <boolean>)
   (socket/linger 4 <linger>)         ; linger is ~ (union #f <integer>)
   (socket/broadcast 5 <boolean>)
   (socket/oob-inline 6 <boolean>)
   (socket/send-buffer 7 <integer>)
   (socket/receive-buffer 8 <integer>)
   (socket/type 9 <integer> read-only)
   (socket/error 10 <integer> read-only)))

(define $portable-sockopt-levels
  '((level/socket 0)
    (level/ip 1)
    (level/tcp 2)
    (level/udp 3)))

(define (do-sockopt socket level option value type)
  (case type
    ((<boolean>)
     (if (boolean? value)
	 (values #t (setsockopt-int socket level option (if value 1 0)))
	 #f))
    ((<integer>)
     (if (fixnum? value)
	 (values #t (setsockopt-int socket level option value))
	 #f))
    ((<interval>)
     (if (instance? value <interval>)
	 (values #t (setsockopt-time socket level option value))
	 (if (instance? value <number>)
	     ;; SCSH compatibility; a number denotes # microseconds
	     (values #t (setsockopt-time
			 socket
			 level
			 option
			 (seconds->interval (* value 0.000001))))
	     #f)))
    ((<linger>)
     (if (or (eq? value #f) (integer? value))
	 (values #t (setsockopt-linger socket level option value))
	 #f))))


(define (set-socket-option socket level option value)
  (let ((opt (assq option $portable-sockopt-options))
	(lvl (assq level $portable-sockopt-levels)))
    (if (and opt lvl)
	(if (null? (cdddr opt))
	    (bind ((ok? rc (do-sockopt
			    socket 
			    (local-socket-level (cadr lvl))
			    (local-socket-option (cadr opt))
			    value 
			    (caddr opt))))
	      (if ok?
		  (if (eq? rc 0)
		      '#undef
		      (error "set-socket-option: error ~d" (errno)))
		  (error "socket option ~s expected a ~s, not ~s"
			 option (caddr opt) value)))
	    (error "socket option ~s is read-only" option))
	(error "socket ~a `~s' is not valid\nexpected: ~j"
	       (if opt "level" "option")
	       (if opt level option)
	       (map car (if opt
			    $portable-sockopt-levels
			    $portable-sockopt-options))))))

(define (get-socket-option socket level option)
  (let ((opt (assq option $portable-sockopt-options))
	(lvl (assq level $portable-sockopt-levels)))
    (if (and opt lvl)
	(let ((lvl (local-socket-level (cadr lvl)))
	      (opt (local-socket-option (cadr opt)))
	      (typ (caddr opt)))
	  (getsockopt socket lvl opt typ))
	(error "socket ~a `~s' is not valid\nexpected: ~j"
	       (if opt "level" "option")
	       (if opt level option)
	       (map car (if opt
			    $portable-sockopt-levels
			    $portable-sockopt-options))))))
	   
(define-syscall-glue (getsockopt (fd <raw-int>)
				 (level <raw-int>)
				 (opt <raw-int>)
				 type)
  literals: ((& error) 
	     '<boolean> 
	     '<integer> 
	     '<interval>
	     (& <interval>) 
	     '<linger>)
{
  int len, rc = 0;

  if (EQ(type,LITERAL(1)))
    {
      int v;
      len = sizeof v;

      rc = getsockopt( fd, level, opt, (void *)&v, &len );
      REG0 = v ? TRUE_OBJ : FALSE_OBJ;
    }
  else if (EQ(type,LITERAL(2)))
    {
      int v;
      len = sizeof v;
      rc = getsockopt( fd, level, opt, (void *)&v, &len );
      REG0 = int2fx(v);
    }
  else if (EQ(type,LITERAL(3)))
    {
      struct timeval v;
      len = sizeof v;
      rc = getsockopt( fd, level, opt, (void *)&v, &len );
      REG0 = os_time( &v, TLREF(4) );
    }
  else if (EQ(type,LITERAL(5)))
    {
      struct linger v;
      len = sizeof v;

      rc = getsockopt( fd, level, opt, (void *)&v, &len );
      if (v.l_onoff)
	REG0 = int2fx(v.l_linger);
      else
	REG0 = FALSE_OBJ;
    }
  else
    {
      scheme_error( "bad sockopt type: ~s", 1, type );
    }
  if (rc < 0)
    scheme_error( "getsockopt: error ~d", 1, errno );
  RETURN1();
})
