#line 1 "events.c"
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include "rs_sys_threads_manager_p.h"

#if !defined(PLATFORM_LINUX) && !defined(PLATFORM_NEXT)
#include <sys/select.h>
#endif

extern int sigq_notify_fd;   /* from osglue.c */
extern unsigned num_sig;     /* from osglue.c */

static int max_num_fds = 0;
int time_slice = 15;
struct sys_time next_select_time = { 0, 0 };
static struct sys_event *event_block_free_list = NULL;
static int next_event_id;

static int notify_pipe[2];

#define notify_read_side notify_pipe[0]
#define notify_write_side notify_pipe[1]


static obj next_event_num = ZERO;

#define EVENT_NUMBER(e) ((int)(e->event_vec_ix/SLOT(1)))

static void free_event( struct sys_event *e )
{
  if (e)
    {
      e->next = event_block_free_list;
      event_block_free_list = e;
      if (DEBUG_THREAD_EVENTS)
	printf( "freeing event #%d {%p}\n", EVENT_NUMBER(e), e );
    }
}

static struct sys_event *alloc_event( enum sys_event_type type )
{
  struct sys_event *e = event_block_free_list;

  if (e)
    {
      event_block_free_list = e->next;
      e->type = type;
      if (DEBUG_THREAD_EVENTS)
	printf( "allocating event #%d {%p} (pop'ing from free list)\n",
		EVENT_NUMBER(e), e );
      return e;
    }

  /*
   *  grow the event ID space, which includes the corresponding
   *  target vector in the scheme heap
   */

  e = (struct sys_event *)malloc( sizeof(struct sys_event) );
  e->event_vec_ix = SLOT(next_event_id);

  if (SLOT(next_event_id) >= SIZEOF_PTR(target_vec))
    {
      obj d, s = target_vec;
      UINT_32 i, n = SIZEOF_PTR(target_vec) * 2;
      if (n < SLOT(3))
	n = SLOT(3);

      if (DEBUG_THREAD_EVENTS)
	printf( "allocating event #%d {%p} (growing vec from %lu => %lu)\n",
		next_event_id, 
		e,
		SIZEOF_PTR(target_vec)/SLOT(1),
		n/SLOT(1));

      d = target_vec = alloc( n, vector_class );
      for (i=0; i<SIZEOF_PTR(s); i+=SLOT(1))
	gvec_write_init( d, i, gvec_ref(s,i) );
      for (; i<n; i+=SLOT(1))
	{
	  gvec_write_init_non_ptr( d, i, ZERO );
	}
    }
  else
    {
      if (DEBUG_THREAD_EVENTS)
	printf( "allocating event #%d {%p} (no need to grow vec)\n",
		next_event_id, e );
    }
  next_event_id++;
  e->type = type;
  return e;
}

static void assert_class_nm( struct sys_event *evt, 
			     obj target, 
			     char *class_nm )
{
  const char *is = symbol_text( class_name( object_class( target ) ) );

  if (strcmp( class_nm, is ) != 0)
    {
      fprintf( stderr,
	       "event #%d {%p} -- class mismatch in target_backptr_index\n",
	       EVENT_NUMBER(evt), evt );
      fprintf( stderr, "got a %s (", is );
      fprinto( stderr, target );
      fprintf( stderr, ") expected a %s\n", class_nm );
      abort();
    }
}

/* returns the index to the target object's `event' slot */

static UINT_32 target_backptr_index( struct sys_event *evt, obj target )
{
#define assert_class_name(str) assert_class_nm( evt, target, str )

  switch (evt->type)
    {
      /* data is an fd, target is an <mbox-input-port> */
    case SYS_EVENT_FILE_READ:
      assert_class_name( "<mbox-input-port>" );
      return MBOX_INPUT_PORT_EVENT;

    case SYS_EVENT_FILE_ACCEPT:
      assert_class_name( "<service>" );
      return SERVICE_EVENT;

    case SYS_EVENT_FILE_WRITE:
      assert_class_name( "<queued-output-port>" );
      return QOUT_EVENT;

    case SYS_EVENT_FILE_X_READ:
      assert_class_name( "<x-event-queue>" );
      return XQUEUE_EVENT;

    case SYS_EVENT_FILE_EXCEPTION:
      break;

    /* target is a <thread> */

    case SYS_EVENT_REALTIME:
      assert_class_name( "<thread>" );
      return THREAD_BLOCKED_ON;
      
    case SYS_EVENT_REALTIME_INTERVAL:
    case SYS_EVENT_THREADTIME:
    case SYS_EVENT_THREADTIME_INTERVAL:
    }
  abort();
  return 0;
}

static void setup_target_biptr( struct sys_event *evt, obj target )
{
  UINT_32 ix = target_backptr_index(evt, target);

  assert( EQ(gvec_ref(target, ix),ZERO) );
  assert( EQ(EVENT_TARGET(evt), ZERO ) );

  gvec_write_non_ptr( target, ix, RAW_PTR_TO_OBJ(evt) );
  gvec_write_ptr( target_vec, evt->event_vec_ix, target );

  if (DEBUG_THREAD_EVENTS)
    printf( "assigning event #%d {%p} for target {%#lx}\n",
	    EVENT_NUMBER(evt), evt, VAL(target) );
}

static void clear_target_biptr( struct sys_event *evt )
{
  UINT_32 ix;
  obj t = EVENT_TARGET(evt);

  ix = target_backptr_index(evt, t);

  assert( !EQ(t, ZERO ) );
  assert( !EQ(gvec_ref(t, ix),ZERO) );
  gvec_write_non_ptr( t, ix, ZERO );
  gvec_write_non_ptr( target_vec, evt->event_vec_ix, ZERO );

  if (DEBUG_THREAD_EVENTS)
    printf( "unassigning event #%d {%p}, from target {%#lx}\n",
	    EVENT_NUMBER(evt), evt, VAL(t) );
}

void init_events( void )
{
  if (pipe( notify_pipe ) < 0)
    os_error( "pipe", 0 );
  if (notify_read_side >= max_num_fds)
    max_num_fds = notify_read_side+1;
}

struct file_event_list {
  fd_set  set;
  struct sys_event *list;
};

static struct file_event_list read_list, write_list, except_list;
static struct sys_event *next_realtime_event = NULL;

static obj handle_exception_event( struct sys_event *e, obj to_do );
static obj handle_read_event( struct sys_event *e, obj to_do );
static obj handle_realtime_event( struct sys_event *e, obj to_do );

#define MALLOC(t) ((t *)malloc(sizeof(t)))

static struct sys_event *event_list_rm( struct sys_event *e,
					struct sys_event *head )
{
  /* pop it out of the list */
  if (e->prev)
    {
      e->prev->next = e->next;
    }
  else
    {
      head = e->next;
    }

  if (e->next)
    {
      e->next->prev = e->prev;
    }
  return head;
}

static struct sys_event *insert_time_event( struct sys_event *e,
					    struct sys_event *head )
{
  if (!head)
    {
      e->prev = e->next = NULL;
      return e;
    }
  if (time_lt( e->data.time.next_time, head->data.time.next_time ))
    {
      head->prev = e;
      e->next = head;
      if (DEBUG_THREAD_EVENTS)
	printf( "inserting event at %ld.%06ld in head of list\n",
		e->data.time.next_time.sec,
		e->data.time.next_time.usec );
      return e;
    }
  else
    {
      struct sys_event *prev, *p;
      prev = head;
      for (p=head->next; p; p=p->next)
	{
	  if (time_lt( e->data.time.next_time, p->data.time.next_time ))
	    {
	      /* insert it before p and after prev */
	      if (DEBUG_THREAD_EVENTS)
		printf( "inserting event at %ld.%06ld just before one at %ld.%06ld\n",
			e->data.time.next_time.sec,
			e->data.time.next_time.usec,
			p->data.time.next_time.sec,
			p->data.time.next_time.usec );
	      prev->next = e;
	      p->prev = e;
	      e->next = p;
	      e->prev = prev;
	      return head;
	    }
	  prev = p;
	}
      /* insert it at the end */
      if (DEBUG_THREAD_EVENTS)
	printf( "inserting event at %ld.%06ld at end of list\n",
		e->data.time.next_time.sec,
		e->data.time.next_time.usec );
      prev->next = e;
      e->next = NULL;
      e->prev = prev;
      return head;
    }
}

struct sys_event *register_time_event( struct sys_time next_time, 
				       obj target,
				       enum sys_event_type t )
{
  struct sys_event *e = alloc_event(t);
  e->next = e->prev = NULL;
  e->data.time.next_time = next_time;
  e->data.time.delta_time.sec = 0;
  e->data.time.delta_time.usec = 0;
  setup_target_biptr( e, target );

  next_realtime_event = insert_time_event( e, next_realtime_event );
  return e;
}

struct sys_event *make_time_event( struct sys_time next_time, obj target )
{
  return register_time_event( next_time, target, SYS_EVENT_REALTIME );
}

void free_time_event( struct sys_event *e )
{
  assert( (e->type == SYS_EVENT_REALTIME) 
	  || (e->type == SYS_EVENT_REALTIME_INTERVAL) );

  next_realtime_event = event_list_rm( e, next_realtime_event );
  clear_target_biptr(e);
  free_event(e);
}

struct sys_event *make_time_interval_event( struct sys_time next_time, 
					    struct sys_time interval,
					    obj target )
{
  struct sys_event *e;
  e = register_time_event( next_time, target, SYS_EVENT_REALTIME_INTERVAL );
  e->data.time.delta_time = interval;
  return e;
}



static void free_file_event( struct sys_event *e, struct file_event_list *fel )
{
  FD_CLR( e->data.fd, &fel->set );
  fel->list = event_list_rm( e, fel->list );
  clear_target_biptr(e);
  free_event(e);
}

static struct sys_event *register_file_event( int fd, 
					      obj target, 
					      struct file_event_list *fel,
					      enum sys_event_type type )
{
  struct sys_event *s = alloc_event(type);

  FD_SET( fd, &fel->set );

  if (fd >= max_num_fds)
    max_num_fds = fd+1;

  s->data.fd = fd;
  s->next = fel->list;
  s->prev = NULL;
  if (fel->list)
    fel->list->prev = s;
  fel->list = s;
  setup_target_biptr( s, target );
  return s;
}

struct sys_event *make_read_event( int fd, obj target )
{
  return register_file_event( fd, target, &read_list, SYS_EVENT_FILE_READ );
}

struct sys_event *make_read_x_event( int fd, obj target )
{
  return register_file_event( fd, target, &read_list, SYS_EVENT_FILE_X_READ );
}

struct sys_event *make_accept_event( int fd, obj target )
{
  return register_file_event( fd, target, &read_list, SYS_EVENT_FILE_ACCEPT );
}

struct sys_event *make_write_event( int fd, obj target )
{
  return register_file_event( fd, target, &write_list, SYS_EVENT_FILE_WRITE );
}

struct sys_event *make_exception_event( int fd, obj target )
{
  return register_file_event( fd, target, &except_list, 
			      SYS_EVENT_FILE_EXCEPTION );
}


void free_read_event( struct sys_event *e )
{
  free_file_event( e, &read_list );
}

void free_accept_event( struct sys_event *e )
{
  free_file_event( e, &read_list );
}

void free_write_event( struct sys_event *e )
{
  free_file_event( e, &write_list );
}

void free_exception_event( struct sys_event *e )
{
  free_file_event( e, &except_list );
}


/* how long until next event?
 * it's the smaller of the current time slice,
 * this thread's next CPU timer,
 * and the next real-time timer
 */
  
int compute_time_slice( obj t )
{
  int ms = time_slice;
  struct sys_event *e;

  e = OBJ_TO_SYS_EVENT(gvec_ref( t, THREAD_NEXT_THREADTIME_EVENT ));

  if (next_realtime_event)
    {
      int c = diff_time_ms( thread_start, 
			    next_realtime_event->data.time.next_time );
      if (DEBUG_THREAD_EVENTS)
	printf( " compute-time-slice: (real) time event is %d ms away\n", c );
      if (c < ms)
	ms = c;
    }
  if (e)
    {
      int c = diff_time_ms( thread_start, e->data.time.next_time );
      if (DEBUG_THREAD_EVENTS)
	printf( " compute-time-slice: (thread) time event is %d ms away\n",c );
      if (c < ms)
	ms = c;
    }
  if (ms < 0)
    ms = 0;
  return ms;
}


int bad_fd_q( struct sys_event *evt )
{
  int rc;

  rc = fcntl(evt->data.fd, F_GETFL);
  if (rc < 0)
    {
      if (DEBUG_UNCOMMON_CASES)
	{
	  printf( "fd %d is bad (%s)\n", evt->data.fd, strerror(errno) );
	  return 1;
	}
    }
  return 0;
}

static void find_and_notify_bad_fd( void )
{
  struct sys_event *p;

  /* a crude approach -- linear search through all the fd's
     (however, go ahead and assume that usually only one fd will
     become bad at a time, so stop when we find the first one)
     */
  for (p=read_list.list; p; p=p->next)
    {
      if (bad_fd_q(p))
	{
	  obj mbox = gvec_ref( EVENT_TARGET(p), MBOX_INPUT_PORT_MBOX );
	  ksend_mailbox( mbox, 
			 make_os_error( "fcntl", 1, int2fx(p->data.fd) ));
	  free_read_event( p );
	  return;
	}
    }
  if (DEBUG_UNCOMMON_CASES)
    {
      printf( "didn't find bad fd by searching read list!\n" );
    }
}

obj check_for_events( rs_bool block_q )
{
  struct timeval *tp, t;
  fd_set r, w, x;
  int ms = -1, n;
  obj to_do = NIL_OBJ;

  get_sys_time( &thread_start );

  if (!block_q && time_le( thread_start, next_select_time ))
    {
      /* don't bother to check but every 1.5 ms or so,
       * (except always check when blocking)
       */
      if (DEBUG_THREAD_EVENTS)
	printf( "check for events (%s)... next select time is %dus away\n", 
		block_q ? "block" : "don't block",
		diff_time_us( thread_start, next_select_time ) );
      return to_do;
    }

  tp = NULL;
  if (!block_q)
    {
      t.tv_sec = t.tv_usec = 0;
      tp = &t;
    }
  else if (next_realtime_event)
    {
      ms = diff_time_ms( thread_start,
			 next_realtime_event->data.time.next_time );
      if (ms > 0)
	{
	  t.tv_sec = ms / 1000;
	  t.tv_usec = (ms % 1000) * 1000;
	}
      else
	{
	  t.tv_sec = t.tv_usec = 0;
	}
      tp = &t;
    }

  
  r = read_list.set;
  w = write_list.set;
  x = except_list.set;

  if (DEBUG_THREAD_EVENTS)
    {
      printf( "select %d fds, block = %s, delay = ", 
	      max_num_fds, 
	      block_q ? "yes" : "no" );
      if (tp)
	printf( "%g s\n", tp->tv_usec / 1000000.0 + tp->tv_sec );
      else
	printf( "INFINITE\n" );
    }

  sigq_notify_fd = notify_write_side;
  FD_SET( notify_read_side, &r );

  if (rssig_ready)
    {
      t.tv_sec = t.tv_usec = 0;
      tp = &t;
    }

  n = select( max_num_fds, &r, &w, &x, tp );
  sigq_notify_fd = -1;

  if (n < 0)
    {
      if (DEBUG_UNCOMMON_CASES)
	{
	  printf( "select returns error (%s)\n", strerror(errno) );
	}
      /* errors... */
      if (errno == EBADF)
	{
	  /* seek out the bad FDs and send it an error signal */
	  find_and_notify_bad_fd();
	}
    }
  else
    {
      if (DEBUG_THREAD_EVENTS)
	printf( "select returns %d\n", n );
      if (n)
	{
	  struct sys_event *p, *nxt = NULL;

	  for (p=read_list.list; p && n; p=nxt)
	    {
	      nxt = p->next;
	      if (FD_ISSET(p->data.fd, &r))
		{
		  n--;
		  if (DEBUG_THREAD_EVENTS)
		    printf( "  fd %d readable\n", p->data.fd );
		  to_do = handle_read_event( p, to_do );
		}
	    }
	  if (FD_ISSET(notify_read_side, &r))
	    {
	      int nb;
	      char temp[11];
	      n--;

	      nb = read( notify_read_side, temp, 10 );
	      if ((nb > 0) && DEBUG_UNCOMMON_CASES)
		{
		  int i;

		  temp[nb] = 0;
		  printf( "  notification pipe: [" );

		  for (i=0; i<nb; i++)
		    {
		      char *s = NULL;

		      switch ((enum RSSIGNUM)(temp[i]-'A'))
			{
			case RSSIG_USER_INTR:   s = "user-intr"; break;
			case RSSIG_TIMEOUT:     s = "timer"; break;
			case RSSIG_CHILD_EXITED:s = "child-exited"; break;
			case RSSIG_FINALIZE:    s = "finalize"; break;
			case RSSIG_GC_FLIP:     s = "gc-flip"; break;
			case RSSIG_C_SIGNAL:    s = "c-signal"; break;
			}
		      if (i > 0)
			printf( " " );
		      printf( "%c:%s", temp[i], s );
		    }
		  printf( "]  %d signals queued, %s\n",
			  num_sig, 
			  rssig_ready ? "RSSIG READY" : "RSSIG NOT ready" );
		}
	    }
	  for (p=write_list.list; p && n; p=nxt)
	    {
	      nxt = p->next;
	      if (FD_ISSET(p->data.fd, &w))
		{
		  n--;
		  if (DEBUG_THREAD_EVENTS)
		    printf( "  fd %d writable\n", p->data.fd );
		  to_do = handle_write_event( p, to_do );
		}
	    }
	  for (p=except_list.list; p && n; p=nxt)
	    {
	      nxt = p->next;
	      if (FD_ISSET(p->data.fd, &x))
		{
		  n--;
		  if (DEBUG_THREAD_EVENTS)
		    printf( "  fd %d exceptional\n", p->data.fd );
		  to_do = handle_exception_event( p, to_do );
		}
	    }
	  if ((n > 0) && DEBUG_UNCOMMON_CASES)
	    printf( "  select returned %d more things than we could handle\n",
		    n );
	}
    }
  
  get_sys_time( &thread_start );

  next_select_time.usec = thread_start.usec + ALLOW_SELECT_LATENCY;
  next_select_time.sec = thread_start.sec;
  if (next_select_time.usec > 1000000)
    {
      next_select_time.usec -= 1000000;
      next_select_time.sec++;
    }
  
  while (next_realtime_event 
	 && time_le( next_realtime_event->data.time.next_time,
		     thread_start ))
    {
      if (DEBUG_THREAD_EVENTS)
	printf( "  timer\n" );
      to_do = handle_realtime_event( next_realtime_event, to_do );
    }
  return to_do;
}

static obj handle_exception_event( struct sys_event *e, obj to_do )
{
  return to_do;
}

static obj handle_realtime_event( struct sys_event *e, obj to_do )
{
  if (e->type == SYS_EVENT_REALTIME)
    {
      obj thr = EVENT_TARGET(e);
      obj suspend_count = gvec_ref( thr, THREAD_SUSPEND_COUNT );

      if (DEBUG_THREAD_EVENTS)
	printf( " realtime event triggered (after %d us delay)\n",
		diff_time_us( e->data.time.next_time, thread_start ) );
      free_time_event( e );
      if (did_remove_from_queue( thr ))
	{
	  mark_thread_ready( thr );
	}
      else
	{
	  /*  Negate the suspend count, and clear `blocked_on'.
	   *
	   *  When (and if) the thread is resumed, it will
	   *  look like it was waiting to run but had been
	   *  removed from the run queue, so it'll be immediately
	   *  requeued.
	   */
	  gvec_write_non_ptr( thr, THREAD_BLOCKED_ON, ZERO );
	}
    }
  else
    {
      /* ignore it */
    }
  return to_do;
}

static obj handle_accept_event( struct sys_event *e, obj to_do )
{
  int n, fd;
  char temp[64];
  obj item, mbox = gvec_read( EVENT_TARGET(e), SERVICE_MBOX );

  n = 64;

  fd = accept( e->data.fd, (struct sockaddr *)temp, &n );
  if (fd >= 0)
    {
      obj addr;

      addr = bvec_alloc( n, byte_vector_class );
      memcpy( PTR_TO_DATAPTR(addr), temp, n );
      item = cons( int2fx(fd), addr );
    }
  else
    {
      if (DEBUG_UNCOMMON_CASES)
        printf( "accept: %s\n", strerror(errno) );
      item = make_os_error("accept", 1, int2fx(e->data.fd));
      free_accept_event(e);
    }
  ksend_mailbox( mbox, item );
  return to_do;
}

static obj handle_std_read_event( struct sys_event *e, obj to_do )
{
  char temp[8192];
  int n;
  obj mbox = gvec_ref( EVENT_TARGET(e), MBOX_INPUT_PORT_MBOX );

  while (1)
    {
      n = read( e->data.fd, temp, 8192 );
      if (n < 0)
	{
	  if ((errno == EAGAIN) || (errno == EWOULDBLOCK))
	    {
	      return to_do;
	    }
	  else
	    {
	      /* error condition... */
	      ksend_mailbox( mbox,
			     make_os_error("read",1,int2fx(e->data.fd)) );
	      /* deallocate the event... if the file should still be
	       * read from, then it will have to be re-registered
	       */
	      free_file_event( e, &read_list );
	      return to_do;
	    }
	}
      else if (n == 0)
	{
	  /* EOF */
	  ksend_mailbox( mbox, FALSE_OBJ );
	  free_file_event( e, &read_list );
	  return to_do;
	}
      else
	{
	  obj str = bvec_alloc( n+1, string_class );
	  memcpy( string_text(str), temp, n );
	  if (DEBUG_THREAD_EVENTS)
	    printf( " event #%d {%p} -- read %d bytes from fd %d\n",
		    EVENT_NUMBER(e), e, n, e->data.fd );
	  ksend_mailbox( mbox, str );
	  /* loop to try to eat some more */
#if PERMIT_BLOCKING_READS
          /* if we are using/permitting blocking reads, don't loop -- wait for
	   * select() to tell use we have some more data ready
	   */
          return to_do;
#endif	  
	}
    }
}

static obj handle_read_event( struct sys_event *e, obj to_do )
{
  switch (e->type)
    {
    case SYS_EVENT_FILE_X_READ:
      return handle_read_x_event( e, to_do );
    case SYS_EVENT_FILE_ACCEPT:
      return handle_accept_event( e, to_do );
    default:
      return handle_std_read_event( e, to_do );
    }
}
