
(define *subprocess-table* #f)

#|
 *  This is used to run scheme-level operations detected in kernel mode.
 *
 *  The procedures are actually _run_ in kernel mode, so limited
 *  functionality is available (ie, there is no current-thread,
 *  so you can't try to block!)
 *
 *  Is this used for anything besides running error-proc's for
 *  <queued-output-port>?
 |#

(define (kernel fns)
  (let loop ((f fns))
    (if (pair? f)
	(begin
	  ((car f))
	  (loop (cdr f)))
	(kernel-next-thread))))

(define-glue (kernel-next-thread)
{
  return dispatch_to_next_thread();
})

(define-glue (get-current-thread)
{
  REG0 = current_thread;
  RETURN1();
})

(define-glue (current-thread-group)
{
  REG0 = gvec_ref( current_thread, THREAD_GROUP );
  RETURN1();
})

;; if we make the glue function be `current-thread', then the C
;; name will conflict with the variable/macro current_thread

(define-syntax (current-thread)
  (get-current-thread))

;;;


;;;

(define-safe-glue (thread-suspend (thread <thread>))
{
  mark_thread_suspended( thread );
  RETURN1();
})

(define-safe-glue (thread-resume (thread <thread>))
{
  mark_thread_resumed( thread );
  RETURN1();
})

(define (start-threads lst)
  (for-each thread-resume lst)
  (kernel-next-thread)) ;; never returns

;;;

(define-class <thunkified-combo> (<function>)
  (procedure type: <function>)
  (args type: <list> init-value: '()))

(define-safe-glue (thunkified-combo) :template
{
  obj self = envt_reg;
  REG0 = gvec_ref( self, SLOT(2) );
  arg_count_reg = 1;
  arg_count_reg = expand_last();
  APPLYF(arg_count_reg,gvec_ref( self, SLOT(1) ));
})

;;;

(define-safe-glue (thread-join (t <thread>))
{
  if (EQ(gvec_ref(t,THREAD_STATE),int2fx(TSTATE_COMPLETE)))
   {
     REG0 = gvec_ref( t, THREAD_STACK );
     arg_count_reg = 1;
     arg_count_reg = expand_last();
     RETURN(arg_count_reg);
   }
  else
   {
     SAVE_CONT1( do_join );
     gvec_set( t, 
	       THREAD_JOINS, 
	       cons( current_thread, gvec_ref( t, THREAD_JOINS ) ) );
     SWITCH_THREAD( t, TSTATE_BLOCKED );
  }
}
("do_join" {
  /* we get resumed with SAVED REG0 => values */
  RESTORE_CONT1();
  arg_count_reg = 1;
  arg_count_reg = expand_last();
  RETURN(arg_count_reg);
}))

(define-glue (thread-entry) :template
{
  RESTORE_CONT0();
  SAVE_CONT0(thread_done);
  APPLY(0,envt_reg);
}
("thread_done" {
  obj p;
  COLLECT0();
  RESTORE_CONT0();

  /* save the return values in the `thread-stack' slot */
  gvec_set( current_thread, THREAD_STACK, REG0 );

  /* clear out other slots whose values are no longer needed */
  gvec_set( current_thread, THREAD_VARS, FALSE_OBJ );
  gvec_set( current_thread, THREAD_DYNAMIC_STATE, FALSE_OBJ );

  if (DEBUG_THREAD_SWITCH)
    printf( " [%s] thread is done\n", thread_name(current_thread) );

  for (p=gvec_ref( current_thread, THREAD_JOINS ); !NULL_P(p); p=pair_cdr(p))
   {
     obj jt = pair_car(p);
     assert( EQ( gvec_ref( jt, THREAD_BLOCKED_ON ), current_thread ));
     gvec_write_non_ptr( jt, THREAD_BLOCKED_ON, ZERO );
     store_resume_value( jt, REG0 );
     mark_thread_ready( jt );
   }
  gvec_write_non_ptr( current_thread, THREAD_JOINS, NIL_OBJ );
  SWITCH_THREAD( ZERO, TSTATE_COMPLETE );
}))

(define-glue (time-slice-over)
{
  if (DEBUG_THREAD_SWITCH)
     printf( " [%s] thread interrupted by timer\n", 
	     thread_name(current_thread) );
  return did_timeout();
})

;;;


(%early-once-only
 (define *thread-sys-classes*
   (vector <thread-queue>         ;; 0
	   <thunkified-combo>     ;; 1
	   <thread>               ;; 2
	   <semaphore>            ;; 3
	   <queued-output-port>   ;; 4
	   <mailbox>              ;; 5
	   )))

(define-glue (init-threads-glue)
  literals: ((& *thread-sys-classes*)
	     (& kernel)
	     (& thunkified-combo))
{
  init_threads( TLREF(0), TLREF(1), TLREF(2) );
  RETURN0();
})

(define (init-threads)
  (set! *subprocess-table* (make-table eq? integer->hash))
  (init-threads-glue))
  
;;; initialize threads during system startup
;;; (note that we don't actually START threads yet)

(init-threads)
