/* Implementation of TLSymbol class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLSymbol.m,v 1.2 1998/02/23 14:17:41 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLSymbol.h"
#import "tl/TLSymbolValue.h"
#import "tl/TLLSubroutine.h"
#import "tl/TLCons.h"
#import "tl/TLDictionary.h"
#import "tl/TLFILEStream.h"
#import "tl/subr.h"
#import "tl/predicates.h"
#if NEXT_RUNTIME
#import <objc/objc-runtime.h>
#endif
#if GNU_RUNTIME
#import <objc/encoding.h>
#endif
#import <ctype.h>

/* The dictionary used to store all symbols.  */
TLDictionary *_tll_symbol_table;

/* Pointer to previous garbage protection function.  */
static id (*previous_garbage_protect) (void);

/* The stack of variable values.  */
static TLSymbolValue **bind_stack;
static int bind_num, bind_cap;

/* Predefined symbols.  */
TLSymbol *Qnil, *Qt, *Qobjc_class_list;
TLSymbol *Q_stdin_, *Q_stdout_, *Q_stderr_;

/* Predefined symbols.  */
struct
{
  /* The symbols name.  */
  char *name;

  /* The location of the value.  */
  void *location;

  /* It's encoding.  */
  char *encoding;
} predefines[] =
{
#define PREDEF(SYM, VAR)   {(SYM), &(VAR), @encode (__typeof__ (VAR))}
  PREDEF ("tlgc-alloc-since-partial", tlgc_alloc_since_partial),
  PREDEF ("tlgc-alloc-since-complete", tlgc_alloc_since_complete),
  PREDEF ("tlgc-num-run-complete", tlgc_num_run_complete),
  PREDEF ("tlgc-num-run-partial", tlgc_num_run_partial),
  PREDEF ("tlgc-num-alloc", tlgc_num_alloc),
  PREDEF ("tlgc-num-total", tlgc_num_total),
  PREDEF ("tlgc-partial-threshold", tlgc_partial_threshold),
  PREDEF ("tlgc-partial-time-limit", tlgc_partial_time_limit),
  PREDEF ("tlgc-total-threshold", tlgc_total_threshold),
  PREDEF ("tlgc-total-time-limit", tlgc_total_time_limit),
  PREDEF ("tlgc-alloc-limit", tlgc_alloc_limit),

#ifdef DEBUG_GC
  PREDEF ("tlgc-debug-level", debug_gc),
#endif

  PREDEF ("*standard-in*", V_stdin_),
  PREDEF ("*standard-out*", V_stdout_),
  PREDEF ("*standard-err*", V_stderr_),

  PREDEF ("symbol-table", _tll_symbol_table),
#undef PREDEF
  {0, 0, 0}
};

static struct class_node
{
  struct class_node *next;
  struct objc_class *class;
} *objc_classes;

/* Create read-only selector symbols for every object-only method in the
   method list.  */
static void
digest_method_list (Class cls)
{
  struct objc_method_list *list;
  const char *s;
  int i, n;

/* On OpenStep OBJC_NEXT_METHOD_LIST is defined in objc-class.h  */
#ifdef OBJC_NEXT_METHOD_LIST
  void *iterator = 0;

  while ((list = class_nextMethodList (cls, &iterator)))
#else
  list = cls->methods;

  for (; list; list = list->method_next)
#endif
    for (i = 0; i < list->method_count; i++)
      {
	for (n = 0, s = list->method_list[i].method_types; *s; s++)
	  if (isdigit (*s) || *s == '+'
#if GNU_RUNTIME
	      || *s == _C_CONST || *s == _C_IN || *s == _C_INOUT
	      || *s == _C_OUT || *s == _C_BYCOPY || *s == _C_ONEWAY
#endif
	      )
	    ;
	  else if (*s == '@'
		   /* Return type can be void.  */
		   || (n == 0 && *s == 'v')
		   /* Second argument is the selector.  */
		   || (n == 2 && *s == ':'))
	    n++;
	  else
	    break;
	if (!*s)
	  [[TLSymbol
	    symbolWithName: [TLString stringWithCString:
			     sel_get_name (list->method_list[i].method_name)]]
	   setIsSelector: n - 3];
      }
} /* digest_method_list */

/* Create read-only selector symbols for every object-only method of the
   class.  */
static void
digest_class_methods (struct objc_class *class)
{
#if GNU_RUNTIME
  digest_method_list (class);
  digest_method_list (class->class_pointer);
#else
  digest_method_list (class);
  digest_method_list (class->isa);
#endif
} /* digest_class_methods */

/* Protect the binding stack.  */
static id
tlsymbol_garbage_protect (void)
{
  int i;

  for (i = 0; i < bind_num; i++)
    MARK (bind_stack[i]);

  if (previous_garbage_protect)
    return (previous_garbage_protect ());
  return (nil);
} /* tlsymbol_garbage_protect */

void
tlsymbol_save_class_list (void)
{
  struct objc_class *class;

#if GNU_RUNTIME
  void *state = NULL;
  while ((class = objc_next_class (&state)))
#endif
#if NEXT_RUNTIME
  NXHashTable *class_hash = objc_getClasses ();
  NXHashState state = NXInitHashState (class_hash);
  while (NXNextHashState (class_hash, &state, (void **) &class))
#endif
    {
      struct class_node *c = xmalloc (sizeof (*c));
      c->class = class;
      c->next = objc_classes;
      objc_classes = c;
    }
} /* tlsymbol_save_class_list */

@implementation TLSymbol

#ifdef TL_0_14_COMPATIBLE
-(id <TLString>) name
{
  return [self symbolName]
}
#else
/* This is here for easy breakpointing.  */
-(const char *) name
{
  return [super name];
}
#endif

+initialize
{
  struct class_node *class;
  TLCons *c = nil;
  int i;

  /* Have we been here?  */
  if (_tll_symbol_table)
    return (self);

  /* Install the symbol table.  */
  _tll_symbol_table = [TLDictionary dictionary];
  [_tll_symbol_table gcLock];

  /* Some very basic symbols.  */
  Qt = [self symbolWithName: @"t"];
  [Qt setVarValue: Qt];
  [Qt setVarReadOnly: Qt];
  Qnil = [self symbolWithName: @"nil" varValue: nil];
  [Qnil setVarReadOnly: Qt];

  /* Define the predefined symbols.  */
  for (i = 0; predefines[i].name; i++)
    [[self symbolWithName: [TLString stringWithCString: predefines[i].name]]
      setCValue: predefines[i].location encoding: predefines[i].encoding];

  /* Install the class lists.  */
  for (class = objc_classes; class; class = class->next)
    {
      /* Set the symbol's class to the class found through the name, so the
         proper class is selected when being posed.  */
      c = CONS ([[self symbolWithName:
		  [TLString stringWithCString: class->class->name]]
		 setIsObjcClass: objc_get_class (class->class->name)], c);
    }
  Qobjc_class_list = [self symbolWithName: @"objc-class-list"];
  [Qobjc_class_list setVarValue: c];
  [Qobjc_class_list setVarReadOnly: Qt];

  /* Install the binding stack garbage protection.  */
  previous_garbage_protect = tl_garbage_protect;
  tl_garbage_protect = tlsymbol_garbage_protect;

  return (self);
} /* +initialize */

+(TLSymbol *) symbolWithName: (id <TLString>) a_name
{
  TLSymbol *o = [_tll_symbol_table objectForKey: a_name];

  if (!o)
    {
      o = [[self gcAlloc] initWithName: a_name];
      [_tll_symbol_table setObject: o forKey: a_name];
    }
  return (o);
} /* +symbolWithName: */

/******************** public methods ********************/

+(TLSymbol *) symbolWithName: (id <TLString>) n funValue: f
{
  TLSymbol *s = [self symbolWithName: n];
  [s setFunValue: f];
  return (s);
} /* +symbolWithName:funValue: */

+(TLSymbol *) symbolWithName: (id <TLString>) n varValue: v
{
  TLSymbol *s = [self symbolWithName: n];
  [s setVarValue: v];
  return (s);
} /* +symbolWithName:varValue: */

+(TLSymbol *) symbolWithName: (id <TLString>) n constantValue: v
{
  TLSymbol *s = [self symbolWithName: n];
  [s setVarValue: v];
  [s setVarReadOnly: Qt];
  return (s);
} /* +symbolWithName:constantValue: */

-autoload: (id <TLString>) file
{
  if (fvt != FVT_FUNCTION)
    {
      [self setFunValue: file];
      fvt = FVT_AUTOLOAD;
    }
  return (self);
} /* -autoload: */

-boundp
{
  return (vvt == VVT_OBJECT || vvt == VVT_ENCODED_C || vvt == VVT_OBJC_CLASS
	  ? Qt : nil);
} /* -boundp */

-(int) compare: (TLSymbol *) s
{
  /* We're always larger than anything which is not a symbol.  */
  if (!SYMBOLP (s))
    return (1);
  /* Compare names if we're not the same, just so symbols can be ordered on
     their name.  */
  return (self == s ? 0 : [name compare: [s symbolName]]);
} /* -compare: */

-equal: o
{
  return (o == self ? Qt : nil);
} /* -equal: */

-eval
{
  return ([self varValue]);
} /* -eval */

-evalWithArguments: (TLCons *) args
{
  switch (fvt)
    {
    case FVT_UNSET:
      if (!selector)
	/* Barf!  */
	return ([self funValue]);
      else
	/* This symbol is not fboundp, but it is a valid selector.
	   Try that.  */
	{
	  TLCons *receiver;

	  if (!args)
	    [self error: "Not enough arguments"];

	  DECONS (args, receiver, args);
	  if (!args)
	    return (tll_invoke_method (receiver, self, NULL, 0, 1));
	  else
	    {
	      int i, num_args;
	      id *argv;

	      num_args = [args length];
	      argv = alloca (num_args * sizeof (*argv));
	      for (i = 0; i < num_args; i++)
		DECONS (args, argv[i], args);

	      return (tll_invoke_method (receiver, self, argv, num_args, 1));
	    }
	}

    case FVT_AUTOLOAD:
    case FVT_FUNCTION:
    {
#if SUPPORT_DEBUG
      int invocation_level = tll_invocation_num;
      tll_invocation_info *ii = &tll_invocation_stack[invocation_level];

      /* Create a new invocation, in case this has not been done by a symbol
         forwarding its funvalue to us.  */
      if (!invocation_level || ii->argc != -2)
	{
	  ii = &tll_invocation_stack[tll_invocation_new ()];
	  ii->argc = -2;
	  ii->name = self;
	}
#endif
      return (EVAL_WITH_ARGS ([self funValue], args));
    }

    default:
      break;
    }

  abort ();

  /* Keep compiler happy.  */
  return (nil);
} /* -evalWithArguments: */

-fboundp
{
  return (fvt == FVT_FUNCTION || fvt == FVT_AUTOLOAD ? Qt : nil);
} /* -fboundp */

-fset: f
{
  [self setFunValue: f];
  return (f);
} /* -fset: */

-funValue
{
  switch (fvt)
    {
    case FVT_FUNCTION:
      break;
    case FVT_UNSET:
      [self error: "symbol without function value"];
      break;
    case FVT_AUTOLOAD:
      EVAL_WITH_ARGS ([isa symbolWithName: @"load"], CONS (fun_value, nil));
      if (fvt == FVT_AUTOLOAD)
	[self error: "loading %# does not define function %#", fun_value, self];
      /* Do this with tail recursion to avoid slowing down the normal case.  */
      return ([self funValue]);
    }
  return (fun_value);
} /* -funValue */

-initWithName: (id <TLString>) a_name
{
  vvt = VVT_UNSET;
  fvt = FVT_UNSET;
  ASGN_IVAR (name, a_name);
  return (self);
} /* -initWithName: */

-makunbound
{
  if (ro_var)
    [self error: "attempt to makunbound read-only symbol"];
  else if (vvt == VVT_ENCODED_C)
    [self error: "attempt to makunbound C-bound symbol"];

  vvt = VVT_UNSET;
  return (self);
} /* makunbound */

-(id <TLString>) symbolName
{
  return (name);
} /* -symbolName */

-(void) print: (id <TLOutputStream>) stream quoted: (BOOL) qp
{
  if (name)
    [[name string] print: stream quoted: qp withQuotes: NO];
  else
    print (name, stream, qp);
} /* -print:quoted: */

-set: v
{
  [self setVarValue: v];
  return (v);
} /* -set: */

-(void) setCValue: (void *) address encoding: (char *) encoding
{
  if (ro_var)
    [self error: "attempt to set read-only symbol"];
  if (!encoding || !encoding[0] || encoding[1])
    [self error: "bad encoding for symbol's c_value: `%s'", encoding];

  vvt = VVT_ENCODED_C;
  extra = *encoding;
  if (extra == '@')
    {
      ASGN_IVAR (var_value, * (id *) address);
      var_value = address;
    }
  else
    var_value = address;
} /* -setCValue:encoding: */

-setName: (id <TLString>) new_name
{
  ASGN_IVAR (name, new_name);
  return (self);
} /* -setName: */

-setIsObjcClass: (struct objc_class *) class
{
  if (vvt == VVT_OBJC_CLASS);
  else if (ro_var)
    [self error: "attempt to setIsObjcClass read-only symbol"];
  else
    {
      var_value = class;
      vvt = VVT_OBJC_CLASS;
      ro_var = 1;
      digest_class_methods (class);
      [self put: [isa symbolWithName: @"variable-documentation"]
       : @"Objective-C class"];
    }
  return (self);
} /* -setIsObjcClass: */

-setIsSelector: (int) num_args
{
  if (!selector)
    {
      selector = sel_get_any_uid ([name cString]);
      if (!selector)
	[self error: "not a valid selector in setIsSelector"];
      sel_num_args = num_args;
    }
  return (self);
} /* -setIsSelector: */

-(void) setVarValue: v
{
  if (ro_var)
    [self error: "attempt to set read-only symbol"];
  else if (vvt == VVT_ENCODED_C)
    switch (extra)
      {
      case '@':
	ASGN_IVAR (*(id *) var_value, v);
	break;

      case 'd':
	*(double *) var_value = [v doubleValue];
	break;

      case 'f':
	*(float *) var_value = [v floatValue];
	break;

      case 'i':
      case 'I':
	*(int *) var_value = [v intValue];
	break;

      case 'l':
      case 'L':
	*(long *) var_value = [v longValue];
	break;

      default:
	[self error: "unhandled c value encoding: `%c'", extra];
	break;
      }
  else
    {
      vvt = VVT_OBJECT;
      var_value = v;
    }
} /* -setVarValue: */

-setVarReadOnly: ynp
{
  ro_var = !!ynp;
  return (self);
} /* -setVarReadOnly: */

-(void) setFunValue: f
{
  if (ro_fun)
    [self error: "attempt to fset read-only symbol"];
  fvt = FVT_FUNCTION;
  fun_value = f;
} /* -setFunValue: */

-setFunReadOnly: ynp
{
  ro_fun = !!ynp;
  return (self);
} /* -setFunReadOnly: */

-symbolp
{
  return (Qt);
} /* -symbolp */

-varValue
{
  switch (vvt)
    {
    case VVT_UNSET:
      [self error: "symbol without variable value"];
      break;

    case VVT_OBJECT:
    case VVT_OBJC_CLASS:
      return (var_value);

    case VVT_ENCODED_C:
      switch (extra)
	{
	case '@':
	  return (*(id *) var_value);
	case 'd':
	  return ([CO_TLNumber numberWithDouble: *(double *) var_value]);
	case 'f':
	  return ([CO_TLNumber numberWithFloat: *(float *) var_value]);
	case 'i':
	  return ([CO_TLNumber numberWithInt: *(int *) var_value]);
	case 'I':
	  return ([CO_TLNumber numberWithUnsignedInt:
		   *(unsigned int *) var_value]);
	case 'l':
	  return ([CO_TLNumber numberWithLong: *(long *) var_value]);
	case 'L':
	  return ([CO_TLNumber numberWithUnsignedLong:
		   *(unsigned long *) var_value]);
	default:
	  [self error: "unhandled c value encoding: `%c'", extra];
	  break;
	}
    }

  abort ();
  return (nil);
} /* -varValue */

/******************** selectors ********************/

-mboundp
{
  return (selector ? Qt : nil);
} /* -mboundp */

-(int) selNumArguments
{
  if (!selector)
    [self error: "selNumArguments asked to a symbol which is not a selector"];
  return (sel_num_args);
} /* -selNumArguments */

-(SEL) selSelector
{
  if (!selector)
    [self error: "selSelector asked to a symbol which is not a selector"];
  return (selector);
} /* -selSelector */

/******************** properties ********************/

-get: property
{
  TLCons *p, *np;

  for (p = properties; p; p = np)
    {
      DECONS (p, p, np);
      if (p == property)
	return ([np car]);
      np = [np cdr];
    }

  return (nil);
} /* -get: */

-(TLCons *) propertyList
{
  return (properties);
} /* -propertyList */

-put: property : value
{
  TLCons *p, *np;

  for (p = properties; p; p = np)
    {
      DECONS (p, p, np);
      if (p == property)
	{
	  [np setCar: value];
	  return (value);
	}
      np = [np cdr];
    }

  ASGN_IVAR (properties, CONS (property, CONS (value, properties)));
  return (value);
} /* -put:: */

/******************** variable binding ********************/

+(binding_stack_level) bindingLevel
{
  return (bind_num);
} /* +bindingLevel */

+(void) popVarValues: (int) level
{
  if (level > bind_num || level < 0)
    abort ();

  while (bind_num != level)
    [bind_stack[--bind_num] restoreSymbolValue];
} /* +popVarValues: */

-(int) pushVarValue: a_value
{
  int previous_level = bind_num;
  id previous_value = var_value;
  int unboundp = vvt == VVT_UNSET;

  if (bind_num == bind_cap)
    {
      bind_cap += 16;
      bind_stack = xrealloc (bind_stack, bind_cap * sizeof (bind_stack));
    }

  /* First set the value, which can fail.  Only then adjust the stack.
     XXX Not very nice in case intermediately the read-only was changed.  */
  /* XXX YYY ZZZ WRONG for for instance (let ((counts-since-gc)) ...).  */
  [self setVarValue: a_value];
  bind_stack[bind_num++] = (unboundp ?
			    [TLSymbolValue symbolValueWithSymbol: self]
			    : [TLSymbolValue symbolValueWithSymbol: self
			       value: previous_value]);
  return (previous_level);
} /* -pushVarValue: */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (name);
  MARK (properties);

  /* There's no need to gcMark Objective-C classes.  */
  if (vvt == VVT_OBJECT)
    MARK (var_value);
  else if (vvt == VVT_ENCODED_C && extra == '@')
    MARK (*(id *) var_value);

  if (fun_value && (fvt == FVT_FUNCTION || fvt == FVT_AUTOLOAD))
    MARK (fun_value);
} /* -gcReference */

@end

TLSymbol *
symbol (const char *name)
{
  return ([TLSymbol symbolWithName: [CO_TLString stringWithCString: name]]);
} /* symbol */
