<copyright> BucketSetElement class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: BucketSetElement.t,v 1.12 1999/09/08 19:36:12 tiggr Exp $</id>
    </copyright>

implementation class
BucketSetElement: BucketElement

end;

implementation instance
BucketSetElement
{
  <doc> The key/value in this bucket element.  </doc>
  Any value;
}

<doc> Apple the {block} to {value} and pass to {next}.  </doc>
void
  do Block block
{
  [block eval value];
  if (next != nil)
    [next do block];
}

Any
  key
{
  = value;
}

<doc> Designated initializer.  </doc>
id
  initWith All v
{
  value = Any (v);
  = self;
}

<doc> Return the value associated with the {key}, asking the {next}
    element if this element does not match.

    The implementation by {BucketSetElement}, considers its {value} as the
    both the key and the value.  </doc>
Any
  member All key
{
  = (value == key || [key equal value] ? value
     : !next ? nil : [next member key]);
}

<doc> Like member, but using the selector {cmp} to have the objects
    compare themselves.  </doc>
Any
  member All key
   equal selector cmp
{
  = (value == key || [key perform cmp with value] ? value
     : !next ? nil : [next member key equal cmp]);
}

Any
  memq All key
{
  = value == key ? value : !next ? nil : [next memq key];
}

<doc> Add the {key} to this bucket, if it is not already present.  Return
    the number by which this bucket's length has increased.  </doc>
int
  add All key
{
  if (value != key && ![key equal value])
    if (!next)
      {
  	next = [[isa alloc] initWith key];
  	= 1;
      }
    else
      = [next add key];
}

<doc> Add the {key} to this bucket, if it is not already present.  Return
    the number by which this bucket's length has increased.  </doc>
int
  addq All key
{
  if (value != key)
    if (!next)
      {
	next = [[isa alloc] initWith key];
	= 1;
      }
    else
      = [next addq key];
}

<doc> Remove this bucket, if it holds the {key}.  Return the number by
    which the length of this bucket list has decreased, and the
    replacement remainder of the bucket list.  </doc>
(id, int) (replacement, decrease)
  remove All key
{
  if (key == value || [key equal value])
    = (next, 1);
  else
    {
      if (!!next)
	{
	  /* What is faster?  A check and branch (frequently taken) or a
	     (frequent) overwrite with the same value, dirtying a cache
	     line?  */
	  id new_next;
	  (new_next, decrease) = [next remove key];
	  if (next != new_next)
	    next = new_next;
	}
      replacement = self;
    }
}

<doc> Remove this bucket, if it holds the {key}.  Return the number by
    which the length of this bucket list has decreased, and the
    replacement remainder of the bucket list.  </doc>
(id, int) (replacement, decrease)
  removeq All key
{
  if (key == value)
    = (next, 1);
  else
    {
      if (!!next)
	{
	  /* What is faster?  A check and branch (frequently taken) or a
	     (frequent) overwrite with the same value, dirtying a cache
	     line?  */
	  id new_next;
	  (new_next, decrease) = [next removeq key];
	  if (next != new_next)
	    next = new_next;
	}
      replacement = self;
    }
}

void
  encodeUsingCoder Encoder coder
{
  if (![coder hasBeenCodedFor [BucketSetElement self]])
    {
      [super encodeUsingCoder coder];

      [coder encode value];
    }
}

void
  initWithCoder Decoder coder
{
  if (![coder hasBeenCodedFor [BucketSetElement self]])
    {
      [super initWithCoder coder];

      value = [coder decode];
    }
}

<doc> Tell the {value} to {gc_container_mark_elements}.  </doc>
void
  gc_mark_containers
{
  [value gc_container_mark_elements];
  if (next != nil)
    [next gc_mark_containers];
}

<doc> Starting with this bucket element, remove the objects of which the
    {value} is {gc_dead}.  Return the replacement for this element, and
    the number of bucket elements that were removed from this bucket list.
    </doc>
(id, int)
  gc_mark_values
{
  int n;

  if (next != nil)
    (next, n) = [next gc_mark_values];

  if ([value gc_dead_p])
    return (next, n + 1);

  return (self, n);
}

<doc> Rehash the key of the receiving element.  </doc>
int
  rehash
{
  = [value hash];
}

<doc> Rehashq the key of the receiving element.  </doc>
int
  rehashq
{
  = [value hashq];
}

end;
