<copyright> IntegerRangeSet class.
    Written by <a href="mailto:michael@thi.nl">Michael L.H. Brouwer</a>
           and <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>.

    Copyright &copy; 1996, 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: IntegerRangeSet.t,v 1.20 1998/05/18 15:21:17 tiggr Exp $</id>
    </copyright>

/******************** IntegerRangeSet ********************/

<doc> The {IntegerRangeSet} is good at holding ranges of integers.  Ranges
    are stored in an unbalanced tree.  The number of ranges dictates the
    memory usage by the set.  Testing for membership is O(log n) where n
    is the number of ranges.  </doc>
implementation class
IntegerRangeSet: State, Enumerable

end;

implementation instance
IntegerRangeSet
{
  <doc> The root of the tree holding information.  </doc>
  IntegerRangeSetNode root;
}

<doc> Add {v} to the receiving set.  Return {FALSE} if it was already
    present; {TRUE} otherwise.  </doc>
boolean
  add int v
{
  if (!root)
    {
      root = [[IntegerRangeSetNode alloc] init (v, 1)];
      = TRUE;
    }
  else
    = [root add v];
}

boolean
  equal id other
{
  = [other equalIntegerRangeSetNode root];
}

boolean
  equalIntegerRangeSetNode IntegerRangeSetNode other
{
  if (other == nil || root == nil)
    = root == other;
  else
    = [root equalIntegerRangeSetNode other];
}

// This is an abominably slow implementation...
// Wed Jun 12 17:37:24 1996, tiggr@cobra.es.ele.tue.nl
<doc> Return a new set being the intersection of the receiving set and the
    {other} set.  </doc>
id
  intersectionWith id other
{
  id result = [isa new];
  boolean b;
  int i;

  for ((b, i) = [self smallestElement]; b; (b, i) = [self nextPresent i])
    if ([other member i])
      [result add i];

  = result;
}

boolean
  isEmpty
{
  = !root;
}

// This is an abominably slow implementation...
// Wed Jun 12 15:43:23 1996, tiggr@cobra.es.ele.tue.nl
<doc> Return {YES} if the receiving set is a subset of the {other} set.  </doc>
boolean
  isSubsetOf id other
{
  boolean b;
  int i;

  for ((b, i) = [self smallestElement]; b; (b, i) = [self nextPresent i])
    if (![other member i])
      return NO;

  = YES;
}

<doc> Return the highest value present in the set.  </doc>
(boolean, int) (non_empty, value)
  highestPresent
{
  if (root != nil)
    return (TRUE, [root highestPresent]);
}

<doc> Return the lowest value present in the set.  </doc>
(boolean, int) (non_empty, value)
  lowestPresent
{
  if (root != nil)
    return (TRUE, [root lowestPresent]);
}

<doc> Return {TRUE} iff the set contains {v}.  </doc>
boolean
  member int v
{
  = !root ? FALSE : [root member v];
}

<doc> Return the smallest element >= {i} that is not yet in the set.
    </doc>
int
  nextNonPresent int i
{
  = !root ? i : [root nextNonPresent i];
}

<doc> Return the smallest element > {i} that is contained in the tree,
    preceded by whether such an element actually is in the tree.  </doc>
(boolean, int)
  nextPresent int i
{
  = !root ? (FALSE, 0) : [root nextPresent i];
}

<doc> Return the largest element <= {i} that is not yet in the set.
    </doc>
int
  previousNonPresent int i
{
  = !root ? i : [root previousNonPresent i];
}

<doc> Return the largest element <= {i} that is in the set.  </doc>
(boolean, int)
  previousPresent int i
{
  if (root != nil)
    = [root previousPresent i];
}

<doc> Remove {v} from the set, returning {TRUE} if it was actually
    contained.  </doc>
boolean (b)
  remove int v
{
  if (root != nil)
    (root, b) = [root remove v];
}

<doc> Increase the value of all elements in the tree >= {i} by one.
    </doc>
void
  shiftFrom int i
{
  if (root != nil)
    [root shiftFrom i];
}

<doc> Return the smallest value contained, preceded by whether we're not
    empty.  </doc>
(boolean, int)
  smallestElement
{
  = !root ? (NO, 0) : (YES, [root smallestElement]);
}

// This is an abominably slow implementation...
// Wed Jun 12 12:37:23 1996, tiggr@cobra.es.ele.tue.nl
<doc> Modify the receiving set by adding the elements from the other set.
    </doc>
void
  uniteWith id other
{
  boolean b;
  int i;

  if (other != nil)
    for ((b, i) = [other smallestElement]; b; (b, i) = [other nextPresent i])
      [self add i];
}

OutputStream
  write OutputStream s
{
  = !root ? ["empty" write s] : [root write s offset 0];
}

/********** Enumerable **********/

Enumerator
  enumerator
{
  = [[IntegerRangeSetNodeEnumerator alloc] init root];
}

protected id
  initWithEnumerator Enumerator e
{
  [self unimplemented cmd];

  = self;
}

end;

/******************** IntegerRangeSetNode ********************/

implementation class
IntegerRangeSetNode: State

end;

implementation instance
IntegerRangeSetNode
{
  <doc> The left and right subtrees.  </doc>
  id left, right;

  <doc> The offset from our parent.  </doc>
  int offset;

  <doc> The number of integers in this node.  </doc>
  int size;
}

/********** public methods **********/

<doc> Add {v} to the tree rooted at the receiving node.  Return {FALSE} if
    it was already present; {TRUE} otherwise.  </doc>
boolean
  add int v
{
  int s;

  v -= offset;

  if (v < -1)
    {
      if (!left)
	{
	  left = [[isa alloc] init (v, 1)];
	  = TRUE;
	}
      else
    	= [left add v];
    }
  else if (v > size)
    {
      if (!right)
	{
	  right = [[isa alloc] init (v, 1)];
	  = TRUE;
	}
      else
    	= [right add v];
    }
  else if (v == size)
    {
      size++;
      if (!!right)
	{
	  (right, s) = [right mergeRLL v];
	  size += s;
	}

      = TRUE;
    }
  else if (v == -1)
    {
      (offset, size) = (offset - 1, size + 1);

      if (!!right)
	[right offset 1];

      if (!!left)
	{
	  [left offset 1];

	  (left, s) = [left mergeLRL 0];
	  if (s != 0)
	    {
	      (offset, size) = (offset - s, size + s);

	      if (!!left)
		[left offset s];

	      if (!!right)
		[right offset s];
	    }
	}

      = TRUE;
    }
}

boolean
  equalIntegerRangeSetNode id other
{
  id l, r;
  int o, s;

  (l, r, o, s) = [other dissect];

  if (offset != o || size != s)
    return NO;

  if (left == nil || l == nil)
    {
      if (left != l)
	return NO;
    }
  else if (![left equalIntegerRangeSetNode l])
    return NO;
    
  if (right == nil || r == nil)
    {
      if (right != r)
	return NO;
    }
  else if (![right equalIntegerRangeSetNode r])
    return NO;

  = YES;
}

<doc> Designated initializer.  </doc>
id
  init (int, int) (o, s)
{
  (offset, size) = (o, s);

  = self;
}

<doc> Return the highest value present in the set.  </doc>
int
  highestPresent
{
  return offset + (!right ? size - 1 : [right highestPresent]);
}

<doc> Return the lowest value present in the set.  </doc>
int
  lowestPresent
{
  return offset + (!left ? 0 : [left lowestPresent]);
}

<doc> Return {TRUE} iff the set contains {v}.  </doc>
boolean
  member int v
{
  v -= offset;

  if (left != nil && v < -1)
    = [left member v];
  else if (right != nil && v > size)
    = [right member v];
  else
    = 0 <= v && v < size;
}

<doc> Return the smallest element >= {i} that is not yet in the tree.
    </doc>
int
  nextNonPresent int i
{
  i -= offset;

  = offset + ({if (!!left && i < -1)
		 [left nextNonPresent i];
	       else if (!!right && i > size)
		 [right nextNonPresent i];
	       else if (0 <= i && i < size)
		 size;
	       else
		 i;});
}

<doc> Return the smallest element > {i} that is contained in the tree,
    preceded by whether such an element actually is in the tree.  </doc>
(boolean, int)
  nextPresent int i
{
  boolean b;

  i -= offset;

  if (i < -1)
    if (!left)
      (b, i) = (YES, 0);
    else
      {
	(b, i) = [left nextPresent i];
	if (!b)
	  (b, i) = (YES, 0);
      }
  else if (i >= size - 1)
    if (!right)
      (b, i) = (NO, 0);
    else
      (b, i) = [right nextPresent i];
  else
    (b, i) = (YES, i + 1);

  = (b, i + offset);
}

<doc> Return the largest element <= {i} that is not yet in the tree.
    </doc>
int
  previousNonPresent int i
{
  i -= offset;

  = offset + ({
	       if (!!left && i < -1)
		 [left previousNonPresent i];
	       else if (!!right && i > size)
		 [right previousNonPresent i];
	       else if (0 <= i && i < size)
		 -1;
	       else
		 i;});
}

<doc> Return the largest element in the tree which is smaller than {i}.  </doc>
(boolean, int)
  previousPresent int i
{
  boolean b;

  i -= offset;

  if (i <= 0)
    if (!left)
      (b, i) = (NO, 0);
    else
      (b, i) = [left previousPresent i];
  else if (i <= size)
    (b, i) = (YES, i - 1);
  else
    if (!right)
      (b, i) = (YES, size - 1);
    else
      {
	(b, i) = [right previousPresent i];
	if (!b)
	  (b, i) = (YES, size - 1);	  
      }

  return (b, offset + i);
}

<doc> Remove {v} from the receiving tree, returning the modified tree, and
    {TRUE} if it was actually removed.  </doc>
(id, boolean)
  remove int v
{
  boolean b;

  v -= offset;

  if (v < -1)
    {
      if (!left)
	= (self, FALSE);
      else
	{
	  (left, b) = [left remove v];
	  = (self, b);
	}
    }
  else if (v > size)
    {
      if (!right)
	= (self, FALSE);
      else
	{
	  (right, b) = [right remove v];
	  = (self, b);
	}
    }
  else if (v == -1 || v == size)
    = (self, FALSE);
  else
    {
      if (!--size)
	{
	  /* We can be removed.  */
	  if (left != nil)
	    {
	      [left offset offset];

	      if (right != nil)
		{
		  [right offset offset];
		  [left setRightMost right];
		}

	      return (left, TRUE);
	    }

	  if (right != nil)
	    {
	      [right offset offset];

	      return (right, TRUE);
	    }

	  return (nil, TRUE);
	}
      else if (!v)
	{
	  offset++;
	  if (!!right)
	    [right offset -1];
	  if (!!left)
	    [left offset -1];
	}
      else if (v != size)
	{
	  id p = [[isa alloc] init (v + 1, size - v)];

	  if (!!right)
	    {
	      [p set_right right];
	      [right offset -(v + 1)];
	    }

	  right = p;
	  size = v;
	}

      = (self, TRUE);
    }
}

<doc> Increase the value of all elements in the tree >= {i} by one.
    </doc>
void
  shiftFrom int i
{
  i -= offset;

  if (i < -1)
    {
      offset++;

      if (left != nil)
	{
	  [left offset -1];
	  [left shiftFrom i - 1];
	}
    }
  else if (i > size)
    {
      if (right != nil)
	[right shiftFrom i];
    }
  else if (i == size)
    {
      if (right != nil)
	[right offset 1];
    }
  else if (!i || i == -1)
    {
      offset++;
      if (left != nil)
	[left offset -1];
    }
  else
    {
      id m = [[isa alloc] init (i + 1, size - i)];

      if (right != nil)
	{
	  [m set_right right];
	  [right offset -i];
	}

      right = m;
      size = i;
    }
}

<doc> Return the smallest value contained.  </doc>
int
  smallestElement
{
  = !left ? offset : offset + [left smallestElement];
}

OutputStream
   write OutputStream s
  offset int i
{
  i += offset;

  if (!!left)
    s = ["," write [left write s offset i]];

  if (size == 1)
    s = [s print i];
  else
    s = [[[s print i] print "-"] print i + size - 1];

  if (!!right)
    s = [right write ["," write s] offset i];

  = s;
}

/********** protected methods **********/

<doc> Return the guts of this object.  </doc>
protected (id, id, int, int)
  dissect
{
  = (left, right, offset, size);
}

<doc> Merge the subtree rooted at this node, to accomodate the value {v},
    returning the modified tree and the extra size for our parent node.
    Our parent actually holds the value {v} (as the first value).  </doc>
protected (id, int)
  mergeLRL int v
{
  v -= offset;

  if (!right)
    {
      if (v == size)
	{
	  if (!!left)
	    [left offset offset];

	  = (left, v);
	}
      else
	= (self, 0);
    }
  else
    {
      int s;

      (right, s) = [right mergeLRL v];

      = (self, s);
    }
}

<doc> Merge the subtree rooted at this node, to accomodate the value {v},
    returning the modified tree and the extra size for our parent node.
    Our parent actually holds the value {v} (as the last value).  </doc>
protected (id, int)
  mergeRLL int v
{
  v -= offset;

  if (!left)
    {
      if (v == -1)
	{
	  if (!!right)
	    [right offset offset];

	  = (right, size);
	}
      else
	= (self, 0);
    }
  else
    {
      int s;

      (left, s) = [left mergeRLL v];

      = (self, s);
    }
}

<doc> Adjust the {offset} by {n}.  </doc>
protected void
  offset int n
{
  offset += n;
}

<doc> Set the {right} node.  </doc>
protected void
  set_right id n
{
  right = n;
}

<doc> Set the right most node of the receiving tree to {r}.  </doc>
protected void
  setRightMost id r
{
  [r offset -offset];

  if (right != nil)
    [right setRightMost r];
  else
    right = r;
}

end;

/******************** IntegerRangeSetNodeEnumerator ********************/

implementation class
IntegerRangeSetNodeEnumerator: State, Enumerator

end;

implementation instance
IntegerRangeSetNodeEnumerator
{
  <doc> The root of the tree of nodes.  </doc>
  IntegerRangeSetNode root;

  <doc> The previous integer value retrieved.  </doc>
  int previous;
}

id
  init IntegerRangeSetNode r
{
  root = r;

  if (root != nil)
    previous = [root smallestElement] - 1;

  = self;
}

/********** Enumerator **********/

(boolean, Number)
  next
{
  boolean b;
  int i;

  (b, i) = [self next];

  = b ? (b, [IntNumber with i]) : (b, nil);
}

(boolean, int) (valid, value)
  next
{
  if (root != nil)
    {
      (valid, previous) = [root nextPresent previous];
      value = previous;
    }
}

end;
