<copyright> MutableArray class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1998 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: MutableArray.t,v 1.41 1998/08/17 14:56:39 tiggr Exp $</id>
    </copyright>

implementation class
MutableArray: Array, MutableIndexed

<doc> Return a new instance of the receiving class which can hold {cap}
    elements without the need for resizing.  </doc>
instance (id)
  withCapacity int cap
{
  = [[self alloc] initWithCapacity cap];
}

end;

implementation instance
MutableArray
{
  <doc> The capacity of the array.  </doc>
  public int capacity;
}

<doc> Store the {object} at the end in the receiving array.  If the
    receiving array stores unboxed values, the {object} is queried for its
    value.  </doc>
void
  add All object
{
  self[length] = object;
}

<doc> Adjust the range ({start}, {len}) to fit the capacity of the receiving
    {MutableArray}.  If {len} == -1, it is adjusted to fit the capacity.
    </doc>
(int, int)
  adjustMutableRange (int, int) (start, len)
{
  if (start >= length)
    = (start, 0);
  else
    {
      if (len == -1)
	len = capacity - start;

      if (start < 0)
	{
	  len += start;
	  start = 0;

	  if (len < 0)
	    len = 0;
	}

      if (start + len > capacity)
	len = capacity - start;
    }

  = (start, len);
}

<doc> Empty the receiving array.  This frees any storage used by the array
    to store its elements.  </doc>
void
  empty
post
  length == 0
{
  length = capacity = 0;
  [C free contents];
  contents = ({pointer null;});
}

<doc> Initialize the newly allocated receiving object to be able to hold
    {capacity} items without needing to resize.  </doc>
deferred id
  initWithCapacity int capacity;

<doc> Initialize with the indicated pointer and integer for contents and
    length.  The {capacity} is set to the {length}.  </doc>
id
  initWith int n
	at pointer addr
{
  [super initWith n at addr];

  capacity = length;
  = self;
}

<doc> Insert the {object} at the {index}, shifting the objects at that or
    a higher index up by 1 position.  </doc>
deferred void
  insert All object
      at int index
pre
  index >= 0 && index <= length;

<doc> Remove the element from {index}, decreasing the index of all
    elements after {index}, and return the element boxed.  If the
    receiving array stores unboxed values, such as integers, the value
    returned is the element boxed.  </doc>
deferred Any
  removeAt int index;

<doc> Remove the element from {index}, decreasing the index of all
    elements after {index}.  </doc>
void
  removeElementAt int index
{
  [self removeElements (index, 1)];
}

<doc> Remove the {length} elements from {start}.  If {length} == -1, all
    elements from {start} are removed.  </doc>
deferred void
  removeElements (int, int) (start, length);

<doc> Adjust the size of the array, filling any newly created entries with
    the default value for the type (i.e. 0).  </doc>
void
  resize int to
pre
  to >= 0
{
  if (to > length)
    [self resize (length, to - length)];
  else if (length > to)
    [self truncate to];
}

<doc> Adjust the size of the array by inserting {num} new entries at
    {start} filling newly created entries with the default values for the
    type (i.e. 0).  </doc>
deferred void
  resize (int, int) (start, num);

<doc> Adjust the length of this array to {new_length}.  </doc>
void
  truncate int new_length
pre
  new_length >= 0
{
  if (new_length < length)
    length = new_length;
}

/******************** sorting ********************/
// Sorting methods should be provided by MutableIndexed.
// Sun Sep 15 22:45:21 1996, tiggr@tricky.es.ele.tue.nl

<doc> Bublesort the receiving array on the {key} of the contained elements
    by comparing them using the {compare} selector.  </doc>
void
  bubbleSortUsingKey selector key
	 comparator: selector cmp = selector (int compare All)
{
  [self bubbleSort (key, cmp) range (0, length)];
}

<doc> Hard working bubblesort (also used by the quicksorter for the last
    bits).  </doc>
private void
  bubbleSort (selector, selector) (key, compare)
       range (int, int) (start, beyond)
{
  int i, j, k;
  All a, b;

  for (i = start; i < beyond; i++)
    for (j = beyond - 1; j > i; j--)
      {
	a = [self[j] perform key];
	b = [self[j - 1] perform key];
	k = [a perform compare with b];

	if (k < 0)
	  [self swap (i, j)];
      }
}

<doc> Quicksort the receiving array on the {key} of the contained elements
    by comparing them using the {compare} selector.  </doc>
void
  quickSortUsingKey selector key
	comparator: selector compare = selector (int compare All)
{
  [self quickSort (key, compare) range (0, length)];
}

<doc> The hard worker for {quickSortUsingKey comparator}.  </doc>
private void
  quickSort (selector, selector) (key, compare)
      range (int, int) (start, beyond)
{
  if (beyond - start < 4)
    {
      [self bubbleSort (key, compare) range (start, beyond)];
      return;
    }

  All pivot_key = [self[(start + beyond - 1) / 2] perform key];
  int left = start - 1, right = beyond;

  for (;;)
    {
      // CCC Without using the O and I, the compiler has a problem.
      // Fri Sep  6 14:31:27 1996, tiggr@cobra.es.ele.tue.nl
      All o;
      int i;

      do
	{
	  o = [self[++left] perform key];
	  i = [o perform compare with pivot_key];
	} while (i < 0);
      do
	{
	  o = [self[--right] perform key];
	  i = [o perform compare with pivot_key];
	} while (i > 0);

      if (left >= right)
	break;

      [self swap (left, right)];
    }

  [self quickSort (key, compare) range (start, right + 1)];
  [self quickSort (key, compare) range (right + 1, beyond)];
}

/******************** copying ********************/
<doc> <h4>Copying</h4> </doc>

<doc> In addition to what our {super} does, adjust our (new) {capacity} to
    fit our {length}.  </doc>
id
  initCopy
{
  [super initCopy];

  capacity = length;

  = self;
}

end;
