<copyright> Sorted classes.
    Written by <a href="mailto:silovic@zesoi.fer.hr">Miroslav Silovic</a>

    Copyright &copy; 1999 Pieter J. Schoenmakers

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Sorted.t,v 1.2 1999/09/27 20:18:31 tiggr Exp $</id>
    </copyright>

/***************************** SortedKeyed ********************/

<doc> The SortedKeyed class keeps its elements in ascending order.  </doc>
implementation class
SortedKeyed: Keyed

end;

implementation instance
SortedKeyed

<doc> Enumerate the elements in an interval.  {nil} at the either end
    signifies the first or the last element.  </doc>
deferred Enumerator
  between (Comparable, Comparable) (start, last);

deferred Any
  lowest;

deferred Any
  highest;

deferred redeclare Any
  at Comparable object;

deferred redeclare Any
  member Comparable object;

deferred redeclare Any
  memq Comparable object;

end;

/************************ MutableSortedKeyed ********************/

implementation class
MutableSortedKeyed: SortedKeyed, MutableKeyed

end;

implementation instance
MutableSortedKeyed

redeclare void
  add Comparable object;

redeclare void
  remove Comparable object;

end;

/************************ SortedMapped *********************/

implementation class
SortedMapped: Mapped, SortedKeyed

end;

implementation instance
SortedMapped

<doc> Enumerate the values in an interval.  </doc>
deferred Enumerator
  valuesOfKeysBetween (Comparable, Comparable) (start, last)
	 includeLeft: boolean incleft = TRUE
	includeRight: boolean incright = TRUE;

end;

/******************** MutableSortedMapped ******************/

implementation class
MutableSortedMapped: MutableSortedKeyed, MutableMapped

end;

implementation instance
MutableSortedMapped

deferred redeclare void
  set All value
   at Comparable key;

end;

/******************** simple implementation ***************/

implementation class
SortedObjectArray: SortedKeyed

end;

implementation instance
SortedObjectArray
{
  <doc> The array we employ to actually store the contents.  </doc>
  public MutableObjectArray contents;
}

boolean (result)
  verifySortedContents
{
  int len = [contents length];
  result = TRUE;

  if (len > 0)
    {
      Comparable current = contents[0];
      int i;

      for (i = 1; i < len; i++)
	{
	  Comparable next = contents[i];
	  if ([current compare next] == 1)
	    {
	      result = FALSE;
	      break;
	    }
	  current = next;
	}
    }
}

id
  initWithSortedEnumerator Enumerator e
post
  [self verifySortedContents]
{
  contents = [[MutableObjectArray alloc] initWithEnumerator e];
}

id
  initWithEnumerator Enumerator e
{
  contents = [[MutableObjectArray alloc] initWithEnumerator e];
  [contents quickSortUsingKey selector (Any self Any)];
}

ObjectArray
  allKeys
{
  = [contents copy];
}

<doc> The guts of the binary search algorithm.  </doc>
(boolean, int)
  indexOf Comparable object
{
  int left = 1, right = [contents length];

  while (left <= right)
    {
      int mid = (left + right) / 2;
      int cmp = [Comparable (contents[mid]) compare object];

      if (cmp == 0)
	return (TRUE, mid);

      if (cmp == -1)
	left = mid + 1;
      else
	right = mid - 1;
    }

  = (FALSE, left);
}

Any
  at Comparable object
{
  boolean present;
  int index;

  (present, index) = [self indexOf object];
  = present ? contents[index] : nil;
}

Enumerator
  valuesOfKeysBetween (Comparable, Comparable) (start, last)
	 includeLeft: boolean incleft = TRUE
	includeRight: boolean incright = TRUE
{
  boolean p1, p2;
  int istart, ilast;

  (p1, istart) = [self indexOf start];
  (p2, ilast) = [self indexOf last];

  if (p1 && !incleft)
    istart++;

  if (!incright)
    ilast--;

  = [[IndexedEnumerator alloc] init contents start: istart
			       length ilast - istart + 1];
}

Any
  lowest
pre
  [contents length] != 0
{
  = contents[0];
}

Any
  highest
pre
  [contents length] != 0
{
  return contents[[contents length] - 1];
}

end;

implementation class
MutableSortedObjectArray: MutableSortedKeyed, SortedObjectArray

end;

implementation instance
MutableSortedObjectArray

void
  empty
{
  contents = [MutableObjectArray withCapacity 0];
}

// Freeze must be re-thought since it can not work without some work.
// (Extensions's state may have a different offset in different classes).
// Mon Sep 27 21:41:07 1999, tiggr@thesis.gerbil.org
void
  freeze
{
  [self set_kind [SortedKeyed kind]];
}

<doc> Add an object.

    Note that adding elements one-by-one to a {MutableSortedObjectArray}
    will work as an insertion sort, with quadratic performance.  Use
    {addElementsFrom} with another sorted set instead, as it is much more
    efficient.  Also, do not use {addElementsFromEnumerator} unless you know
    that the enumerator is sorted (and then use
    {addElementsFromSortedEnumerator}.  </doc>
void
  add Comparable object
{
  int i, len = [contents length];

  // Why is this here?
  // Mon Sep 27 21:42:37 1999, tiggr@thesis.gerbil.org
  [contents add nil];

  Comparable last = object, current;

  for (i = len - 1; i >= 0; i--)
    {
      current = contents[i];
      if ([last compare current] == -1)
	{
	  contents[i + 1] = last;
	  last = current;
	}
      else
	break;
    }

  contents[i + 1] = last;
}

<doc> {addElementsFrom} performs in time proportional to the *sum* of the
    number of elements in both collections, if the other collection is a
    {SortedKeyed}.  Otherwise the complexity is equal to the *product* of
    the number of the elements.  </doc>
void
  addElementsFrom Collection object
{
  Enumerator e = [object enumerator];

  // Wrong! Use of isKindOf is not allowed!
  // Instead, tell the other collection to {feedElementsToSorted self},
  // with a speedy implementation of that method by SortedKeyed.
  // Mon Sep 27 21:44:27 1999, tiggr@thesis.gerbil.org
  if ([object isKindOf [SortedKeyed kind]]) {
    /* we can do this efficiently */
    [self addElementsFromSortedEnumerator e];
  }
  /* fallback to the *slow* Collection implementation */
  else
    [self addElementsFromEnumerator e];
}

void
  addElementsFromSortedEnumerator Enumerator j
post
  [self verifySortedContents]
{
  int i, len = [contents length];
  MutableObjectArray out = [MutableObjectArray withCapacity len];
  boolean b = FALSE;
  Comparable o2;

  (b, o2) = [j next];
  while (i < len && b)
    {
      Comparable o1 = contents[i];
      int res = [o1 compare o2];

      if (!res)
	{
	  [out add o1];
	  i++;
	  (b, o2) = [j next];
	}
      else if (res > 0)
	{
	  [out add o2];
	  (b, o2) = [j next];
	}
      else
	{
	  [out add o1];
	  i++;
	}
    }

  while (i < len)
    [out add contents[i++]];

  while (b)
    {
      [out add o2];
      (b, o2) = [j next];
    }

  contents = out;
}

void
  removeElementsFrom Collection object
{
  Enumerator e = [object enumerator];

  // Likewise: do not use isKindOf.
  // Mon Sep 27 21:46:35 1999, tiggr@thesis.gerbil.org
  if ([object isKindOf [SortedKeyed kind]]) {
    [self removeElementsFromSortedEnumerator e];
  }
  else
    [self removeElementsFromEnumerator e];
}

void
  removeElementsFromSortedEnumerator Enumerator j
post
  [self verifySortedContents]
{
  int i, len = [contents length];
  MutableObjectArray out = [MutableObjectArray withCapacity len];
  boolean b = FALSE;
  Comparable o2;

  (b, o2) = [j next];
  while (i < len && b)
    {
      Comparable o1 = contents[i];
      int res = [o1 compare o2];

      if (!res)
	{
	  i++;
	  (b, o2) = [j next];
	}
      else if (res > 0)
	(b, o2) = [j next];
      else
	{
	  [out add o1];
	  i++;
	}
    }
  while (i < len)
    [out add contents[i++]];

  contents = out;
}

void
  keepElementsFrom Collection object
{
  // Likewise, do not use isKindOf.
  // Mon Sep 27 21:48:07 1999, tiggr@thesis.gerbil.org
  if ([object isKindOf [SortedKeyed kind]]) {
    Enumerator j = [object enumerator];
    int i, len = [contents length];
    boolean b = FALSE; Comparable o2;
    MutableObjectArray out = [MutableObjectArray withCapacity len];
    (b, o2) = [j next];
    while (i < len && b) {
      Comparable o1;
      int res = [(o1 = contents[i]) compare o2];
      if (!res) {
	[out add o1];
	i++;
	(b, o2) = [j next];
      } else if (res > 0) {
	(b, o2) = [j next];
      } else {
	i++;
      }
    }
    contents = out;
  }
  else
    [super keepElementsFrom object];
}

end;
