/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMCompound.m,v 1.46 1998/01/05 01:12:31 tiggr Exp $  */

#import "OTMCompound.h"
#import "OTMBasic.h"
#import "OTMCustomMethod.h"
#import "OTMVariable.h"
#import "OTMLocalVar.h"

@implementation OTMCompound

+(OTMCompound *) compoundWithContainer: (id) c
{
  return [[self gcAlloc] initWithContainer: c];
}

-(void) addStatement: (OTMTop *) stmt
{
  stmt = [stmt precompile];

  if (stmt)
    {
      //      if (![stats length] && [stmt line])
      //	[self getLocationFrom: stmt];
      [stats addElement: stmt];
    }
}

-(void) addVariable: (OTMVariable *) v
{
  id prev = nil;

  if (!vars)
    ASGN_IVAR (vars, [TLDictionary dictionary]);
  else
    {
      prev = [vars objectForKey: [v variableName]];
      if (prev)
	{
	  error_for (v, @"redeclaration of variable `%@'", [v variableName]);
	  cerror_for (prev, @"previous declaration was here");
	}
    }

  if (!prev)
    [vars setObject: v forKey: [v variableName]];
}

-(void) assignResult: (OTMExpr *) v
{
  if (!v)
    abort ();

  if (!result && [v type] == basic_type[BT_VOID])
    result = void_expr;
  else if (result != void_expr)
    {
      if (!result)
	{
	  id previous_compound = current_compound;

	  current_compound = container;
	  result = temp_something_with_type ([v type], YES);
	  current_compound = previous_compound;
	}

      emit_assignment (result, v);
    }
}

-(void) compile
{
#if STACK_REF_STRUCT
  TLVector *local_refs = nil;
#endif
  OTMVariable *v;

  formac (of, @"%@{", [self nl]);
  indent_level++;

  /* Output local variable declarations.  */
  if (vars)
    {
      id <TLEnumerator> e = [vars valueEnumerator];

      while ((v = [e nextObject]))
#if STACK_REF_STRUCT
	if ([[v type] isObjectType])
	  {
	    if (!local_refs)
	      local_refs = [CO_TLVector vector];
	    [local_refs addElement: v];
	  }
	else
#endif
	  {
	    formac (of, @"%@", [self nl]);
	    [v outputDeclaration: of];
	    formac (of, @";");
	  }
    }

#if STACK_REF_STRUCT
  /* Output the stack protection.  */
  if ([container isKindOf: [CO_OTMMethod class]]
      && ([(OTMCustomMethod *) container referencingArguments]
	  || local_refs))
    {
      OTMCustomMethod *m = (OTMCustomMethod *) container;
      TLVector *args = [m referencingArguments];
      int num_locals = [local_refs length];
      int num_args = [args length];
      int i;

      [m setStackProtected];

      if (num_args)
	formac (of, @"void **%@[%d];\n", TO_PROTECT_STACK_ARGS, num_args);

      formac (of, @"struct {\nstruct trt_protect_stack _c;\n");

      for (i = 0; i < num_locals; i++)
	{
	  v = [local_refs _elementAtIndex: i];
	  [v outputDeclaration: of];
	  formac (of, @";\n");
	}

      formac (of, @"} %@;\n", TO_PROTECT_STACK);

      for (i = 0; i < num_args; i++)
	{
	  v = [args _elementAtIndex: i];
	  formac (of, @"%@[%d] = (void **) &%@;\n", TO_PROTECT_STACK_ARGS, i,
		  [v outputReference]);
	}

      formac (of, @"%@._c.next = %@;\n",
	      TO_PROTECT_STACK, TO_TRT_PROTECT_STACK);
      formac (of, @"%@._c.num_args = %d;\n", TO_PROTECT_STACK, num_args);
      formac (of, @"%@._c.num_locals = %d;\n", TO_PROTECT_STACK, num_locals);
      formac (of, @"%@._c.arguments = %@;\n", TO_PROTECT_STACK,
	      num_args ? TO_PROTECT_STACK_ARGS : @"0");

      formac (of, @"%@ = &%@._c;\n", TO_TRT_PROTECT_STACK, TO_PROTECT_STACK);

      if (num_locals)
	{
	  v = [local_refs _elementAtIndex: 0];
	  EFILLZERO ([v outputReference], num_locals);
	}

      formac (of, @"\n");
    }
#endif

  /* If we're a method's body, tell the method to output the
     initializations of the extension and other pointers.  */
  if ([container isKindOf: [CO_OTMMethod class]])
    [(OTMCustomMethod *) container compileExtensionPointerDefinitions];

  /* Compile the actual body.  */
  {
    int i, n = [stats length];

    [self compileCondition: 0];

    for (i = 0; i < n && !this_num_errors; i++)
      [[stats _elementAtIndex: i] compileStatement];

    [self compileCondition: 1];
  }

  /* Output the method epilogue.  */
  if ((id) container == output_current_method)
    {
      OTMExpr *e;

#if STACK_REF_STRUCT
      {
	BOOL protected = [output_current_method stackProtected];

	if (protected)
	  formac (of, @"%@%@ = %@._c.next; ", [self nl],
		  TO_TRT_PROTECT_STACK, TO_PROTECT_STACK);
      }
#endif

      e = [output_current_method returnValue];
      if (e)
	{
	  if ([e isTuple])
	    [output_current_method assignOutArgsFrom: (id) e];
	  formac (of, @"%@return %@;", [self nlPlain], [e result]);
	}
      else if ([output_current_method returnType] != basic_type[BT_VOID])
	error (@"return value never assigned in method returning non-void");
    }

  indent_level--;
  formac (of, @"%@}", [self nlPlain]);
}

-(void) compileAssignmentToTuple: (OTMTuple *) lhs
{
  if (!value)
    error_for (self, @"compound has no value");
  else
    [value compileAssignmentToTuple: lhs];
}

-(void) compileCondition: (BOOL) end_not_start
{
  /* Usefully implemented by subclasses.  */
}

-(OTMCompound *) container
{
  return container;
}

-(id) elaborate
{
  /* If the VALUE has been set, make sure it exists in the enclosing
     compound.  */
  if (value)
    {
      id previous_compound = current_compound;

      current_compound = self;
      [self assignResult: value];
      current_compound = previous_compound;
    }

  return self;
}

-(void) gcReference
{
  MARK (value);
  MARK (result);
  MARK (container);
  MARK (stats);
  MARK (vars);
  MARK (temp_free);
  MARK (temp_use);

  [super gcReference];
}

-initWithContainer: (id) c
{
  if (![super init])
    return nil;

  container = c;
  stats = [CO_TLVector vector];

  return self;
}

-(OTMLoop *) loopForContinue: (int) level
{
  return [container loopForContinue: level];
}

-(void) releaseTemporaryVariables
{
  if (temp_use)
    {
      if (!temp_free)
	temp_free = temp_use;
      else
	[temp_free addElementsFromSequence: temp_use];
      temp_use = nil;
    }
}

-(TLCons *) resolveWithExpected: (TLCons *) expected
		    convertible: (OTMType *) to
			context: (OTMType *) cxt
			indices: (int *) indices
			  index: (int) index
{
  if (!type)
    {
      if (!value)
	{
	  warning_for (self, @"compound without value");
	  value = void_expr;
	}
      type = [value type];
    }

  return [super resolveWithExpected: expected convertible: to
		context: cxt indices: indices index: index];
}

-(id <TLString>) result
{
  if (!result)
    ABORT ();

  return [result result];
}

-(OTMVariable *) searchVariableNamed: (id <TLString>) n
{
  if (vars)
    {
      id v = [vars objectForKey: n];
      if (v)
	return v;
    }
  return [container searchVariableNamed: n];
}

-(void) setBreakValue: (OTMExpr *) e
{
  [container setBreakValue: e];
}

-(void) setValue: (OTMExpr *) v
{
  ASGN_IVAR (value, v);
}

-(void) startLoopEnd
{
}

-(OTMVariable *) temporaryWithType: (OTMType *) t
{
  OTMVariable *v = nil;

  if (temp_free)
    {
      int i, n = [temp_free length];

      for (i = 0; i < n; i++)
	{
	  OTMVariable *o = [temp_free _elementAtIndex: i];

	  if ([o type] == t)
	    {
	      [temp_free _removeElementAtIndex: i];

	      v = o;
	      break;
	    }
	}
    }

  if (!v)
    {
      v = [CO_OTMLocalVar temporaryVariableWithType: t];
      emit_local_var (v);
    }

  if (!temp_use)
    temp_use = [CO_TLVector vector];

  [temp_use addElement: v];

  return v;
}

-(OTMExpr *) value
{
  return result;
}

@end
