/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMDynamicType.m,v 1.17 1998/03/16 14:09:05 tiggr Exp $  */

#import "OTMDynamicType.h"
#import "OTMInvocation.h"
#import "OTMMethod.h"

/* The single instance of the dynamic type.  */
OTMDynamicType *the_dynamic_type;

/* Dictionary from selector, involved with a dynamic type, to a dictionary
   from invocation to actual selector.  */
static TLDictionary *dynamic_selectors;

@implementation OTMDynamicType

+(void) init
{
  the_dynamic_type = [[self gcAlloc] init];
  [the_dynamic_type gcLock];
}

+(void) addInvocation: (OTMInvocation *) inv
    toDynamicSelector: (LTTSelector *) sel
{
  TLDictionary *d;

  if (dynamic_selectors)
    d = [dynamic_selectors objectForKey: sel];
  else
    {
      dynamic_selectors = [[TLDictionary gcAlloc] init];
      [dynamic_selectors gcLock];
      d = nil;
    }

  if (!d)
    {
      d = [TLDictionary dictionary];
      [dynamic_selectors setObject: d forKey: sel];
    }

  [d setObject: [CO_OTMMethod selectorForMethod: [inv method] invocation: inv
			      used: YES] forKey: inv];
}

+(void) dumpInfo: (id <TLOutputStream>) inf
{
  id <TLEnumerator> ke;
  LTTSelector *sel;

  if (!dynamic_selectors)
    return;

  formac (inf, @"(dynamic-selectors");
  ke = [dynamic_selectors enumerator];
  while ((sel = [ke nextObject]))
    {
      TLDictionary *d = [dynamic_selectors objectForKey: sel];
      id <TLEnumerator> se = [d valueEnumerator];
      LTTSelector *alias;

      formac (inf, @"\n (%#", [[sel lttName] internal]);

      while ((alias = [se nextObject]))
	formac (inf, @"\n  %#", [[alias lttName] internal]);

      formac (inf, @")");
    }

  formac (inf, @")\n");
}

+(LTTSelector *) selectorForInvocation: (OTMInvocation *) inv
		     ofDynamicSelector: (LTTSelector *) sel
{
  TLDictionary *d = [dynamic_selectors objectForKey: sel];
  LTTSelector *alias = [d objectForKey: inv];

  if (!alias)
    fatal (1, @"no actual selector for invocation %# of dynamic selector %#",
	   inv, [sel lttName]);

  return alias;
}

-(BOOL) allowedTypeForArgumentRedeclaration: (OTMType *) t
				 inSubclass: (BOOL) subclass_p
{
  return t == self;
}

-(id <TLString>) frobnicatedName
{
  return @"x";
}

-(BOOL) isFullyDefinedType
{
  return NO;
}

-(BOOL) matches: (OTMType *) t
{
  return t == self;
}

-(int) matchesConvertibly: (OTMType *) t
{
  return 0;
}

-(id <TLString>) typeName
{
  return @"dynamic";
}

-(id) outputTypeName
{
  return @"void";
}

-(id <TLString>) outputTypeEncoding
{
  return @"TRT_TE_DYNAMIC";
}

-(OTMType *) typeAt: (int) index in: (int *) indices
{
  return self;
}

@end
