/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMTuple.m,v 1.41 1998/03/27 13:57:04 tiggr Exp $  */

#define OTMTUPLE_DECLARE_PRIVATE_METHODS
#import "OTMTuple.h"
#import "OTMDynamicType.h"
#import "OTMTypeTuple.h"
#import "OTMVariable.h"
#import "OTMArgument.h"

@implementation OTMTuple

+(OTMTuple *) tuple
{
  return [[self gcAlloc] init];
}

+(OTMTuple *) tupleWithSequence: (id) s
{
  return [[self gcAlloc] initWithSequence: s];
}

-addElement: (id) o
{
  [elements addElement: o];
  return self;
}

-(OTMArgument *) argumentNumbered: (int) n
{
  int i, l = [elements length];

  for (i = 0; i < l; i++)
    {
      OTMArgument *a = [elements _elementAtIndex: i];

      a = [a argumentNumbered: n];
      if (a)
	return a;
    }

  return nil;
}

-(BOOL) checkProperArgument: (OTMMethod *) m
{
  BOOL r = NO;
  int mi, mn;

  for (mi = 0, mn = [elements length]; mi < mn; mi++)
    {
      OTMVariable *v = [elements _elementAtIndex: mi];
      r |= [v checkProperArgument: m];
    }

  return r;
}

-(void) compileAddressListWithFirst: (id <TLString>) f
			  separator: (id <TLString>) s
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    {
      [[elements _elementAtIndex: i] compileAddressListWithFirst: f
				     separator: s];
      f = s;
    }
}

-(void) compileAssignment: (OTMExpr *) rhs
{
  [rhs compileAssignmentToTuple: self];
}

-(void) compileAssignmentToTuple: (OTMTuple *) lhs
{
  TLVector *vl = [(OTMTuple *) lhs elements];
  int i, n = [vl length];

  for (i = 0; i < n; i++)
    {
      OTMExpr *l = [vl _elementAtIndex: i];
      [l compileAssignment: implicit_cast ([elements _elementAtIndex: i],
					   [l type])];
    }
}

-(void) declareReturnArguments: (int) n
{
  int i, l;

  for (i = 0, l = [elements length]; i < l; i++)
    [[elements _elementAtIndex: i] declareReturnArguments: n];
}

-(void) description: (id <TLMutableStream>) stream
{
  [super description: stream];
  formac (stream, @" %#", elements);
}

-(id) elaborate
{
  int i, n = [elements length];
  TLVector *ea = [CO_TLVector vectorWithCapacity: n];
  OTMTuple *ta;

  for (i = 0; i < n; i++)
    {
      OTMExpr *ex = [[elements _elementAtIndex: i] elaborate];

      if ([ex isTuple] || [ex constantp])
	[ea addElement: ex];
      else
	[ea addElement:
	      emit_assignment (temp_something_with_type ([ex type], 1), ex)];
      }

  ta = [CO_OTMTuple tupleWithSequence: ea];
  [ta setType: type];

  return ta;
}

-(TLVector *) elements
{
  return elements;
}

-(int) flatElementCount
{
  int c = 0, i, n = [elements length];

  for (i = 0; i < n; i++)
    c += [[elements _elementAtIndex: i] flatElementCount];

  return c;
}

-(void) gcReference
{
  MARK (elements);
  MARK (results);

  [super gcReference];
}

-(id) init
{
  [super init];

  elements = [CO_TLVector vector];

  return self;
}

-initWithSequence: (id) s
{
  if (![super init])
    return nil;

  elements = [CO_TLVector vectorWithSequence: s];

  return self;
}

-(BOOL) isTemporary
{
  return temporary;
}

-(BOOL) isTuple
{
  return YES;
}

-(OTMExpr *) lhsInvalid
{
  int mi, mn;

  for (mi = 0, mn = [elements length]; mi < mn; mi++)
    {
      OTMExpr *o = [elements _elementAtIndex: mi];
      id p = [o lhsInvalid];

      if (p)
	return p;
    }

  return nil;
}

-(id <TLString>) exprName
{
  return @"tuple";
}

-(id) outputDeclaration: (id) s
{
  return [self outputDeclaration: s separator: @","];
}

-(id) outputDeclaration: (id) s separator: (id <TLString>) sep
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    {
      if (i)
	s = formac (s, @"%@\n\t", sep);
      s = [[elements _elementAtIndex: i] outputDeclaration: s];
    }
  return s;
}

-(id) oldsEliminated
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    {
      id p = [elements _elementAtIndex: i];
      id o = [p oldsEliminated];

      if (o != p)
	[elements _replaceElementAtIndex: i by: o];
    }

  return self;
}

-(id) precompile
{
  int i, n = [elements length];

  for (i = 0; i < n; i++)
    {
      id p = [elements _elementAtIndex: i];
      id o = [p precompile];

      if (o != p)
	[elements _replaceElementAtIndex: i by: o];
    }
  return self;
}

-(TLCons *) resolveWithExpected: (TLCons *) expected
		    convertible: (OTMType *) to
			context: (OTMType *) cxt
			indices: (int *) indices
			  index: (int) index
{
  int  i, num_elts, our_index, is_dynamic;
  __typeof__ (indices) our_indices;
  TLVector *possible_elt_types;
  BOOL fully_typed, changes;

  if (type)
    return [super resolveWithExpected: expected convertible: to
		  context: cxt indices: indices index: index];

  our_index = index + 1;
  our_indices = alloca ((2 + index) * sizeof (*our_indices));
  memcpy (our_indices, indices, our_index * sizeof (*our_indices));

  num_elts = [elements length];
  possible_elt_types = [CO_TLVector vectorWith: num_elts copies: nil];

  is_dynamic = expected && [expected memq: the_dynamic_type];

  if (expected)
    {
      TLCons *list, *last, *t;
      TLCons *c, *next;
      OTMType *tp;

      /* First drop all tuples from EXPECTED that do not have a matching
	 number of arguments.  */
      for (t = expected, list = last = 0; t; t = next)
	{
	  DECONS (t, tp, next);
	  tp = [tp typeAt: index in: indices];
	  if ([tp elementCount] == num_elts)
	    {
	      c = CONS (tp, nil);
	      if (!list)
		list = last = c;
	      else
		{
		  [last setCdr: c];
		  last = c;
		}
	    }
	}
      expected = list;
    }

  do
    {
      fully_typed = YES;
      changes = NO;

      /* Resolve our elements.  */
      for (i = 0; i < num_elts; i++)
	{
	  OTMExpr *o = [elements _elementAtIndex: i];
	  TLCons *pt;

	  if (!o)
	    {
	      if (!expected)
		[possible_elt_types _replaceElementAtIndex: i
				    by: CONS (the_any_type, nil)];
	      continue;
	    }

	  if (expected && !is_dynamic)
	    {
	      our_indices[our_index] = i;
	      pt = [o resolveWithExpected: expected convertible: to
		      context: cxt indices: our_indices index: our_index];
	    }
	  else
	    pt = [o resolveWithExpected: NULL convertible: to
		    context: cxt indices: NULL index: -1];

	  if (!pt)
	    {
	      /* This element has no possible types.  Bummer.  */
	      return nil;
	    }

	  if (![o type])
	    fully_typed = NO;

	  if (expected && !is_dynamic)
	    {
	      if (i && !types_equal (pt, expected, expected))
		changes = YES;
	      expected = pt;
	      if (!is_dynamic)
		{
		  is_dynamic = (expected && ![expected cdr]
				&& [expected car] == the_dynamic_type);
		  if (is_dynamic)
		    changes = YES;
		}
	    }
	  else
	    {
	      OTMType *t = [o type];

	      /* Amend the list of possible types of this argument.  */
	      [possible_elt_types _replaceElementAtIndex: i
				  by: t ? CONS (t, nil) : pt];
	    }
	}
    } while (changes);

  if (fully_typed)
    {
      if (!expected || is_dynamic)
	type = [CO_OTMTypeTuple typeTupleWithSequence:
			     [CO_TLVector vectorByPerforming: @selector (car)
				     onElementsOfSequence: possible_elt_types]];
      else if ([expected cdr] || ![[expected car] isFullyDefinedType])
	{
	  ABORT ();
	}
      else
	type = [expected car];

      return [super resolveWithExpected: expected convertible: to
		    context: cxt indices: indices index: index];
    }

  error_for (self, @"unable to type tuple");

  return nil;
}

-(id <TLString>) result
{
  return [self result: 0];
}

-(id <TLString>) result: (int) index
{
  return (results ? [results _elementAtIndex: index]
	  : [[elements _elementAtIndex: index] result]);
}

-(void) setIsTemporary
{
  temporary = !0;
}

-(OTMExpr *) tupleSingleElement
{
  return [elements length] == 1 ? [elements _elementAtIndex: 0] : self;
}

@end
