/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: dload.c,v 1.71 1999/03/01 23:11:29 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>
#include <stdio.h>  /* For ABORT.  */

static int dload_initialized = 0;

/* Iff !0, we're dynamically loading.  Otherwise, we're doing startup
   resolving.  */
static int dynamic_loading;

/* Iff !0, thread-local data was added.  */
static int modified_locals;

/* The number of existing buckets.  */
int trt_num_buckets;

/* The number of buckets each dispatch table is supposed to have.  */
int num_buckets;

/* The method implementation hashtable.  This does not contain all methods.
   It is only used as a cache for metas_find_method().  */
struct hashtable method_table;

/* Selector table, mapping from selector name to selector.  */
static struct hashtable selector_table;

static struct trt_method_dispatch_table *empty_table;
static struct trt_method_dispatch_bucket *empty_bucket;
static struct trt_method_dispatch_table *install_table;
static struct trt_method_dispatch_bucket *install_bucket;

static GENERIC_RETURN_TYPE dtable_install (tom_object self, selector cmd, ...);

/* The collection of known of units.  */
/* XXX This is not yet set when resolving statically.
   Tue Apr  1 01:33:57 1997, tiggr@tricky.es.ele.tue.nl  */
struct trt_units *trt_units;

/* The collection of known modules.  */
/* XXX This is not yet set when resolving statically.
   Tue Apr  8 22:07:14 1997, tiggr@tricky.es.ele.tue.nl  */
struct trt_modules *trt_modules;


static __inline__ void
retain_bucket (struct trt_method_dispatch_bucket **b,
	       struct trt_method_dispatch_bucket *new)
{
  new->ref_count++;
  *b = new;
}

static __inline__ void
release_bucket (struct trt_method_dispatch_bucket **b)
{
  if (!*b || !(*b)->ref_count || (*b)->static_p)
    ABORT ();
  else
    {
      if (!--(*b)->ref_count)
	{
#ifdef DEBUG
	  BZERO (*b, sizeof (**b));
#endif
	  xfree (*b);
	}
      *b = 0;
    }
}

struct trt_method_dispatch_table *
dtable_resize (struct trt_method_dispatch_table *dt,
	       struct trt_method_dispatch_bucket *bucket)
{
  struct trt_method_dispatch_table *pdt = dt;
  int i, n = dt ? dt->st.num : 0;

  stalloc ((void *) &dt, sizeof (*dt), sizeof (dt->b[0]), num_buckets, 1);

  /* If the address of the table was changed, patch the owner of the
     buckets.  */
  if (pdt && pdt != dt)
    for (i = 0; i < n; i++)
      if (dt->b[i]->owner == pdt)
	dt->b[i]->owner = dt;

  /* Fill the extra slots created with the BUCKET.  */
  if (n != dt->st.num)
    for (i = n; i < num_buckets; i++)
      retain_bucket (&dt->b[i], bucket);

  return dt;
}

/* Free the dispatch table DT.  */
static void
dtable_free (struct trt_method_dispatch_table *dt)
{
  if (dt->st.cap)
    {
      int i;

      for (i = 0; i < dt->st.num; i++)
	if (!dt->b[i]->static_p)
	  release_bucket (&dt->b[i]);
      xfree (dt);
    }
}

/* Register the selectors from SELS with the runtime.  */
static void
trt_include_selectors (struct trtd_selectors *sels)
{
  struct trt_method_dispatch_table *et, *it;
  int previous_num_buckets = num_buckets;
  struct trt_class *m;
  int i, j;

  if (!selector_table.equal)
    {
      selector_table.equal = hash_equal_string;
      selector_table.hash = hash_hash_string;
    }

  if (!dynamic_loading)
    {
      /* This is the easy case: we're invoked during startup to resolve the
         whole application.  This means we can directly use all the
         selectors.  */
      for (i = 1; i < sels->st.num; i++)
	{
	  hash_add (&selector_table, sels->selectors[i]->name.s,
		    sels->selectors[i]);
	  sels->selectors[i]->sel_id = i;
	}
      trt_selectors = sels;
      num_buckets = NUM_BUCKETS;
    }
  else if (sels->st.num)
    {
      /* Handle the (possibly new) selectors.  */
      for (i = 0; i < sels->st.num; i++)
	{
	  selector sel;

	  sel = hash_find (&selector_table, sels->selectors[i]->name.s);
	  if (sel)
	    sels->selectors[i]->sel_id = sel->sel_id;
	  else
	    {
	      hash_add (&selector_table, sels->selectors[i]->name.s,
			sels->selectors[i]);
	      sels->selectors[i]->sel_id = trt_selectors->st.num;
	      stalloc ((void *) &trt_selectors, sizeof (*trt_selectors),
		       sizeof (trt_selectors->selectors[0]),
		       trt_selectors->st.num + 1, 0);
	      trt_selectors->selectors[sels->selectors[i]->sel_id]
		= sels->selectors[i];
	    }
	}

      num_buckets = NUM_BUCKETS;
    }

  et = dtable_resize (empty_table, empty_bucket);
  it = dtable_resize (install_table, install_bucket);
  if (et != empty_table || it != install_table)
    {
      if (trt_metas)
	/* Patch all installed references to these tables.  */
	for (i = 0; i < trt_metas->st.num; i++)
	  for (j = 0, m = trt_metas->metas[i]; j < 2; j++, m = m->isa)
	    if (m->info.mdt == empty_table)
	      m->info.mdt = et;
	    else if (m->info.mdt == install_table)
	      m->info.mdt = it;

      /* These are missed since they have not a single extension.  */
      _md_m__builtin__Top.c_tom_State.info.mdt = et;
      _md_c__builtin__Top.c_tom_State.info.mdt = et;
      _md_m__builtin__Any.c_tom_State.info.mdt = et;
      _md_c__builtin__Any.c_tom_State.info.mdt = et;

      empty_table = et;
      install_table = it;
    }

  /* Resize remaining existing dispatch tables.  */
  if (num_buckets != previous_num_buckets && trt_metas)
    for (i = 0; i < trt_metas->st.num; i++)
      for (j = 0, m = trt_metas->metas[i]; j < 2; j++, m = m->isa)
	m->info.mdt = dtable_resize (m->info.mdt, empty_bucket);
}

static void
trt_include_dynamic_selectors (struct trtd_selector_map *ds, int num_ds)
{
  int init_n = trt_dynamic_selectors ? trt_dynamic_selectors->st.num : 0;
  int i, j, n = init_n;

  for (i = 0; i < num_ds; i++)
    {
      for (j = 0; j < init_n; j++)
	if (trt_selectors_equal (trt_dynamic_selectors->maps[j]->actual,
				 ds[i].actual))
	  break;
      if (j == init_n)
	{
	  stalloc ((void *) &trt_dynamic_selectors,
		   sizeof (*trt_dynamic_selectors),
		   sizeof (trt_dynamic_selectors->maps[0]), n + 1, num_ds);
	  trt_dynamic_selectors->maps[n++] = &ds[i];
	}
    }
}

/* Given the method implementation IMP, return the definition of the method.
   This searches in from the meta M towards the supermost class.  */
static struct trtd_method *
metas_find_method (struct trt_class *m, int_imp imp, int k)
{
  int i, j;

  if (m->info.mark == k)
    return 0;
  m->info.mark = k;

  if (m->info.extensions)
    for (i = 0; i < m->info.extensions->st.num; i++)
      if (m->info.extensions->extensions[i]->methods)
	{
	  struct trtd_methods *ms = m->info.extensions->extensions[i]->methods;

	  for (j = 0; j < ms->num_methods; j++)
	    if (ms->methods[j].imp == imp)
	      return &ms->methods[j];
	}

  if (m->info.supers)
    for (i = 0; i < m->info.supers->st.num; i++)
      {
	struct trtd_method *found;

	found = metas_find_method (m->info.supers->metas[i], imp, k);
	if (found)
	  return found;
      }

  return 0;
}

static struct trtd_method *
find_method_info (struct trt_class *m, int_imp imp)
{
  struct trtd_method *mt = hash_find (&method_table, imp);

  if (!mt)
    {
      mt = metas_find_method (m, imp, ++trt_search_mark);
      if (!mt)
	fatal ("probable dtable_conflict for `%s'", m->info.name.s);

      hash_add (&method_table, imp, mt);
    }

  return mt;
}

/* Add the meta M to the metas pointed to through PMETAS.  Return 0 if the
   addition was unnecessary, or 1 if M was added.  */
static int
metas_add (struct trt_metas **pmetas, struct trt_class *m)
{
  struct trt_metas *metas;
  int i, n = *pmetas ? (*pmetas)->st.num : 0;

  if (n)
    for (i = 0; i < n; i++)
      if ((*pmetas)->metas[i] == m)
	return 0;

  metas = stalloc ((void *) pmetas, sizeof (*metas),
		   sizeof (metas->metas[0]), n + 1, 4);
  metas->metas[n] = m;

  return 1;
}

static void
metas_install_default_mdt (struct trt_class *m, int k)
{
  if (m->info.mdt != install_table)
    {
      int i;

      if (m->info.mark == k)
	return;
      m->info.mark = k;

      /* If this is _builtin_.Top, the dispatch table will be empty and
         remain empty.  Install the INSTALL_TABLE just to be sure.  */
      if (m->info.mdt && m->info.mdt != empty_table)
	dtable_free (m->info.mdt);
      m->info.mdt = install_table;

      /* Propagate to our subclasses.  */
      if (m->info.subs)
	for (i = 0; i < m->info.subs->st.num; i++)
	  metas_install_default_mdt (m->info.subs->metas[i], k);
    }
}

/* Mark the receiving meta M and its supers.  */
static void
meta_mark_supers (struct trt_class *m, int k)
{
  int i;

  if (m->info.mark == k)
    return;
  m->info.mark = k;

  if (m->info.supers)
    for (i = 0; i < m->info.supers->st.num; i++)
      meta_mark_supers (m->info.supers->metas[i], k);
}

/* Adjust the extension offset table and refvar offset table of the meta M
   to incorporate the state of the extension X.  */
static void
meta_add_extension_state (struct trt_class *m, struct trtd_extension *x)
{
  struct trt_extension_offset_table *eot;
  int i, n, x_offset, base;

  /* XXX This is for debugging some weird behaviour of tesla code.  */
  if (!m->info.instance_size
      && !(x->eid_in_a_global == &_ei_i_tom_State
	   || x->eid_in_a_global == &_ei_c_tom_State))
    ABORT ();
  if (m->info.instance_size
      && (x->eid_in_a_global == &_ei_i_tom_State
	  || x->eid_in_a_global == &_ei_c_tom_State))
    ABORT ();

  /* XXX This is a regular check, but catching errors rather late.  */
  if (!x->num_vars
      || (m->info.eot && m->info.eot->st.num > *x->eid_in_a_global
	  && m->info.eot->offset[*x->eid_in_a_global]))
    ABORT ();

  if (TGC_CLASS_P (m->asi) && m->info.num_instances)
    fatal ("attempt to add state to class %s with %d allocated instances",
	   m->info.name.s, m->info.num_instances);

  /* Adjust the extension offset table.  */
  eot = stalloc ((void *) &m->info.eot, sizeof (*m->info.eot),
		 sizeof (eot->offset[0]), trt_num_eids, 1);

  base = ((m->info.instance_size + x->state_align / 8 - 1)
	  & ~(x->state_align / 8 - 1));
  if (base < m->info.instance_size)
    ABORT ();

  eot->offset[*x->eid_in_a_global] = x_offset = base;
  m->info.instance_size = base + x->state_size;
  m->info.state_extensions = NULL;

  /* Adjust the reference variables offset table.  */
  for (i = 0, n = m->info.rvo ? m->info.rvo->st.num : 0; i < x->num_vars; i++)
    if (x->vars[i].c.type == TRT_TE_REFERENCE)
      {
	stalloc ((void *) &m->info.rvo, sizeof (*m->info.rvo),
		 sizeof (m->info.rvo->offset[0]), n + 1, 1);
	m->info.rvo->offset[n++] = x_offset + x->vars[i].offset;
      }
}

/* Invoke meta_add_extension_state to the meta M and all its subclasses.  */
static void
metas_add_extension_state (struct trt_class *m, struct trtd_extension *x, int k)
{
  int i;

  if (m->info.mark2 == k)
    return;
  m->info.mark2 = k;

  meta_add_extension_state (m, x);

  /* Propagate to our subclasses.  */
  if (m->info.subs)
    for (i = 0; i < m->info.subs->st.num; i++)
      metas_add_extension_state (m->info.subs->metas[i], x, k);
}

/* XXX This does not update the INITIAL_ASI.  */
/* Add the state introduced by the class M and its superclasses to the newly
   added subclass TO and its subclasses.  */
static void
meta_inherit_state (struct trt_class *to, struct trt_class *m, int k)
{
  int i;

  if (m->info.mark == k)
    return;
  m->info.mark = k;

  /* Have our superclasses do this first (since State must end up as the
     first extension).  */
  if (m->info.supers)
    for (i = 0; i < m->info.supers->st.num; i++)
      meta_inherit_state (to, m->info.supers->metas[i], k);

  if (m->info.extensions)
    for (i = 0; i < m->info.extensions->st.num; i++)
      {
	struct trtd_extension *x = m->info.extensions->extensions[i];

	if (!x->state_size)
	  continue;

	/* The State main extension _must_ _always_ be the first.  */
	if (!to->info.instance_size
	    && !(x->eid_in_a_global == &_ei_i_tom_State
		 || x->eid_in_a_global == &_ei_c_tom_State))
	  ABORT ();

	metas_add_extension_state (to, x, ++trt_search_mark);
      }
}

/* Allocate space for any thread-local static variables introduced by the
   extension X.  */
static void
ext_note_locals (struct trtd_extension *x)
{
  if (x->num_statics)
    {
      int i;

      for (i = 0; i < x->num_statics; i++)
	if (x->statics[i].th_local)
	  {
	    int size;

	    if (trt_thread_num > 1)
	      unimplemented ("adding thread-locals while having >1 thread");

	    switch (x->statics[i].c.type)
	      {
	      case TRT_TE_BOOLEAN:
	      case TRT_TE_BYTE:
		size = sizeof (tom_byte);
		break;

	      case TRT_TE_CHAR:
		size = sizeof (tom_char);

	      case TRT_TE_INT:
		size = sizeof (tom_int);
		break;

	      case TRT_TE_LONG:
		size = sizeof (tom_long);
		break;

	      case TRT_TE_FLOAT:
		size = sizeof (tom_float);
		break;

	      case TRT_TE_DOUBLE:
		size = sizeof (tom_double);
		break;

	      case TRT_TE_POINTER:
	      case TRT_TE_SELECTOR:
	      case TRT_TE_REFERENCE:
		size = sizeof (void *);
		break;

	      default:
		ABORT ();
	      }

	    trt_thread_local_size = ((trt_thread_local_size + size - 1)
				     & ~(size - 1));
	    *(int *) x->statics[i].address = trt_thread_local_size;
	    trt_thread_local_size += size;
	    modified_locals = 1;
	  }
    }
}

/* Add any new superclasses introduced by the extension X to the meta M.  */
static void
meta_add_extension_classes (struct trtd_extension *x, struct trt_class *m)
{
  int i;

  /* Reshape the inheritance graph, inheriting state.  */
  if (x->supers)
    for (i = 0; i < x->supers->st.num; i++)
      if (metas_add (&x->supers->metas[i]->info.subs, m))
	{
	  /* Inherit the world from the newly added superclass.  */
	  meta_mark_supers (m, ++trt_search_mark);
	  meta_inherit_state (m, x->supers->metas[i], trt_search_mark);
	  metas_add (&m->info.supers, x->supers->metas[i]);

	  if (dynamic_loading)
	    {
	      /* Invalidate dispatch tables in ourselves and our subclasses.  */
	      metas_install_default_mdt (m, ++trt_search_mark);
	    }
	}

  if (x->state_size)
    {
      *x->eid_in_a_global = trt_num_eids++;
      metas_add_extension_state (m, x, ++trt_search_mark);
    }

  /* Reset the dispatch table.  */
  if (x->methods || !dynamic_loading)
    metas_install_default_mdt (m, ++trt_search_mark);

  /* Add the extension to its meta.  (Used by the dtable building.)  */
  i = m->info.extensions ? m->info.extensions->st.num : 0;
  stalloc ((void *) &m->info.extensions, sizeof (*m->info.extensions),
	   sizeof (m->info.extensions->extensions[0]), i + 1, 1);
  m->info.extensions->extensions[i] = x;
}

/* Make the meta POSER pose as the POSED meta.  The POSER must already be
   a subclass of the POSED.

   This adjusts the meta reference for the POSED to point to the POSER.
   Furthermore, all subclasses of the POSED have their super reference to
   the POSED replaced by the POSER, any extensions introduced by the POSER
   (either directly or through inheritance) are added to the subclasses,
   similar to what would happen if the POSED had a new extension added.

   If the POSED is already posed, this simply adjusts the meta reference,
   and the old poser becomes the POSED.  */
static void
meta_pose (struct trt_class *poser, struct trt_class *posed)
{
  int i, j, found_self = -1;

  if (posed->info.poser)
    unimplemented ("posing of an already posed class");
  if (poser->info.poser)
    unimplemented ("posing by a posed class");

  /* Add any new state introduced by the POSER, or any of the superclasses
     it introduces, to the subclasses of the POSED.  Also, replace the
     POSED by the POSER in the SUPERS administration of said subclasses.  */
  for (i = 0; i < posed->info.subs->st.num; i++)
    {
      struct trt_class *m = posed->info.subs->metas[i];

      if (m == poser)
	found_self = i;
      else
	{
	  meta_mark_supers (m, ++trt_search_mark);

	  if (poser->info.mark == trt_search_mark)
	    {
	      /* The poser already is a superclass of M.    */
	      unimplemented ("subclass of POSED already has POSER as a super");
	    }
	  else
	    {
	      meta_inherit_state (m, poser, trt_search_mark);

	      for (j = 0; j < m->info.supers->st.num; j++)
		if (m->info.supers->metas[j] == posed)
		  {
		    m->info.supers->metas[j] = poser;
		    break;
		  }

	      if (j == m->info.supers->st.num)
		ABORT ();
	    }
	}
    }

  if (found_self < 0)
    ABORT ();

  metas_install_default_mdt (posed, ++trt_search_mark);

  /* Update the super references to reference the POSER where they
     previously referenced the POSED.  */
  /* XXX Maybe these references should be administered with the
     referencing class...
     Tue Apr  8 22:12:07 1997, tiggr@tricky.es.ele.tue.nl  */
  for (i = 0; i < trt_modules->st.num; i++)
    for (j = 0; j < trt_modules->modules[i]->num_super_refs; j++)
      {
	struct trtd_super *sr = &trt_modules->modules[i]->super_refs[j];

	if (*sr->super == posed && *sr->sub != poser)
	  *sr->ref = poser;
      }

  *posed->info.class_reference = poser;
}

static struct trt_method_dispatch_table *
dtable_copy (struct trt_method_dispatch_table *ot)
{
  struct trt_method_dispatch_table *dt = 0;
  int i;

  stalloc ((void *) &dt, sizeof (*dt), sizeof (dt->b[0]), ot->st.num, 1);

  for (i = 0; i < dt->st.num; i++)
    retain_bucket (&dt->b[i], ot->b[i]);

  return dt;
}

/* Put the method IMP for the selector with SEL_ID into the dispatch table
   DT, doing copy-on-write and all that.  */
static __inline__ void
dtable_put (struct trt_method_dispatch_table *dt,
	    int sel_id, int_imp imp)
{
  struct trt_method_dispatch_bucket *b;
  int bidx, midx;

  if (sel_id >= TRT_BUCKET_SIZE * dt->st.num)
    ABORT ();

  bidx = sel_id / TRT_BUCKET_SIZE;
  midx = sel_id % TRT_BUCKET_SIZE;

  b = dt->b[bidx];
  if (b->m[midx] != imp)
    {
      if (b->owner != dt)
	{
	  /* Copy on write.  */
	  struct trt_method_dispatch_bucket *new = xmalloc (sizeof (*new));

	  memcpy (&new->m[0], &b->m[0], sizeof (new->m));
	  new->static_p = new->ref_count = 0;
	  new->owner = dt;

	  release_bucket (&dt->b[bidx]);
	  retain_bucket (&dt->b[bidx], new);
	  b = dt->b[bidx];

	  trt_num_buckets++;
	}

      b->m[midx] = imp;
    }
}

/* This is the method implementation for selectors with conflicting method
   implementations.  */
static void
dtable_conflict (tom_object self, selector cmd)
{
  /* Invoking fatal() is safer in case Condition's raise method is
     affected, but in general, Conditions are to be preferred.  */
  trt_raise (0, self, cmd, c_tom_Conditions_program_condition,
	     "method implementation inheritance conflict");
}

/* This is the method implementation of deferred methods.  Pointers to
   this method are only created by tomr in static resolution mode.  */
static void
dtable_deferred (tom_object self, selector cmd)
{
  fatal ("invocation of deferred method for selector %s of %s %s",
	 cmd->name.s, TGC_CLASS_P (self->isa->asi) ? "instance" : "class",
	 self->isa->info.name.s);
}

/* Recursing upwards in the hierarchy, install the dispatch table for the
   class M, using K as the search mark.  */
static void
metas_install_dtable (struct trt_class *m, int k)
{
  struct trt_class **inherited;
  int i, num_inherited = 0;

  if (m->info.mark == k || m->info.mdt != install_table)
    return;
  m->info.mark = k;

  /* First let our superclasses have a go at it.  */
  if (m->info.supers)
    for (i = 0; i < m->info.supers->st.num; i++)
      metas_install_dtable (m->info.supers->metas[i], k);
  else if (!m->info.extensions)
    {
      m->info.mdt = empty_table;
      return;
    }

  if (!m->info.supers)
    {
      /* Somebody did something against the rules, and did not inherit
         from instance (All).  No punishment.  */
      inherited = NULL;
    }
  else
    {
      inherited = alloca (m->info.supers->st.num * sizeof (*inherited));
      for (i = 0; i < m->info.supers->st.num; i++)
	if (!(m->info.supers->metas[i]->info.mdt == empty_table
	      || !m->info.supers->metas[i]->info.mdt
	      || m->info.supers->metas[i] == m))
	  inherited[num_inherited++] = m->info.supers->metas[i];
    }

  /* Build a dispatch table of the inherited functionality.  */
  if (num_inherited == 1)
    {
      /* This is an easy case: inheritance from a single superclass
         (a.k.a. single inheritance).  */
      m->info.mdt = dtable_copy (inherited[0]->info.mdt);
    }
  else if (num_inherited == 0)
    {
      /* No inheritance: start with an empty table.  XXX Actually, the copy
         should not be done since this case probably indicates an unused
         instance or a deferred instance (which should not be allocatable
         anyway) ...  */
      m->info.mdt = dtable_copy (empty_table);
    }
  else
    {
      /* General case: multiple inheritance.  */
      struct trt_method_dispatch_table *dt = 0;
      struct trt_method_dispatch_bucket *b;
      int super, bucket, method, inherited_super;

      m->info.mdt = stalloc ((void *) &dt, sizeof (*dt), sizeof (dt->b[0]),
			     empty_table->st.num, 1);

      for (bucket = 0; bucket < num_buckets; bucket++)
	{
	  /* First see if we can inherit any of the super buckets straight
             away without modification.  */
	  for (super = 0, b = 0; super < num_inherited; super++)
	    if (inherited[super]->info.mdt->b[bucket] != empty_bucket)
	      if (!b)
		b = inherited[super]->info.mdt->b[bucket];
	      else
		break;

	  if (super == num_inherited)
	    {
	      retain_bucket (&dt->b[bucket], b ? b : empty_bucket);
	      continue;
	    }

	  /* The fast way didn't work.  Do it the slow way.  */
	  inherited_super = -1;
	  for (method = 0; method < TRT_BUCKET_SIZE; method++)
	    {
	      int_imp imp = 0;
	      int imp_super;

	      /* GGG NeXTstation NS3.3 with GNU CC 2.7.2
		 If this extra compound here is removed, the invocation of
		 dtable_put() afterwards is drawn into the loop!
		 Fri Jun 28 22:38:52 1996, tiggr@tricky.es.ele.tue.nl  */
	      {
		for (super = imp_super = 0; super < num_inherited; super++)
		  {
		    int_imp this;

		    this = inherited[super]->info.mdt->b[bucket]->m[method];

		    if (!this || this == (void *) dtable_install)
		      ABORT ();

		    if (this != (void *) trt_forward)
		      {
			if (!imp
			    || imp == (void *) dtable_deferred
			    /* XXX This is incorrect...  */
			    || imp == (void *) dtable_conflict)
			  {
			    imp_super = super;
			    imp = this;
			  }
			else if (imp != this
				 && this != (void *) dtable_deferred
				 /* XXX This is a hack!
				    v_load_r has 0 as sel_id  */
				 && !(this == (void *) dtable_conflict
				      && !bucket && !method))
			  {
			    /* This is inheritance of a different
                               implementation of the same method.  Let the
                               most specialized class (if they're related)
                               win.  XXX These semantics are questionable.  */
			    struct trtd_method *d_this, *d_imp;

			    d_this = find_method_info (inherited[super], this);
			    d_imp = find_method_info (m, imp);

			    if (trt_supermeta_star (d_this->meta, d_imp->meta,
						    ++trt_search_mark))
			      {
				imp_super = super;
				imp = this;
			      }
			    else if (!trt_supermeta_star (d_imp->meta,
							  d_this->meta,
							  ++trt_search_mark))
			      break;
			  }
		      }
		  }
	      }

	      imp = (super != num_inherited ? (int_imp) dtable_conflict
		     : imp ? imp : (int_imp) trt_forward);

	      if (imp != (void *) trt_forward)
		{
		  if (!dt->b[bucket])
		    {
		      if (inherited_super == -1)
			inherited_super = imp_super;
		      if (inherited[inherited_super]
			  ->info.mdt->b[bucket]->m[method] != imp)
			retain_bucket (&dt->b[bucket],
				       (inherited[inherited_super]
					->info.mdt->b[bucket]));
		    }

		  if (dt->b[bucket])
		    dtable_put (dt, method + bucket * TRT_BUCKET_SIZE, imp);
		}
	    }

	  if (!dt->b[bucket])
	    retain_bucket (&dt->b[bucket],
			   (inherited_super != -1
			    ? inherited[inherited_super]->info.mdt->b[bucket]
			    : empty_bucket));
	}
    }

  /* Override methods with our own definitions.  */
  if (m->info.extensions)
    for (i = 0; i < m->info.extensions->st.num; i++)
      {
	struct trtd_extension *x = m->info.extensions->extensions[i];
	int j;

	if (x->methods)
	  for (j = 0; j < x->methods->num_methods; j++)
	    dtable_put (m->info.mdt, x->methods->methods[j].sel->sel_id,
			x->methods->methods[j].imp ?: (void *) dtable_deferred);
      }

  /* Fix the dynamic selectors.  */
  if (trt_dynamic_selectors)
    {
      struct trtd_selector_map **maps = trt_dynamic_selectors->maps;

      for (i = 0; i < trt_dynamic_selectors->st.num; i++)
	{
	  void *act_imp = trt_lookup_super (m, maps[i]->actual);

	  if (act_imp == trt_forward || act_imp == dtable_deferred)
	    dtable_put (m->info.mdt, maps[i]->actual->sel_id,
			trt_lookup_super (m, maps[i]->dynamic));
	}
    }
}

static GENERIC_RETURN_TYPE
dtable_install (tom_object self, selector cmd, ...)
{
  void *args, *result;
  int_imp imp;
  builtin_return_type local_result;

  LOCK_RUNTIME ();

  metas_install_dtable (self->isa, ++trt_search_mark);

  UNLOCK_RUNTIME ();

  imp = trt_lookup (self, cmd);
  args = APPLY_ARGS_ARGS ();
  result = APPLY_ARGS_APPLY ((void_imp) imp, args, 
			     trt_all_arguments_size (cmd), &local_result);
  APPLY_ARGS_RETURN (result);
}

static void
trt_resolve_unit (struct trtd_unit *u)
{
  int i, n;

  for (i = 0; i < u->num_extensions; i++)
    {
      /* Add newly inherited superclasses to this extension's meta.  */
      meta_add_extension_classes (u->extensions[i], u->extensions[i]->meta);

      /* Handle any thread-local variables.  */
      ext_note_locals (u->extensions[i]);
    }

  if (modified_locals)
    {
      modified_locals = 0;
      if (trt_started_main && dynamic_loading)
	trt_thread_update_locals ();
      else
	{
	  trt_thread_local_cap = trt_thread_local_size;
	  trt_init_thread_data = xrealloc (trt_init_thread_data,
					   trt_thread_local_cap);
	  memset (trt_init_thread_data, 0, trt_thread_local_cap);
	}
    }

  /* Remove the metas collection, as we won't be able to safely maintain
     it during resolution.  */
  trt_all_classes = 0;
  c_tom_Runtime_classes_by_name = 0;

  /* Add the classes from this unit to the recorded metas.  */
  n = trt_metas ? trt_metas->st.num : 0;
  stalloc ((void *) &trt_metas, sizeof (*trt_metas),
	   sizeof (trt_metas->metas[0]), n + u->num_classes, 0);
  for (i = 0; i < u->num_classes; i++)
    trt_metas->metas[n++] = u->classes[i];

  /* Record this unit.  */
  /* XXX Check for duplicate units with the same name.  Maybe even check
     this upon entry to this function, not this late.
     Tue Apr  1 01:36:34 1997, tiggr@tricky.es.ele.tue.nl  */
  n = trt_units ? trt_units->st.num : 0;
  stalloc ((void *) &trt_units, sizeof (*trt_units),
	   sizeof (trt_units->units[0]), n + 1, 0);
  trt_units->units[n] = u;

  /* Update the units collection, if it exists.
     XXX It will be rebuilt when needed...  */
  c_tom_Unit_units = 0;
}

void
trt_resolve_module (struct trtd_module *mi)
{
  int i, n;

  /* Interaction between functioning contructors and Bundle's load method
     can cause multiple invocations of trt_result_module.  Hence, protect
     instead of abort.  */
  /* XXX mis-using the NUM_UNITS field is a hack.  */
  if (!mi->num_units)
    return;

  TRT_PANIC_MODE_P ();

  if (!trt_started_main)
    trt_module_constructors = 1;
  dynamic_loading = !!trt_metas;

  if (!empty_bucket)
    {
      int i;

      empty_bucket = xmalloc (sizeof (*empty_bucket));
      empty_bucket->static_p = empty_bucket->ref_count = 0;

      install_bucket = xmalloc (sizeof (*empty_bucket));
      install_bucket->static_p = install_bucket->ref_count = 0;

      trt_num_buckets += 2;

      for (i = 0; i < TRT_BUCKET_SIZE; i++)
	{
	  empty_bucket->m[i] = (int_imp) trt_forward;
	  install_bucket->m[i] = (int_imp) dtable_install;
	}

      empty_table = dtable_resize (0, empty_bucket);
      empty_bucket->owner = empty_table;
      install_table = dtable_resize (0, install_bucket);
      install_bucket->owner = install_table;

      method_table.equal = hash_equal_pointer;
      method_table.hash = hash_hash_pointer;
    }

  trt_include_selectors (mi->selectors);
  trt_include_dynamic_selectors (mi->dynamic_selectors,
				 mi->num_dynamic_selectors);

  {
    int num_units = mi->num_units;

    mi->num_units = 0;

    for (i = 0; i < num_units; i++)
      trt_resolve_unit (mi->units[i]);
  }

  for (i = 0; i < mi->num_super_refs; i++)
    *mi->super_refs[i].ref = *mi->super_refs[i].super;

  if (!dynamic_loading)
    {
      /* These are missed since they have not a single extension.  */
      _md_m__builtin__Top.c_tom_State.info.mdt = empty_table;
      _md_c__builtin__Top.c_tom_State.info.mdt = empty_table;
      _md_m__builtin__Any.c_tom_State.info.mdt = empty_table;
      _md_c__builtin__Any.c_tom_State.info.mdt = empty_table;
    }
  else if (mi->num_dynamic_selectors)
    {
      /* (Expensively) reset _all_ dispatch tables to include the new
         dynamic selectors.
	 XXX This could of course be done lasily, from within trt_forward.
	 Tue Mar 25 11:13:27 1997, tiggr@akebono.ics.ele.tue.nl.  */
      metas_install_default_mdt ((void *) &_md_m__builtin__Top,
				 ++trt_search_mark);
      metas_install_default_mdt ((void *) &_md_c__builtin__Top,
				 ++trt_search_mark);
    }

  /* Record this module.  */
  n = trt_modules ? trt_modules->st.num : 0;
  stalloc ((void *) &trt_modules, sizeof (*trt_modules),
	   sizeof (trt_modules->modules[0]), n + 1, 0);
  trt_modules->modules[n] = mi;

  if (mi->num_posers)
    for (i = 0; i < mi->num_posers; i++)
      meta_pose (mi->posers[i].poser, mi->posers[i].posed);

  /* See if this is the module containing the main method.  Ignore any
     main method if loading after we've already started.  */
  if (mi->main_class && !trt_started_main)
    {
      if (trt_main_class)
	ABORT ();
      trt_main_class = mi->main_class;
      trt_main_selector = mi->main_selector;
    }

  TRT_PANIC_MODE_V ();
}

void
trt_execute_all_load_imps (void *args)
{
  int i, n = trt_modules->st.num;

  /* The order is correct.  */
  for (i = 0; i < n; i++)
    trt_execute_load_imps (trt_modules->modules[i], args);
}

/* Execute the implementations of the `void load Array args' methods
   introduced by the module MI.  */
void
trt_execute_load_imps (struct trtd_module *mi, void *args)
{
  int i;

  for (i = 0; i < mi->num_load_imps; i++)
    mi->load_imps[i].imp (*mi->load_imps[i].class, SEL (v_load_r), args);
}

void *
c_tom_Bundle_p_load_r_unit_r (tom_object self, selector cmd,
			      tom_object object_name, tom_object unit_name)
{
  void *handle;
  tom_int len;
  char *s;

  if (!dload_initialized)
    {
      dload_initialized = 1;
      dload_initialize ();
    }

  C_STRING_WITH_TOM_STRING (s, len, object_name);
  handle = dload_load (s);

  if (!handle)
    TRT_SIGNAL (self, c_tom_Conditions_error,
		byte_string_with_c_string ("bundle load failed"));
  else
    {
      struct trtd_module *mi = 0;

      if (unit_name)
	{
	  char *name;

	  C_STRING_WITH_TOM_STRING (s, len, unit_name);

	  name = alloca (sizeof (DLOAD_SYMBOL_PREFIX TRT_MODULE_INFO)
			 + 1 + strlen (s));
	  sprintf (name, "%s_%s", DLOAD_SYMBOL_PREFIX TRT_MODULE_INFO, s);

	  {
	    /* Convert name to safe name.
	       XXX Safe name should be part of the unit description.  */
	    char *s;
	    for (s = name; *s; s++)
	      if (strchr ("-", *s))
		*s = '_';
	  }

	  mi = dload_lookup (handle, name);
	}
      if (!mi)
	mi = dload_lookup (handle, DLOAD_SYMBOL_PREFIX TRT_MODULE_INFO);

      if (mi)
	{
	  trt_resolve_module (mi);
	  trt_execute_load_imps (mi, 0);
	}
    }

  return handle;
}

void
c_tom_Unit_v_fillUnits (tom_object self, selector cmd)
{
  int i, j;

  c_tom_Unit_units = TRT_SEND (_PI_, CREF (tom_MutableDictionary), SEL (r_new));

  for (i = 0; i < trt_units->st.num; i++)
    {
      struct trtd_unit *u = trt_units->units[i];
      if (strcmp (u->name.s, "_builtin_"))
	{
	  tom_object u_obj, un, c_dict
	    = TRT_SEND (_PI_, CREF (tom_MutableDictionary), SEL (r_new));

	  for (j = 0; j < u->num_classes; j++)
	    {
	      struct trt_class *c = u->classes[j];
	      tom_object cn = TRT_SEND (_PI_, (void *) c, SEL (r_name));

	      TRT_SEND (, c_dict, SEL (v_set_r_at_r), c, cn);
	    }

	  un = byte_string_with_string (u->name.s, u->name.len);
	  u_obj = TRT_SEND (_PI_, CREF (tom_Unit), SEL (r_alloc));
	  u_obj = TRT_SEND (_PI_, u_obj, SEL (r_initWithName_r_classes_r),
			    un, c_dict);
	  TRT_SEND (, c_tom_Unit_units, SEL (v_set_r_at_r), u_obj, un);
	}
    }
}
