/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: hash.c,v 1.5 1998/03/17 17:46:49 tiggr Exp $  */

#include <string.h>
#include <stdio.h>
#include "trt.h"
#include <tom/tom-r.h>

#define GOLDEN_BITS -1640531527

#ifdef DEBUG
void
hash_dump (struct hashtable *ht)
{
  int i, b, n = 1 << ht->size_shift;
  fprintf (stderr, "size_shift=%d (size=%d)\n", ht->size_shift, n);
  fprintf (stderr, "num_elts=%d load=%.3f\n",
	   ht->num_elts, (double) ht->num_elts / n);
  for (i = b = 0; i < n; i++)
    if (ht->buckets[i])
      b++;
  fprintf (stderr, "occ-buckets=%d bucket-load=%.2f elt-per-occ-bucket=%.2f\n",
	   b, (double) b / n, (double) ht->num_elts / b);
}
#endif

int
hash_hash_pointer (void *elt)
{
  return HASH_POINTER (elt);
}

int
hash_hash_string (void *elt)
{
  char *s = elt;
  int h = 0, i;

  for (i = 0; i < 128 && s[i]; i++)
    h = (h << 2) ^ (s[i] << 1) ^ h;

  return h;
}

int
hash_hash_selector (selector sel)
{
  return sel->sel_id;
}

int
hash_equal_pointer (void *a, void *b)
{
  return a == b;
}

int
hash_equal_string (void *a, void *b)
{
  return a == b || !strcmp (a, b);
}

void
hash_resize (struct hashtable *ht)
{
  int h, i, old_n = ht->size_shift ? 1 << ht->size_shift : 0;
  struct bucket *c, *nc;
  /* I know `new' as a name screws bad C compilers...  */
  struct bucket **new;

  if (!ht->size_shift)
    ht->size_shift = 4;
  else
    ht->size_shift++;

  new = xcalloc (1 << ht->size_shift, sizeof (*new));

  for (i = 0; i < old_n; i++)
    for (c = ht->buckets[i]; c; c = nc)
      {
	h = ((unsigned) (ht->hash (c->key) * GOLDEN_BITS)
	     >> (32 - ht->size_shift));
	nc = c->cdr;
	c->cdr = new[h];
	new[h] = c;
      }

  xfree (ht->buckets);
  ht->buckets = new;
}

struct bucket *
hash_add (struct hashtable *ht, void *key, void *value)
{
  struct bucket *b;
  int hash;

  if (!ht->size_shift || 1 << ht->size_shift < ht->num_elts)
    hash_resize (ht);

  hash = ((unsigned) (ht->hash (key) * GOLDEN_BITS) >> (32 - ht->size_shift));
  for (b = ht->buckets[hash]; b; b = b->cdr)
    if (ht->equal (b->key, key))
      {
	b->value = value;
	return b;
      }

  b = xmalloc (sizeof (*b));
  b->key = key;
  b->value = value;
  b->cdr = ht->buckets[hash];
  ht->buckets[hash] = b;
  ht->num_elts++;

  return b;
}

void *
hash_find (struct hashtable *ht, void *key)
{
  struct bucket *b;
  int hash;

  if (!ht->size_shift)
    return NULL;

  hash = ((unsigned) (ht->hash (key) * GOLDEN_BITS) >> (32 - ht->size_shift));
  for (b = ht->buckets[hash]; b; b = b->cdr)
    if (ht->equal (b->key, key))
      return b->value;

  return NULL;
}
